/*
 * jdbcStatement.java
 *
 * Copyright (c) 2001, The HSQL Development Group
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 *
 * Neither the name of the HSQL Development Group nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This package is based on HypersonicSQL, originally developed by Thomas Mueller.
 *
 */
package org.hsqldb;

import java.sql.*;

/**
 * <P>The object used for executing a static SQL statement
 * and obtaining the results produced by it.
 *
 * <P>Only one ResultSet per Statement can be open at any point in
 * time. Therefore, if the reading of one ResultSet is interleaved
 * with the reading of another, each must have been generated by
 * different Statements. All statement <code>execute</code> methods implicitly
 * close a statment's current ResultSet if an open one exists.
 *
 * @see jdbcConnection#createStatement
 * @see jdbcResultSet
 */
public class jdbcStatement implements Statement {
    private jdbcConnection cConnection;
    private jdbcResultSet  rSet;
    private boolean	   bEscapeProcessing;
    private int		   iMaxRows;

    /**
     * Executes a SQL statement that returns a single ResultSet.
     *
     * @param sql typically this is a static SQL SELECT statement
     * @return a ResultSet that contains the data produced by the
     * query; never null
     * @exception SQLException if a database access error occurs
     */
    public ResultSet executeQuery(String sql) throws SQLException {
	fetchResult(sql);

	return rSet;
    }

    /**
     * Executes an SQL INSERT, UPDATE or DELETE statement. In addition,
     * SQL statements that return nothing, such as SQL DDL statements,
     * can be executed.
     *
     * @param sql a SQL INSERT, UPDATE or DELETE statement or a SQL
     * statement that returns nothing
     * @return either the row count for INSERT, UPDATE or DELETE or 0
     * for SQL statements that return nothing
     * @exception SQLException if a database access error occurs
     */
    public int executeUpdate(String sql) throws SQLException {
	fetchResult(sql);

	if (rSet == null) {
	    return -1;
	}

	return rSet.getUpdateCount();
    }

    /**
     * Releases this <code>Statement</code> object's database
     * and JDBC resources immediately instead of waiting for
     * this to happen when it is automatically closed.
     * It is generally good practice to release resources as soon as
     * you are finished with them to avoid tying up database
     * resources.
     * <P><B>Note:</B> A Statement is automatically closed when it is
     * garbage collected. When a Statement is closed, its current
     * ResultSet, if one exists, is also closed.
     */
    public void close() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	closeOldResult();

	rSet = null;
    }

    /**
     * Returns the maximum number of bytes allowed for any column value.
     * This limit is the maximum number of bytes that can be
     * returned for any column value.
     * The limit applies only to BINARY,
     * VARBINARY, LONGVARBINARY, CHAR, VARCHAR, and LONGVARCHAR
     * columns.  If the limit is exceeded, the excess data is silently
     * discarded.
     * <P><font color="#009900">
     * In HSQL, field sizes are always unlimited.
     * </font><P>
     * @return the current max column size limit; zero means unlimited
     */
    public int getMaxFieldSize() {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}

	return 0;
    }

    /**
     * Sets the limit for the maximum number of bytes in a column to
     * the given number of bytes.  This is the maximum number of bytes
     * that can be returned for any column value.  This limit applies
     * only to BINARY, VARBINARY, LONGVARBINARY, CHAR, VARCHAR, and
     * LONGVARCHAR fields.  If the limit is exceeded, the excess data
     * is silently discarded. For maximum portability, use values
     * greater than 256.
     * <P><font color="#009900">
     * In HSQL, field sizes are always unlimited.
     * </font><P>
     * @param max the new max column size limit; zero means unlimited
     */
    public void setMaxFieldSize(int max) {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}
    }

    /**
     * Retrieves the maximum number of rows that a
     * ResultSet can contain.  If the limit is exceeded, the excess
     * rows are silently dropped.
     *
     * @return the current max row limit; zero means unlimited
     */
    public int getMaxRows() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return iMaxRows;
    }

    /**
     * Sets the limit for the maximum number of rows that any
     * ResultSet can contain to the given number.
     * If the limit is exceeded, the excess
     * rows are silently dropped.
     *
     * @param max the new max rows limit; zero means unlimited
     */
    public void setMaxRows(int max) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	iMaxRows = max;
    }

    /**
     * Sets escape processing on or off.
     * If escape scanning is on (the default), the driver will do
     * escape substitution before sending the SQL to the database.
     *
     * Note: Since prepared statements have usually been parsed prior
     * to making this call, disabling escape processing for prepared
     * statements will have no effect.
     *
     * @param enable true to enable; false to disable
     */
    public void setEscapeProcessing(boolean enable) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	bEscapeProcessing = enable;
    }

    /**
     * Retrieves the number of seconds the driver will
     * wait for a Statement to execute. If the limit is exceeded, a
     * SQLException is thrown.
     * <P><font color="#009900">
     * HSQL currently does not support this feature.
     * Queries that take 1 second are very seldom.
     * </font><P>
     * @return the current query timeout limit in seconds; zero means unlimited
     */
    public int getQueryTimeout() {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}

	return 0;
    }

    /**
     * Sets the number of seconds the driver will
     * wait for a Statement to execute to the given number of seconds.
     * If the limit is exceeded, a SQLException is thrown.
     * <P><font color="#009900">
     * HSQL currently does not support this feature.
     * Queries that take 1 second are very seldom.
     * </font><P>
     * @param seconds the new query timeout limit in seconds; zero means
     * unlimited
     */
    public void setQueryTimeout(int seconds) {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}
    }

    /**
     * Cancels this <code>Statement</code> object if both the DBMS and
     * driver support aborting an SQL statement.
     * This method can be used by one thread to cancel a statement that
     * is being executed by another thread.
     * <P><font color="#009900">
     * HSQL currently does not support this feature.
     * </font><P>
     */
    public void cancel() {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

    /**
     * Retrieves the first warning reported by calls on this Statement.
     * Subsequent Statement warnings will be chained to this
     * SQLWarning.
     *
     * <p>The warning chain is automatically cleared each time
     * a statement is (re)executed.
     *
     * <P><B>Note:</B> If you are processing a ResultSet, any
     * warnings associated with ResultSet reads will be chained on the
     * ResultSet object.
     * <P><font color="#009900">
     * HSQL never produces warnings and returns always null.
     * </font><P>
     * @return the first SQLWarning or null
     */
    public SQLWarning getWarnings() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return null;
    }

    /**
     * Clears all the warnings reported on this <code>Statement</code>
     * object. After a call to this method,
     * the method <code>getWarnings</code> will return
     * null until a new warning is reported for this Statement.
     */
    public void clearWarnings() {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

    /**
     * Defines the SQL cursor name that will be used by
     * subsequent Statement <code>execute</code> methods. This name can then be
     * used in SQL positioned update/delete statements to identify the
     * current row in the ResultSet generated by this statement.  If
     * the database doesn't support positioned update/delete, this
     * method is a noop.  To insure that a cursor has the proper isolation
     * level to support updates, the cursor's SELECT statement should be
     * of the form 'select for update ...'. If the 'for update' phrase is
     * omitted, positioned updates may fail.
     * <P><font color="#009900">
     * HSQL currently does not support this feature.
     * </font>
     * <P><B>Note:</B> By definition, positioned update/delete
     * execution must be done by a different Statement than the one
     * which generated the ResultSet being used for positioning. Also,
     * cursor names must be unique within a connection.
     *
     * @param name the new cursor name, which must be unique within
     * a connection
     */
    public void setCursorName(String name) {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

    /**
     * Executes a SQL statement that may return multiple results.
     * Under some (uncommon) situations a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.  The  methods <code>execute</code>,
     * <code>getMoreResults</code>, <code>getResultSet</code>,
     * and <code>getUpdateCount</code> let you navigate through multiple results.
     *
     * The <code>execute</code> method executes a SQL statement and indicates the
     * form of the first result.  You can then use getResultSet or
     * getUpdateCount to retrieve the result, and getMoreResults to
     * move to any subsequent result(s).
     *
     * @param sql any SQL statement
     * @return true if the next result is a ResultSet; false if it is
     * an update count or there are no more results
     * @exception SQLException if a database access error occurs
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults
     */
    public boolean execute(String sql) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	fetchResult(sql);

	if (rSet == null) {
	    return false;
	}

	return rSet.isResult();
    }

    /**
     * Returns the current result as a <code>ResultSet</code> object.
     * This method should be called only once per result.
     *
     * @return the current result as a ResultSet; null if the result
     * is an update count or there are no more results
     * @see #execute
     */
    public ResultSet getResultSet() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

    if (rSet != null && rSet.isResult())
      return rSet;
    return null;
    }

    /**
     * Returns the current result as an update count;
     * if the result is a ResultSet or there are no more results, -1
     * is returned.
     * This method should be called only once per result.
     *
     * @return the current result as an update count; -1 if it is a
     * ResultSet or there are no more results
     * @see #execute
     */
    public int getUpdateCount() {
	if (rSet == null) {
	    return -1;
	}

	return rSet.getUpdateCount();
    }

    /**
     * Moves to a Statement's next result.  It returns true if
     * this result is a ResultSet.  This method also implicitly
     * closes any current ResultSet obtained with getResultSet.
     *
     * There are no more results when (!getMoreResults() &&
     * (getUpdateCount() == -1)
     * <P><font color="#009900">
     * HSQL currently does not support multiple ResultSets.
     * </font><P>
     * @return true if the next result is a ResultSet; false if it is
     * an update count or there are no more results
     * @see #execute
     */
    public boolean getMoreResults() {
	if (Trace.TRACE) {
	    Trace.trace();
	}
	if (rSet != null) {
		rSet.close();
		rSet = null;
	}

	return false;
    }

    /**
     * JDBC 2.0
     *
     * Gives the driver a hint as to the direction in which
     * the rows in a result set
     * will be processed. The hint applies only to result sets created
     * using this Statement object.  The default value is
     * ResultSet.FETCH_FORWARD.
     * <p>Note that this method sets the default fetch direction for
     * result sets generated by this <code>Statement</code> object.
     * Each result set has its own methods for getting and setting
     * its own fetch direction.
     * @param direction the initial direction for processing rows
     * @exception SQLException if a database access error occurs
     * or the given direction
     * is not one of ResultSet.FETCH_FORWARD, ResultSet.FETCH_REVERSE, or
     * ResultSet.FETCH_UNKNOWN
     */

//#ifdef JAVA2
    public void setFetchDirection(int direction) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	if (direction != ResultSet.FETCH_FORWARD) {
	    throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Retrieves the direction for fetching rows from
     * database tables that is the default for result sets
     * generated from this <code>Statement</code> object.
     * If this <code>Statement</code> object has not set
     * a fetch direction by calling the method <code>setFetchDirection</code>,
     * the return value is implementation-specific.
     *
     * @return the default fetch direction for result sets generated
     * from this <code>Statement</code> object
     */

//#ifdef JAVA2
    public int getFetchDirection() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return ResultSet.FETCH_FORWARD;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gives the JDBC driver a hint as to the number of rows that should
     * be fetched from the database when more rows are needed.  The number
     * of rows specified affects only result sets created using this
     * statement. If the value specified is zero, then the hint is ignored.
     * The default value is zero.
     *
     * @param rows the number of rows to fetch
     * condition 0 <= rows <= this.getMaxRows() is not satisfied.
     */

//#ifdef JAVA2
    public void setFetchSize(int rows) {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Retrieves the number of result set rows that is the default
     * fetch size for result sets
     * generated from this <code>Statement</code> object.
     * If this <code>Statement</code> object has not set
     * a fetch size by calling the method <code>setFetchSize</code>,
     * the return value is implementation-specific.
     * @return the default fetch size for result sets generated
     * from this <code>Statement</code> object
     */

//#ifdef JAVA2
    public int getFetchSize() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return 1;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Retrieves the result set concurrency.
     */

//#ifdef JAVA2
    public int getResultSetConcurrency() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return ResultSet.CONCUR_READ_ONLY;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Determine the result set type.
     */

//#ifdef JAVA2
    public int getResultSetType() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return ResultSet.TYPE_FORWARD_ONLY;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Adds a SQL command to the current batch of commmands for the statement.
     * This method is optional.
     * <P><font color="#009900">
     * HSQL currently does not support this feature because there is
     * no performance win in using batches.
     * </font><P>
     * @param sql typically this is a static SQL INSERT or UPDATE statement
     * @exception SQLException if a database access error occurs, or the
     * driver does not support batch statements
     */

//#ifdef JAVA2
    public void addBatch(String sql) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Makes the set of commands in the current batch empty.
     * This method is optional.
     *
     * @exception SQLException if a database access error occurs or the
     * driver does not support batch statements
     */

//#ifdef JAVA2
    public void clearBatch() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Submits a batch of commands to the database for execution.
     * This method is optional.
     *
     * @return an array of update counts containing one element for each
     * command in the batch.  The array is ordered according
     * to the order in which commands were inserted into the batch.
     * @exception SQLException if a database access error occurs or the
     * driver does not support batch statements
     */

//#ifdef JAVA2
    public int[] executeBatch() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the <code>Connection</code> object
     * that produced this <code>Statement</code> object.
     * @return the connection that produced this statement
     */

//#ifdef JAVA2
    public Connection getConnection() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return cConnection;
    }

//#endif JAVA2

    /**
     * Constructor declaration
     *
     *
     * @param c
     *
     * @see
     */
    jdbcStatement(jdbcConnection c) {
	bEscapeProcessing = true;
	cConnection = c;
    }

    /**
     * Method declaration
     *
     *
     * @param sql
     *
     * @throws SQLException
     *
     * @see
     */
    private void fetchResult(String sql) throws SQLException {
	if (bEscapeProcessing) {
	    sql = cConnection.nativeSQL(sql);
	}

	closeOldResult();

	if (iMaxRows == 0) {
	    rSet = cConnection.execute(sql);
	} else {
	    try {
		sql = "SET MAXROWS " + iMaxRows + ";" + sql;
		rSet = cConnection.execute(sql);

		cConnection.execute("SET MAXROWS 0");
	    } catch (SQLException e) {
		cConnection.execute("SET MAXROWS 0");

		throw e;
	    }
	}
    }

    /**
     * Method declaration
     *
     *
     * @see
     */
    private void closeOldResult() {

	// this is necessary to conform the JDBC standard
	if (rSet != null) {
	    rSet.close();

	    rSet = null;
	}
    }

}
