/*
 * jdbcResultSet.java
 *
 * Copyright (c) 2001, The HSQL Development Group
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 *
 * Neither the name of the HSQL Development Group nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This package is based on HypersonicSQL, originally developed by Thomas Mueller.
 *
 */
package org.hsqldb;

import java.sql.*;
import java.io.*;
import java.math.BigDecimal;
import java.util.*;

/**
 * <P><font color="#009900">
 * In HSQL Database Engine, the class jdbcResultSet implements both the
 * ResultSet and ResultSetMetaData interfaces. To conform the JDBC standard,
 * it is anyway required to call the 'getMetaData' function to get the
 * ResultSetMetaData of the ResultSet. This is so because in other JDBC drivers
 * the interfaces are not implemented in the same class.
 * <P>
 * ResultSetMetaData can be used to find out about the types
 * and properties of the columns in a ResultSet.
 * </font><P>
 * <P>A ResultSet provides access to a table of data.  A ResultSet
 * object is usually generated by executing a Statement.
 *
 * <P>A ResultSet maintains a cursor pointing to its current row of
 * data.  Initially the cursor is positioned before the first row.
 * The 'next' method moves the cursor to the next row.
 *
 * <P>The getXXX methods retrieve column values for the current
 * row.  You can retrieve values using either the index number of the
 * column or the name of the column.  In general, using the
 * column index will be more efficient.  Columns are numbered from 1.
 *
 * <P>For maximum portability, ResultSet columns within each row should be
 * read in left-to-right order and each column should be read only once.
 *
 * <P>For the getXXX methods, the JDBC driver attempts to convert the
 * underlying data to the specified Java type and returns a suitable
 * Java value.  See the JDBC specification for allowable mappings
 * from SQL types to Java types with the ResultSet.getXXX methods.
 *
 * <P>Column names used as input to getXXX methods are case
 * insensitive.  When performing a getXXX using a column name, if
 * several columns have the same name, then the value of the first
 * matching column will be returned. The column name option is
 * designed to be used when column names are used in the SQL
 * query. For columns that are NOT explicitly named in the query, it
 * is best to use column numbers. If column names are used, there is
 * no way for the programmer to guarantee that they actually refer to
 * the intended columns.
 *
 * <P>A ResultSet is automatically closed by the Statement that
 * generated it when that Statement is closed, re-executed, or used
 * to retrieve the next result from a sequence of multiple results.
 *
 * <P>The number, types and properties of a ResultSet's columns are
 * provided by the ResulSetMetaData object returned by the getMetaData
 * method.
 *
 * @see jdbcStatement#executeQuery
 * @see jdbcStatement#getResultSet
 */
public class jdbcResultSet implements ResultSet, ResultSetMetaData {
    private Result  rResult;
    private Record  nCurrent;
    private int     iCurrentRow;
    private int     iUpdateCount;
    private boolean bInit;    // false if before first row
    private int     iColumnCount;
    private boolean bWasNull;

    /**
     * Moves the cursor down one row from its current position.
     * A ResultSet cursor is initially positioned before the first row; the
     * first call to next makes the first row the current row; the
     * second call makes the second row the current row, and so on.
     *
     * <P>If an input stream is open for the current row, a call
     * to the method <code>next</code> will
     * implicitly close it. The ResultSet's warning chain is cleared
     * when a new row is read.
     *
     * @return true if the new current row is valid; false if there
     * are no more rows
     */
    public boolean next() throws SQLException {
	bWasNull = false;    // todo: test if other databases do the same

	if (rResult == null) {
	    return false;    // throw exception ?
	}

	if (bInit == false) {
	    nCurrent = rResult.rRoot;
	    bInit = true;
	    iCurrentRow = 1;
	} else {
	    if (nCurrent == null) {
		return false;
	    }

	    nCurrent = nCurrent.next;
	    iCurrentRow++;
	}

	if (nCurrent == null) {
	    iCurrentRow = 0;

	    return false;
	}

	return true;
    }

    /**
     * Releases this <code>ResultSet</code> object's database and
     * JDBC resources immediately instead of waiting for
     * this to happen when it is automatically closed.
     *
     * <P><B>Note:</B> A ResultSet is automatically closed by the
     * Statement that generated it when that Statement is closed,
     * re-executed, or is used to retrieve the next result from a
     * sequence of multiple results. A ResultSet is also automatically
     * closed when it is garbage collected.
     */
    public void close() {
	iUpdateCount = -1;
	rResult = null;
    }

    /**
     * Reports whether
     * the last column read had a value of SQL NULL.
     * Note that you must first call getXXX on a column to try to read
     * its value and then call wasNull() to see if the value read was
     * SQL NULL.
     *
     * @return true if last column read was SQL NULL and false otherwise
     */
    public boolean wasNull() {
	return bWasNull;
    }

    /**
     * Maps the given Resultset column name to its ResultSet column index.
     *
     * @param columnName the name of the column
     * @return the column index
     * @exception SQLException if a database access error occurs
     */
    public int findColumn(String columnName) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace(columnName);
	}

	for (int i = 0; i < iColumnCount; i++) {
	    if (columnName.equalsIgnoreCase(rResult.sLabel[i])) {
		return i + 1;
	    }
	}

	throw Trace.error(Trace.COLUMN_NOT_FOUND);
    }

    /**
     * Gets the value of a column in the current row as a Java String.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public String getString(int column) throws SQLException {
	checkColumn(--column);
	checkAvailable();

	Object o = nCurrent.data[column];

	// use checknull because getColumnInType is not used
	checkNull(o);

	return o == null ? null : o.toString();
    }

    /**
     * Gets the value of a column in the current row as a Java boolean.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is false
     * @exception SQLException if a database access error occurs
     */
    public boolean getBoolean(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.BIT);

	return o == null ? false : ((Boolean) o).booleanValue();
    }

    /**
     * Gets the value of a column in the current row as a Java byte.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public byte getByte(int column) throws SQLException {
	return (byte) getShort(column);
    }

    /**
     * Gets the value of a column in the current row as a Java short.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public short getShort(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.SMALLINT);

	return o == null ? 0 : ((Short) o).shortValue();
    }

    /**
     * Gets the value of a column in the current row as a Java int.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public int getInt(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.INTEGER);

	return o == null ? 0 : ((Integer) o).intValue();
    }

    /**
     * Gets the value of a column in the current row as a Java long.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public long getLong(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.BIGINT);

	return o == null ? 0 : ((Long) o).longValue();
    }

    /**
     * Gets the value of a column in the current row as a Java float.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public float getFloat(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.REAL);

	return o == null ? (float) 0.0 : ((Float) o).floatValue();
    }

    /**
     * Gets the value of a column in the current row as a Java double.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public double getDouble(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.DOUBLE);

	return o == null ? 0.0 : ((Double) o).doubleValue();
    }

    /**
     * Gets the value of a column in the current row as a java.math.BigDecimal
     * object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param scale the number of digits to the right of the decimal
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     * @deprecated
     */
    public BigDecimal getBigDecimal(int column,
				    int scale) throws SQLException {
	return (BigDecimal) getColumnInType(--column, Column.DECIMAL);
    }

    /**
     * Gets the value of a column in the current row as a Java byte array.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public byte[] getBytes(int column) throws SQLException {
	Object o = getColumnInType(--column, Column.BINARY);

	return o == null ? null : ((ByteArray) o).byteValue();
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Date object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public java.sql.Date getDate(int column) throws SQLException {
	return (java.sql.Date) getColumnInType(--column, Column.DATE);
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Time object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public Time getTime(int column) throws SQLException {
	return (Time) getColumnInType(--column, Column.TIME);
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Timestamp
     * object.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public Timestamp getTimestamp(int column) throws SQLException {
	return (Timestamp) getColumnInType(--column, Column.TIMESTAMP);
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * ASCII characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream.  Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of one byte ASCII characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     */
    public InputStream getAsciiStream(int column) throws SQLException {
	return getUnicodeStream(column);
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * Unicode characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into Unicode.
     * The byte format of the Unicode stream must Java UTF-8,
     * as specified in the Java Virtual Machine Specification.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of two-byte Unicode characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     * @deprecated
     */
    public InputStream getUnicodeStream(int column) throws SQLException {
	return new ByteArrayInputStream(getBytes(column));
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * uninterpreted bytes. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARBINARY values.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a Java input stream that delivers the database column value
     * as a stream of uninterpreted bytes.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     */
    public InputStream getBinaryStream(int column) throws SQLException {
	return new ByteArrayInputStream(getBytes(column));
    }

    /**
     * Gets the value of a column in the current row as a Java String.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public String getString(String columnName) throws SQLException {
	return getString(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java boolean.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is false
     * @exception SQLException if a database access error occurs
     */
    public boolean getBoolean(String columnName) throws SQLException {
	return getBoolean(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java byte.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public byte getByte(String columnName) throws SQLException {
	return getByte(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java short.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public short getShort(String columnName) throws SQLException {
	return getShort(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java int.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public int getInt(String columnName) throws SQLException {
	return getInt(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java long.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public long getLong(String columnName) throws SQLException {
	return getLong(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java float.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public float getFloat(String columnName) throws SQLException {
	return getFloat(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a Java double.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is 0
     * @exception SQLException if a database access error occurs
     */
    public double getDouble(String columnName) throws SQLException {
	return getDouble(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a java.math.BigDecimal
     * object.
     *
     * @param columnName the SQL name of the column
     * @param scale the number of digits to the right of the decimal
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     * @deprecated
     */
    public BigDecimal getBigDecimal(String columnName,
				    int scale) throws SQLException {
	return getBigDecimal(findColumn(columnName), scale);
    }

    /**
     * Gets the value of a column in the current row as a Java byte array.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public byte[] getBytes(String columnName) throws SQLException {
	return getBytes(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Date object.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public java.sql.Date getDate(String columnName) throws SQLException {
	return getDate(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Time object.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public Time getTime(String columnName) throws SQLException {
	return getTime(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a java.sql.Timestamp
     * object.
     *
     * @param columnName the SQL name of the column
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */
    public Timestamp getTimestamp(String columnName) throws SQLException {
	return getTimestamp(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * ASCII characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into ASCII.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnName the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of one byte ASCII characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     */
    public InputStream getAsciiStream(String columnName) throws SQLException {
	return getAsciiStream(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * Unicode characters. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARCHAR values.  The JDBC driver will
     * do any necessary conversion from the database format into Unicode.
     * The byte format of the Unicode stream must be Java UTF-8,
     * as defined in the Java Virtual Machine Specification.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnName the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of two-byte Unicode characters.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     * @deprecated
     */
    public InputStream getUnicodeStream(String columnName)
	    throws SQLException {
	return getUnicodeStream(findColumn(columnName));
    }

    /**
     * Gets the value of a column in the current row as a stream of
     * uninterpreted bytes. The value can then be read in chunks from the
     * stream. This method is particularly
     * suitable for retrieving large LONGVARBINARY values.  The JDBC driver will
     * do any necessary conversion from the database format into uninterpreted
     * bytes.
     *
     * <P><B>Note:</B> All the data in the returned stream must be
     * read prior to getting the value of any other column. The next
     * call to a get method implicitly closes the stream. Also, a
     * stream may return 0 when the method <code>available</code>
     * is called whether there is data
     * available or not.
     *
     * @param columnName the SQL name of the column
     * @return a Java input stream that delivers the database column value
     * as a stream of uninterpreted bytes.  If the value is SQL NULL
     * then the result is null.
     * @exception SQLException if a database access error occurs
     */
    public InputStream getBinaryStream(String columnName)
	    throws SQLException {
	return getBinaryStream(findColumn(columnName));
    }

    /**
     * <p>The first warning reported by calls on this ResultSet is
     * returned. Subsequent ResultSet warnings will be chained to this
     * SQLWarning.
     *
     * <P>The warning chain is automatically cleared each time a new
     * row is read.
     *
     * <P><B>Note:</B> This warning chain only covers warnings caused
     * by ResultSet methods.  Any warning caused by statement methods
     * (such as reading OUT parameters) will be chained on the
     * Statement object.
     * <P><font color="#009900">
     * HSQL Database Engine never produces warnings and returns always null.
     * </font><P>
     * @return the first SQLWarning or null
     */
    public SQLWarning getWarnings() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return null;
    }

    /**
     * After this call getWarnings returns null until a new warning is
     * reported for this ResultSet.
     */
    public void clearWarnings() {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

    /**
     * Gets the name of the SQL cursor used by this ResultSet.
     *
     * <P>In SQL, a result table is retrieved through a cursor that is
     * named. The current row of a result can be updated or deleted
     * using a positioned update/delete statement that references the
     * cursor name. To insure that the cursor has the proper isolation
     * level to support update, the cursor's select statement should be
     * of the form 'select for update'. If the 'for update' clause is
     * omitted the positioned updates may fail.
     *
     * <P>JDBC supports this SQL feature by providing the name of the
     * SQL cursor used by a ResultSet. The current row of a ResultSet
     * is also the current row of this SQL cursor.
     *
     * <P><B>Note:</B> If positioned update is not supported a
     * SQLException is thrown
     *
     * @return the ResultSet's SQL cursor name
     * @exception SQLException if a database access error occurs
     */
    public String getCursorName() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

    /**
     * Retrieves the  number, types and properties of a ResultSet's columns.
     * <P><font color="#009900">
     * HSQL Database Engine implements both the ResultSet and the ResultSetMetaData
     * interface in the class jdbcResultSet, so this function will return the
     * same Object. It is required to call this function to conform the
     * JDBC standard. Other drivers have the ResultSetMetaData in a different
     * class.
     * </font><P>
     * @return the description of a ResultSet's columns
     * @exception SQLException if a database access error occurs
     */
    public ResultSetMetaData getMetaData() throws SQLException {
	return this;
    }

    /**
     * <p>Gets the value of a column in the current row as a Java object.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java object type corresponding to the column's SQL type,
     * following the mapping for built-in types specified in the JDBC
     * spec.
     *
     * <p>This method may also be used to read datatabase-specific
     * abstract data types.
     *
     * JDBC 2.0
     *
     * In the JDBC 2.0 API, the behavior of method
     * <code>getObject</code> is extended to materialize
     * data of SQL user-defined types.  When the a column contains
     * a structured or distinct value, the behavior of this method is as
     * if it were a call to: getObject(columnIndex,
     * this.getStatement().getConnection().getTypeMap()).
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return a java.lang.Object holding the column value
     * @exception SQLException if a database access error occurs
     */
    public Object getObject(String columnName) throws SQLException {
	return getObject(findColumn(columnName));
    }

    /**
     * <p>Gets the value of a column in the current row as a Java object.
     *
     * <p>This method will return the value of the given column as a
     * Java object.  The type of the Java object will be the default
     * Java object type corresponding to the column's SQL type,
     * following the mapping for built-in types specified in the JDBC
     * spec.
     *
     * <p>This method may also be used to read datatabase-specific
     * abstract data types.
     *
     * JDBC 2.0
     *
     * In the JDBC 2.0 API, the behavior of method
     * <code>getObject</code> is extended to materialize
     * data of SQL user-defined types.  When the a column contains
     * a structured or distinct value, the behavior of this method is as
     * if it were a call to: getObject(columnIndex,
     * this.getStatement().getConnection().getTypeMap()).
     *
     * @param columnName the SQL name of the column
     * @return a java.lang.Object holding the column value.
     * @exception SQLException if a database access error occurs
     */
    public Object getObject(int column) throws SQLException {
	checkColumn(--column);
	checkAvailable();

	Object o = nCurrent.data[column];

	// use checknull because getColumnInType is not used
	checkNull(o);

	if (o == null) {
	    return null;
	}

	if (rResult.iType[column] == Column.OTHER) {
	    o = ((ByteArray) o).deserialize();
	} else if (o instanceof ByteArray) {
	    o = ((ByteArray) o).byteValue();
	}

	return o == null ? null : o;
    }

    /**
     * JDBC 2.0
     *
     * <p>Gets the value of a column in the current row as a java.io.Reader.
     * @param columnName the name of the column
     * @return the value in the specified column as a <code>java.io.Reader</code>
     */

//#ifdef JAVA2
    public Reader getCharacterStream(String columnName) throws SQLException {
	return getCharacterStream(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets the value of a column in the current row as a java.math.BigDecimal
     * object with full precision.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @return the column value (full precision); if the value is SQL NULL,
     * the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public BigDecimal getBigDecimal(int column) throws SQLException {
	return getBigDecimal(column, 0);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets the value of a column in the current row as a java.math.BigDecimal
     * object with full precision.
     * @param columnName the column name
     * @return the column value (full precision); if the value is SQL NULL,
     * the result is null
     * @exception SQLException if a database access error occurs
     *
     */

//#ifdef JAVA2
    public BigDecimal getBigDecimal(String columnName) throws SQLException {
	return getBigDecimal(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Indicates whether the cursor is before the first row in the result
     * set.
     *
     * @return true if the cursor is before the first row, false otherwise. Returns
     * false when the result set contains no rows.
     */

//#ifdef JAVA2
    public boolean isBeforeFirst() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	if (bInit == false) {
	    return true;
	}

	return false;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Indicates whether the cursor is after the last row in the result
     * set.
     *
     * @return true if the cursor is  after the last row, false otherwise.
     * Returns false when the result set contains no rows.
     */

//#ifdef JAVA2
    public boolean isAfterLast() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	if (bInit == false) {
	    return false;
	}

	if (nCurrent == null) {
	    return true;
	}

	return false;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Indicates whether the cursor is on the first row of the result set.
     *
     * @return true if the cursor is on the first row, false otherwise.
     */

//#ifdef JAVA2
    public boolean isFirst() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return iCurrentRow == 1;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Indicates whether the cursor is on the last row of the result set.
     * Note: Calling the method <code>isLast</code> may be expensive
     * because the JDBC driver
     * might need to fetch ahead one row in order to determine
     * whether the current row is the last row in the result set.
     *
     * @return true if the cursor is on the last row, false otherwise.
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public boolean isLast() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the front of the result set, just before the
     * first row. Has no effect if the result set contains no rows.
     *
     * @exception SQLException if a database access error occurs or the
     * result set type is TYPE_FORWARD_ONLY
     */

//#ifdef JAVA2
    public void beforeFirst() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the end of the result set, just after the last
     * row.  Has no effect if the result set contains no rows.
     *
     * @exception SQLException if a database access error occurs or the
     * result set type is TYPE_FORWARD_ONLY
     */

//#ifdef JAVA2
    public void afterLast() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the first row in the result set.
     *
     * @return true if the cursor is on a valid row; false if
     * there are no rows in the result set
     * @exception SQLException if a database access error occurs or the
     * result set type is TYPE_FORWARD_ONLY
     */

//#ifdef JAVA2
    public boolean first() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the last row in the result set.
     *
     * @return true if the cursor is on a valid row;
     * false if there are no rows in the result set
     * @exception SQLException if a database access error occurs or the
     * result set type is TYPE_FORWARD_ONLY.
     */

//#ifdef JAVA2
    public boolean last() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Retrieves the current row number.  The first row is number 1, the
     * second number 2, and so on.
     *
     * @return the current row number; 0 if there is no current row
     */

//#ifdef JAVA2
    public int getRow() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return iCurrentRow;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the given row number in the result set.
     *
     * <p>If the row number is positive, the cursor moves to
     * the given row number with respect to the
     * beginning of the result set.  The first row is row 1, the second
     * is row 2, and so on.
     *
     * <p>If the given row number is negative, the cursor moves to
     * an absolute row position with respect to
     * the end of the result set.  For example, calling
     * <code>absolute(-1)</code> positions the
     * cursor on the last row, <code>absolute(-2)</code> indicates the
     * next-to-last row, and so on.
     *
     * <p>An attempt to position the cursor beyond the first/last row in
     * the result set leaves the cursor before/after the first/last
     * row, respectively.
     *
     * <p>Note: Calling <code>absolute(1)</code> is the same
     * as calling <code>first()</code>.
     * Calling <code>absolute(-1)</code> is the same as calling
     * <code>last()</code>.
     *
     * @return true if the cursor is on the result set; false otherwise
     * @exception SQLException if a database access error occurs or
     * row is 0, or result set type is TYPE_FORWARD_ONLY.
     */

//#ifdef JAVA2
    public boolean absolute(int row) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor a relative number of rows, either positive or negative.
     * Attempting to move beyond the first/last row in the
     * result set positions the cursor before/after the
     * the first/last row. Calling <code>relative(0)</code> is valid, but does
     * not change the cursor position.
     *
     * <p>Note: Calling <code>relative(1)</code>
     * is different from calling <code>next()</code>
     * because is makes sense to call <code>next()</code> when there is no
     * current row,
     * for example, when the cursor is positioned before the first row
     * or after the last row of the result set.
     *
     * @return true if the cursor is on a row; false otherwise
     * @exception SQLException if a database access error occurs, there
     * is no current row, or the result set type is TYPE_FORWARD_ONLY
     */

//#ifdef JAVA2
    public boolean relative(int rows) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	if (rows < 0) {
	    throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
	}

	while (rows-- > 0) {
	    next();
	}

	return nCurrent != null;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Moves the cursor to the previous row in the result set.
     *
     * <p>Note: <code>previous()</code> is not the same as
     * <code>relative(-1)</code> because it
     * makes sense to call</code>previous()</code> when there is no current row.
     *
     * @return true if the cursor is on a valid row; false if it is off the
     * result set
     * @exception SQLException if a database access error occurs or the
     * result set type is TYPE_FORWARD_ONLY
     */

//#ifdef JAVA2
    public boolean previous() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gives a hint as to the direction in which the rows in this result set
     * will be processed.  The initial value is determined by the statement
     * that produced the result set.  The fetch direction may be changed
     * at any time.
     *
     * @exception SQLException if a database access error occurs or
     * the result set type is TYPE_FORWARD_ONLY and the fetch direction is not
     * FETCH_FORWARD.
     */

//#ifdef JAVA2
    public void setFetchDirection(int direction) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace(direction);
	}

	if (direction != ResultSet.FETCH_FORWARD) {
	    throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the fetch direction for this result set.
     *
     * @return the current fetch direction for this result set
     */

//#ifdef JAVA2
    public int getFetchDirection() {
	return ResultSet.FETCH_FORWARD;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gives the JDBC driver a hint as to the number of rows that should
     * be fetched from the database when more rows are needed for this result
     * set.  If the fetch size specified is zero, the JDBC driver
     * ignores the value and is free to make its own best guess as to what
     * the fetch size should be.  The default value is set by the statement
     * that created the result set.  The fetch size may be changed at any
     * time.
     *
     * @param rows the number of rows to fetch
     * @exception SQLException if a database access error occurs or the
     * condition 0 <= rows <= this.getMaxRows() is not satisfied.
     */

//#ifdef JAVA2
    public void setFetchSize(int rows) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace(rows);
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the fetch size for this result set.
     *
     * @return the current fetch size for this result set
     */

//#ifdef JAVA2
    public int getFetchSize() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return 1;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the type of this result set.  The type is determined by
     * the statement that created the result set.
     *
     * @return TYPE_FORWARD_ONLY, TYPE_SCROLL_INSENSITIVE, or
     * TYPE_SCROLL_SENSITIVE
     */

//#ifdef JAVA2
    public int getType() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return TYPE_FORWARD_ONLY;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the concurrency mode of this result set.  The concurrency
     * used is determined by the statement that created the result set.
     *
     * @return the concurrency type, CONCUR_READ_ONLY or CONCUR_UPDATABLE
     */

//#ifdef JAVA2
    public int getConcurrency() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return CONCUR_READ_ONLY;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Indicates whether the current row has been updated.  The value returned
     * depends on whether or not the result set can detect updates.
     *
     * @return true if the row has been visibly updated by the owner or
     * another, and updates are detected
     *
     * @see DatabaseMetaData#updatesAreDetected
     */

//#ifdef JAVA2
    public boolean rowUpdated() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Indicates whether the current row has had an insertion.  The value
     * returned
     * depends on whether or not the result set can detect visible inserts.
     *
     * @return true if a row has had an insertion and insertions are detected
     *
     * @see DatabaseMetaData#insertsAreDetected
     */

//#ifdef JAVA2
    public boolean rowInserted() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Indicates whether a row has been deleted.  A deleted row may leave
     * a visible "hole" in a result set.  This method can be used to
     * detect holes in a result set.  The value returned depends on whether
     * or not the result set can detect deletions.
     *
     * @return true if a row was deleted and deletions are detected
     *
     * @see DatabaseMetaData#deletesAreDetected
     */

//#ifdef JAVA2
    public boolean rowDeleted() {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateNull(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace(column);
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBoolean(int column, boolean x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateByte(int column, byte x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateShort(int column, short x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateInt(int column, int x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateLong(int column, long x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateFloat(int column, float x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateDouble(int column, double x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBigDecimal(int column,
				 BigDecimal x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateString(int column, String x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBytes(int column, byte[] x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateDate(int column, java.sql.Date x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateTime(int column, Time x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateTimestamp(int column, Timestamp x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateAsciiStream(int column, InputStream x,
				  int length) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBinaryStream(int column, InputStream x,
				   int length) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateCharacterStream(int column, Reader x,
				      int length) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     * @param scale
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateObject(int column, Object x,
			     int scale) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param column
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateObject(int column, Object x) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateNull(String columnName) throws SQLException {
	updateNull(findColumn(columnName));
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBoolean(String columnName,
			      boolean x) throws SQLException {
	updateBoolean(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateByte(String columnName, byte x) throws SQLException {
	updateByte(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateShort(String columnName, short x) throws SQLException {
	updateShort(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateInt(String columnName, int x) throws SQLException {
	updateInt(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateLong(String columnName, long x) throws SQLException {
	updateLong(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateFloat(String columnName, float x) throws SQLException {
	updateFloat(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateDouble(String columnName,
			     double x) throws SQLException {
	updateDouble(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBigDecimal(String columnName,
				 BigDecimal x) throws SQLException {
	updateBigDecimal(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateString(String columnName,
			     String x) throws SQLException {
	updateString(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBytes(String columnName, byte[] x) throws SQLException {
	updateBytes(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateDate(String columnName,
			   java.sql.Date x) throws SQLException {
	updateDate(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateTime(String columnName, Time x) throws SQLException {
	updateTime(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateTimestamp(String columnName,
				Timestamp x) throws SQLException {
	updateTimestamp(findColumn(columnName), x);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateAsciiStream(String columnName, InputStream x,
				  int length) throws SQLException {
	updateAsciiStream(findColumn(columnName), x, length);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateBinaryStream(String columnName, InputStream x,
				   int length) throws SQLException {
	updateBinaryStream(findColumn(columnName), x, length);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     * @param length
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateCharacterStream(String columnName, Reader x,
				      int length) throws SQLException {
	updateCharacterStream(findColumn(columnName), x, length);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     * @param scale
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateObject(String columnName, Object x,
			     int scale) throws SQLException {
	updateObject(findColumn(columnName), x, scale);
    }

//#endif JAVA2
    /**
     * Method declaration
     *
     *
     * @param columnName
     * @param x
     *
     * @throws SQLException
     */
//#ifdef JAVA2

    public void updateObject(String columnName,
			     Object x) throws SQLException {
	updateObject(findColumn(columnName), x);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Inserts the contents of the insert row into the result set and
     * the database.  Must be on the insert row when this method is called.
     *
     * @exception SQLException if a database access error occurs,
     * if called when not on the insert row, or if not all of non-nullable
     * columns in
     * the insert row have been given a value
     */

//#ifdef JAVA2
    public void insertRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Updates the underlying database with the new contents of the
     * current row.  Cannot be called when on the insert row.
     *
     * @exception SQLException if a database access error occurs or
     * if called when on the insert row
     */

//#ifdef JAVA2
    public void updateRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Deletes the current row from the result set and the underlying
     * database.  Cannot be called when on the insert row.
     *
     * @exception SQLException if a database access error occurs or if
     * called when on the insert row.
     */

//#ifdef JAVA2
    public void deleteRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Refreshes the current row with its most recent value in
     * the database.  Cannot be called when on the insert row.
     *
     * The <code>refreshRow</code> method provides a way for an application to
     * explicitly tell the JDBC driver to refetch a row(s) from the
     * database.  An application may want to call <code>refreshRow</code> when
     * caching or prefetching is being done by the JDBC driver to
     * fetch the latest value of a row from the database.  The JDBC driver
     * may actually refresh multiple rows at once if the fetch size is
     * greater than one.
     *
     * All values are refetched subject to the transaction isolation
     * level and cursor sensitivity.  If <code>refreshRow</code> is called after
     * calling <code>updateXXX</code>, but before calling <code>updateRow</code>,
     * then the
     * updates made to the row are lost.  Calling the method
     * <code>refreshRow</code> frequently
     * will likely slow performance.
     *
     * @exception SQLException if a database access error occurs or if
     * called when on the insert row
     */

//#ifdef JAVA2
    public void refreshRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Cancels the updates made to a row.
     * This method may be called after calling an
     * <code>updateXXX</code> method(s) and before calling <code>updateRow</code>
     * to rollback
     * the updates made to a row.  If no updates have been made or
     * <code>updateRow</code> has already been called, then this method has no
     * effect.
     *
     * @exception SQLException if a database access error occurs or if
     * called when on the insert row
     *
     */

//#ifdef JAVA2
    public void cancelRowUpdates() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Moves the cursor to the insert row.  The current cursor position is
     * remembered while the cursor is positioned on the insert row.
     *
     * The insert row is a special row associated with an updatable
     * result set.  It is essentially a buffer where a new row may
     * be constructed by calling the <code>updateXXX</code> methods prior to
     * inserting the row into the result set.
     *
     * Only the <code>updateXXX</code>, <code>getXXX</code>,
     * and <code>insertRow</code> methods may be
     * called when the cursor is on the insert row.  All of the columns in
     * a result set must be given a value each time this method is
     * called before calling <code>insertRow</code>.
     * The method <code>updateXXX</code> must be called before a
     * <code>getXXX</code> method can be called on a column value.
     *
     * @exception SQLException if a database access error occurs
     * or the result set is not updatable
     */

//#ifdef JAVA2
    public void moveToInsertRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Moves the cursor to the remembered cursor position, usually the
     * current row.  This method has no effect if the cursor is not on the insert
     * row.
     *
     * @exception SQLException if a database access error occurs
     * or the result set is not updatable
     */

//#ifdef JAVA2
    public void moveToCurrentRow() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the Statement that produced this <code>ResultSet</code> object.
     * If the result set was generated some other way, such as by a
     * <code>DatabaseMetaData</code> method, this method returns <code>null</code>.
     *
     * @return the Statment that produced the result set or
     * null if the result set was produced some other way
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public Statement getStatement() throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();

	    // todo
	}

	return null;
    }

//#endif

    /**
     * JDBC 2.0
     *
     * Returns the value of a column in the current row as a Java object.
     * This method uses the given <code>Map</code> object
     * for the custom mapping of the
     * SQL structured or distinct type that is being retrieved.
     *
     * @param i the first column is 1, the second is 2, ...
     * @param map the mapping from SQL type names to Java classes
     * @return an object representing the SQL value
     */

//#ifdef JAVA2

    public Object getObject(int column, Map map) throws SQLException {
	return getObject(column);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a REF(&lt;structured-type&gt;) column value from the current row.
     *
     * @param i the first column is 1, the second is 2, ...
     * @return a <code>Ref</code> object representing an SQL REF value
     */

//#ifdef JAVA2

    public Ref getRef(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a BLOB value in the current row of this <code>ResultSet</code> object.
     *
     * @param i the first column is 1, the second is 2, ...
     * @return a <code>Blob</code> object representing the SQL BLOB value in
     * the specified column
     */

//#ifdef JAVA2

    public Blob getBlob(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a CLOB value in the current row of this <code>ResultSet</code> object.
     *
     * @param i the first column is 1, the second is 2, ...
     * @return a <code>Clob</code> object representing the SQL CLOB value in
     * the specified column
     */

//#ifdef JAVA2
    public Clob getClob(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets an SQL ARRAY value from the current row of this
     * <code>ResultSet</code> object.
     *
     * @param i the first column is 1, the second is 2, ...
     * @return an <code>Array</code> object representing the SQL ARRAY value in
     * the specified column
     */

//#ifdef JAVA2
    public Array getArray(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Returns the value in the specified column as a Java object.
     * This method uses the specified <code>Map</code> object for
     * custom mapping if appropriate.
     *
     * @param colName the name of the column from which to retrieve the value
     * @param map the mapping from SQL type names to Java classes
     * @return an object representing the SQL value in the specified column
     */

//#ifdef JAVA2
    public Object getObject(String columnName, Map map) throws SQLException {
	return getObject(findColumn(columnName), map);
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a REF(&lt;structured-type&gt;) column value from the current row.
     *
     * @param colName the column name
     * @return a <code>Ref</code> object representing the SQL REF value in
     * the specified column
     */

//#ifdef JAVA2
    public Ref getRef(String columnName) throws SQLException {
	return getRef(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a BLOB value in the current row of this <code>ResultSet</code> object.
     *
     * @param colName the name of the column from which to retrieve the value
     * @return a <code>Blob</code> object representing the SQL BLOB value in
     * the specified column
     */

//#ifdef JAVA2
    public Blob getBlob(String columnName) throws SQLException {
	return getBlob(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets a CLOB value in the current row of this <code>ResultSet</code> object.
     *
     * @param colName the name of the column from which to retrieve the value
     * @return a <code>Clob</code> object representing the SQL CLOB value in
     * the specified column
     */

//#ifdef JAVA2
    public Clob getClob(String columnName) throws SQLException {
	return getClob(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets an SQL ARRAY value in the current row of this <code>ResultSet</code>
     * object.
     *
     * @param colName the name of the column from which to retrieve the value
     * @return an <code>Array</code> object representing the SQL ARRAY value in
     * the specified column
     */

//#ifdef JAVA2
    public Array getArray(String columnName) throws SQLException {
	return getArray(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * Gets the value of a column in the current row as a java.sql.Date
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Date if the underlying database does not store
     * timezone information.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the calendar to use in constructing the date
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public java.sql.Date getDate(int column,
				 Calendar cal) throws SQLException {
	return getDate(column);
    }

//#endif JAVA2

    /**
     * Gets the value of a column in the current row as a java.sql.Date
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Date, if the underlying database does not store
     * timezone information.
     *
     * @param columnName the SQL name of the column from which to retrieve the
     * value
     * @param cal the calendar to use in constructing the date
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public java.sql.Date getDate(String columnName,
				 Calendar cal) throws SQLException {
	return getDate(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * Gets the value of a column in the current row as a java.sql.Time
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Time if the underlying database does not store
     * timezone information.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the calendar to use in constructing the time
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public Time getTime(int column, Calendar cal) throws SQLException {
	return getTime(column);
    }

//#endif JAVA2

    /**
     * Gets the value of a column in the current row as a java.sql.Time
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Time if the underlying database does not store
     * timezone information.
     *
     * @param columnName the SQL name of the column
     * @param cal the calendar to use in constructing the time
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public Time getTime(String columnName, Calendar cal) throws SQLException {
	return getTime(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * Gets the value of a column in the current row as a java.sql.Timestamp
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Timestamp if the underlying database does not store
     * timezone information.
     *
     * @param columnIndex the first column is 1, the second is 2, ...
     * @param cal the calendar to use in constructing the timestamp
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public Timestamp getTimestamp(int column,
				  Calendar cal) throws SQLException {
	return getTimestamp(column);
    }

//#endif JAVA2

    /**
     * Gets the value of a column in the current row as a java.sql.Timestamp
     * object. This method uses the given calendar to construct an appropriate
     * millisecond
     * value for the Timestamp if the underlying database does not store
     * timezone information.
     *
     * @param columnName the SQL name of the column
     * @param cal the calendar to use in constructing the timestamp
     * @return the column value; if the value is SQL NULL, the result is null
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public Timestamp getTimestamp(String columnName,
				  Calendar cal) throws SQLException {
	return getTimestamp(findColumn(columnName));
    }

//#endif JAVA2

    /**
     * JDBC 2.0
     *
     * <p>Gets the value of a column in the current row as a java.io.Reader.
     * @param columnIndex the first column is 1, the second is 2, ...
     */

//#ifdef JAVA2
    public Reader getCharacterStream(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2

    /**
     * Constructor declaration
     *
     *
     * @param r
     */
    jdbcResultSet(Result r) throws SQLException {
	if (r.iMode == Result.UPDATECOUNT) {
	    iUpdateCount = r.iUpdateCount;
	} else if (r.iMode == Result.ERROR) {
	    throw Trace.getError(r.sError);
	} else {
	    iUpdateCount = -1;
	    rResult = r;
	    iColumnCount = r.getColumnCount();
	}

	bWasNull = false;
    }

    /**
     * Method declaration
     *
     *
     * @return
     */
    int getUpdateCount() {
	return iUpdateCount;
    }

    /**
     * Method declaration
     *
     *
     * @return
     */
    boolean isResult() {
	return rResult == null ? false : true;
    }

    /**
     * Method declaration
     *
     *
     * @param column
     *
     * @throws SQLException
     */
    private void checkColumn(int column) throws SQLException {
	if (column < 0 || column >= iColumnCount) {
	    Trace.error(Trace.COLUMN_NOT_FOUND, column);
	}
    }

    /**
     * Method declaration
     *
     *
     * @throws SQLException
     */
    private void checkAvailable() throws SQLException {
	if (rResult == null ||!bInit || nCurrent == null) {
	    Trace.error(Trace.NO_DATA_IS_AVAILABLE);
	}
    }

    /**
     * Method declaration
     *
     *
     * @param column
     * @param type
     *
     * @return
     *
     * @throws SQLException
     */
    private Object getColumnInType(int column, int type) throws SQLException {
	checkColumn(column);
	checkAvailable();

	int    t = rResult.iType[column];
	Object o = nCurrent.data[column];

	checkNull(o);

	// no conversion necessary
	if (type == t) {
	    return o;
	}

	// try to convert
	try {
	    return Column.convertObject(o, type);
	} catch (Exception e) {
	    String s = "type: " + Column.getType(t) + " (" + t
		       + ") expected: " + Column.getType(type) + " value: "
		       + o.toString();

	    throw Trace.error(Trace.WRONG_DATA_TYPE, s);
	}
    }

    /**
     * Method declaration
     *
     *
     * @param o
     */
    private void checkNull(Object o) {
	if (o == null) {
	    bWasNull = true;
	} else {
	    bWasNull = false;
	}
    }

    // implementation of ResultSetMetaData

    /**
     * ResultSetMetaData:
     * Returns the number of columns in this ResultSet.
     *
     * @return the number of columns
     */
    public int getColumnCount() {
	return iColumnCount;
    }

    /**
     * ResultSetMetaData:
     * Indicates whether the column is automatically numbered, thus read-only.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isAutoIncrement(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;
    }

    /**
     * ResultSetMetaData:
     * Indicates whether a column's case matters.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isCaseSensitive(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return true;
    }

    /**
     * ResultSetMetaData:
     * Indicates whether the column can be used in a where clause.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isSearchable(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return true;
    }

    /**
     * ResultSetMetaData:
     * Indicates whether the column is a cash value.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isCurrency(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;    // todo
    }

    /**
     * ResultSetMetaData:
     * Indicates the nullability of values in the designated column.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return the nullability status of the given column; one of columnNoNulls,
     * columnNullable or columnNullableUnknown
     */
    public int isNullable(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return columnNullable;    // todo
    }

    /**
     * ResultSetMetaData:
     * Indicates whether values in the column are signed numbers.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isSigned(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return true;    // todo
    }

    /**
     * ResultSetMetaData:
     * Indicates the column's normal max width in chars.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return the normal maximum number of characters allowed as the width
     * of the designated column
     * @exception SQLException if a database access error occurs
     */
    public int getColumnDisplaySize(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	checkColumn(--column);

	// Some program expect that this is the maximum allowed length
	// for this column, so it is dangerous to return the size required
	// to display all records
	return 0;
    }

    /**
     * ResultSetMetaData:
     * Gets the suggested column title for use in printouts and
     * displays.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return the suggested column title
     * @exception SQLException if a database access error occurs
     */
    public String getColumnLabel(int column) throws SQLException {
	checkColumn(--column);

	return rResult.sLabel[column];
    }

    /**
     * ResultSetMetaData:
     * Gets a column's name.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return column name
     * @exception SQLException if a database access error occurs
     */
    public String getColumnName(int column) throws SQLException {
	checkColumn(--column);

	return rResult.sName[column];
    }

    /**
     * ResultSetMetaData:
     * Gets a column's table's schema.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return schema name or "" if not applicable
     */
    public String getSchemaName(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return "";
    }

    /**
     * ResultSetMetaData:
     * Gets a column's number of decimal digits.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return precision
     */
    public int getPrecision(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return 0;
    }

    /**
     * ResultSetMetaData:
     * Gets a column's number of digits to right of the decimal point.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return scale
     */
    public int getScale(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return 0;
    }

    /**
     * ResultSetMetaData:
     * Gets a column's table name.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return table name or "" if not applicable
     * @exception SQLException if a database access error occurs
     */
    public String getTableName(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	checkColumn(--column);

	return rResult.sTable[column];
    }

    /**
     * ResultSetMetaData:
     * Gets a column's table's catalog name.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return column name or "" if not applicable.
     */
    public String getCatalogName(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return "";
    }

    /**
     * ResultSetMetaData:
     * Retrieves a column's SQL type.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return SQL type from java.sql.Types
     * @exception SQLException if a database access error occurs
     */
    public int getColumnType(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	checkColumn(--column);

	return rResult.iType[column];
    }

    /**
     * ResultSetMetaData:
     * Retrieves a column's database-specific type name.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return type name used by the database. If the column type is
     * a user-defined type, then a fully-qualified type name is returned.
     * @exception SQLException if a database access error occurs
     */
    public String getColumnTypeName(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	checkColumn(--column);

	return Column.getType(rResult.iType[column]);
    }

    /**
     * ResultSetMetaData:
     * Indicates whether a column is definitely not writable.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isReadOnly(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return false;    // todo
    }

    /**
     * ResultSetMetaData:
     * Indicates whether it is possible for a write on the column to succeed.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isWritable(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return true;    // todo
    }

    /**
     * ResultSetMetaData:
     * Indicates whether a write on the column will definitely succeed.
     *
     * @param column the first column is 1, the second is 2, ...
     * @return true if so
     */
    public boolean isDefinitelyWritable(int column) {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	return true;    // todo
    }

    /**
     * ResultSetMetaData:
     * JDBC 2.0
     *
     * <p>Returns the fully-qualified name of the Java class whose instances
     * are manufactured if the method <code>ResultSet.getObject</code>
     * is called to retrieve a value
     * from the column.  <code>ResultSet.getObject</code> may return a subclass
     * of the class returned by this method.
     *
     * @return the fully-qualified name of the class in the Java programming
     * language that would be used by the method
     * <code>ResultSet.getObject</code> to retrieve the value in the specified
     * column. This is the class name used for custom mapping.
     * @exception SQLException if a database access error occurs
     */

//#ifdef JAVA2
    public String getColumnClassName(int column) throws SQLException {
	if (Trace.TRACE) {
	    Trace.trace();
	}

	throw Trace.error(Trace.FUNCTION_NOT_SUPPORTED);
    }

//#endif JAVA2
}
