/*++UWICal.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package edu.washington.cac.calendar.uwical;


import java.util.Date;
import java.util.Iterator;
import java.util.ListIterator;					/* for main */
import java.util.GregorianCalendar;
import java.util.Vector;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParsePosition;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.icalendar.Component;
import edu.washington.cac.calendar.icalendar.Property;
import edu.washington.cac.calendar.icalendar.Categories;
import edu.washington.cac.calendar.icalendar.Class;    // vs java.lang.Class
import edu.washington.cac.calendar.icalendar.Description;
import edu.washington.cac.calendar.icalendar.DTEnd;
import edu.washington.cac.calendar.icalendar.DTStamp;
import edu.washington.cac.calendar.icalendar.DTStart;
import edu.washington.cac.calendar.icalendar.Duration;
import edu.washington.cac.calendar.icalendar.ICalendar;
import edu.washington.cac.calendar.icalendar.ICalendarException;
import edu.washington.cac.calendar.icalendar.IncompleteComponentException;
import edu.washington.cac.calendar.icalendar.InvalidComponentException;
import edu.washington.cac.calendar.icalendar.Location;
import edu.washington.cac.calendar.icalendar.Names;
import edu.washington.cac.calendar.icalendar.NoValueException;
import edu.washington.cac.calendar.icalendar.ProdId;
import edu.washington.cac.calendar.icalendar.Properties;
import edu.washington.cac.calendar.icalendar.Summary;
import edu.washington.cac.calendar.icalendar.UID;
import edu.washington.cac.calendar.icalendar.URL;
import edu.washington.cac.calendar.icalendar.Strings;
import edu.washington.cac.calendar.icalendar.Utilities;
import edu.washington.cac.calendar.icalendar.Version;
import edu.washington.cac.calendar.icalendar.VEvent;
import edu.washington.cac.calendar.icalendar.XParam;
import edu.washington.cac.calendar.icalendar.XProp;

import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.Events;
import edu.washington.cac.calendar.data.EventsI;
import edu.washington.cac.calendar.data.ItemException;
import edu.washington.cac.calendar.data.ItemAccessException;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Locations;
import edu.washington.cac.calendar.data.NoRecurrence;
import edu.washington.cac.calendar.data.PersonalEvents;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.Sponsors;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.filter.FilteredEvents;
import edu.washington.cac.calendar.filter.PublicCalendars;

import edu.washington.cac.calendar.uwical.instrumentation.UCEventsDumper;


/**
 * Interface between UWCal and ICalendar.
 *
 * Adds methods to ICalendar to convert/interface to/from UWCal system.
 *
 * @author  slh, Greg Barnes
 * @version  0.10 2003/11/25 slh
 */
public class UWICal extends ICalendar
{

/*----------------------------------------------------------------------------
 *                                              Constructors
 *--------------------------------------------------------------------------*/

  /**
   * Create an empty calendar.
   */
  public
  UWICal  (             )
  {
    super(  );
  }


  /**
   * Create an empty calendar with ProdId and Version set.
   */
  public
  UWICal  (ProdId       prodid  ,
           Version      version )
  throws Exception      /* for calls */
  {
    super( prodid , version );
  }


  /**
   * Create a calendar from an ICalendar object.
   */
  public
  UWICal  (ICalendar	icalendar	)
  throws Exception
  {
    Iterator	iter;

    for (iter = icalendar.getComponents(  ).elements(  ) ;
	 iter.hasNext(  ) ; ) {
      /*void*/addComponent( (Component)iter.next(  ) );
    }
    for (iter = icalendar.getProperties(  ).elements(  ) ;
	 iter.hasNext(  ) ; ) {
      /*void*/addProperty( (Property)iter.next(  ) );
    }
  }


  /**
   * Create a calendar from ProdId, Version and VEvent objects.
   */
  public
  UWICal  (ProdId	prodid	,
	   Version	version	,
	   VEvent	vevent	)
  throws Exception
  {
    super( prodid , version );

    /*void*/addComponent( vevent );
  }


  /**
   * Create a calendar from an unencoded (unmimed, unfolded) ICalendar stream.
   */
  public
  UWICal  (String       strICalendar    )
  throws Exception
  {
    super( strICalendar );
  }


  /**
   * Create a calendar from UWCal events.
   */
  public
  UWICal  (EventsI	events  )
  throws Exception      /* for calls */
  {
    this(  );

    /*Note: addComponents() does null check*/

    addComponents( events );
  }


  /**
   * Create a calendar from UWCal events
   * with ProdId and Version set.
   */
  public
  UWICal  (ProdId       prodid  ,
           Version      version ,
           EventsI	events  )
  throws Exception      /* for calls */
  {
    this( prodid , version );

    if (events != null) {
      /*Note: addComponents() does null check*/
      addComponents( events );
    }
  }


  /**
   * Create a calendar from UWCal events
   * with ProdId and Version set.
   */
  public
  UWICal  (String       strProdid       ,
           String       strVersion      ,
           EventsI	events          )
  throws Exception      /* for calls */
  {
    super( strProdid , strVersion );

    if (events != null) {
      /*Note: addComponents() does null check*/
      addComponents( events );
    }
  }


  /**
   * Create a calendar from UWCal events
   * with ProdId and Version set.
   */
  public
  UWICal  (String	strProdid	,
	   String	strVersion	,
	   Iterator	iterEvent	,
	   EventsI	eventsi		)
  throws Exception	/* for calls */
  {
    super( strProdid , strVersion );

    if (iterEvent != null && eventsi != null) {
      /*Note: addComponents() does null check*/
      addComponents( iterEvent , eventsi );
    }
  }


/*----------------------------------------------------------------------------
 *                                              Object Public Methods
 *--------------------------------------------------------------------------*/

  /**
   * Add VEvent components based on UWCal events.
   */
  public
  void
  addComponents  (EventsI	eventsi	)
  throws Exception      /* NullPointerException */
  {
    /*void*/addComponents( eventsi.sortedElements(  ) , eventsi );
  }


  /**
   * Add VEvent components based on UWCal events.
   */
  public
  void
  addComponents  (Iterator	iterEvent	,
		  EventsI	eventsi		)
  throws Exception	/* NullPointerException */
  {
    Iterator iter;

    if (iterEvent == null) {
      throw new NullPointerException( getName(  ) +
		": (Iterator) iterEvent is null" );
    }
    if (eventsi == null) {
      throw new NullPointerException( getName(  ) +
		": (EventsI) eventsi is null" );
    }

    try {
      for ( ; iterEvent.hasNext(  ) ; ) {
        addUWCalEvent( (Event)iterEvent.next(  ) , eventsi );
      }
    } catch (InvalidComponentException	e	) {
/*???do something; throw if can be inputs fault*/
      e.printStackTrace(  );
    }
  }


  /**
     Convert an ICalendar VEvent into a UWCal event, saving the event to
     the UWCal db.

     Assumes: events's keywords, locations and sponsors have been loaded
     with all relevant entries from db.

     @param vevent ICalendar event to convert
     @param events cache of UWCal events
     @param user User for who we are doing this
     @exception ICalendarException If there's a problem with the VEvent
     @exception ItemException If the data in the event is bad
     @exception SQLException If there's a problem saving to the database
   */
  protected Event convertToUWCalEvent(VEvent vevent, EventsI events, User user)
      throws ICalendarException, ItemException, SQLException
  {
    return convertToUWCalEvent(vevent, events, user, true);
  }


  /**
     Convert an ICalendar VEvent into a UWCal event
   *
   * Assumes: events's keywords, locations and sponsors have been loaded
   * with all relevant entries from db.

     @param vevent ICalendar event to convert
     @param events cache of UWCal events
     @param user User for who we are doing this
     @param bSaveToDB Save event to the database?
     @exception ICalendarException If there's a problem with the VEvent
     @exception ItemException If the data in the event is bad
     @exception SQLException If there's a problem saving to the database
   */
  protected Event convertToUWCalEvent(VEvent vevent,
                                      EventsI events,
                                      User user,
                                      boolean bSaveToDB)
      throws ICalendarException, ItemException, SQLException
  {
    int         idx;
    Properties  properties;
    Iterator	iter;
    UID         uid;
    Summary     summary;
    Description description;
    DTStart     dtstart;
    DTEnd       dtend;
    Duration    duration;
    Class       classx;
    URL         url;
    XProp       xprop;
    Location    location;
    DTEnd       dtendNew;
    Event               event           = null;
    int                 Eventid         = 0;            /* default: no id */
    String              strShortdesc    = null;         /* default */
    String              strLongdesc     = null;         /* default */
    java.sql.Date       dateStartdate   = null;         /* default */
    java.sql.Time       timeStarttime   = null;         /* default */
    java.sql.Date       dateEnddate     = null;         /* default */
    java.sql.Time       timeEndtime     = null;         /* default */
    boolean             bIsPublic       = false;        /* default */
    String              strLink         = null;         /* default */
    User		creator		= user;		/* default */
    String              strCost         = null;         /* default */
    edu.washington.cac.calendar.data.Location   uclocation      = null;
    Sponsor             ucsponsor       = null;
    Vector              vuckeyword      = null;         /* of Keyword */
    Keywords            uckeywords;
    Locations           uclocations;
    Sponsors            ucsponsors;

    if (vevent == null || events == null || user == null) {
      throw new NullPointerException( getName(  ) +
                ": null argument passed" );
    }

    properties = vevent.getProperties();
    if (properties == null) {
      return null;                                              /*RETURN*/
    }

    if ((uid = (UID)properties.getFirst( Names.strProp_UID )) != null) {
      try {
        Eventid = Integer.parseInt( uid.get( true ) );
      } catch (NumberFormatException    e       ) {
        if (bSaveToDB) {
          Eventid = UNASSIGNED_ID;
        } else {
          /* if eventids are the same, events drops all but one */
          Eventid = (int)System.currentTimeMillis(  );
        }
      }
    }
//??? else problem

    if ((summary = (Summary)properties.getFirst( Names.strProp_Summary ))
        != null) {
      strShortdesc = summary.get( true );
    }

    if ((description =
         (Description)properties.getFirst( Names.strProp_Description ))
        != null) {
      strLongdesc = description.get( true );
    }

    if ((dtstart = (DTStart)properties.getFirst( Names.strProp_DTStart ))
        != null) {
      dateStartdate = new java.sql.Date( dtstart.get(  ).getTime(  ) );
      if (dtstart.getType(  ) == DTStart.DATETIME) {
        timeStarttime = new java.sql.Time( dtstart.get(  ).getTime(  ) );
      }
    }

    dtend = (DTEnd)properties.getFirst( Names.strProp_DTEnd );
    duration = (Duration)properties.getFirst( Names.strProp_Duration );
    if ((dtendNew = endstartdurationToEnd( dtend , dtstart , duration ))
        != null) {
      dateEnddate = new java.sql.Date( dtendNew.get(  ).getTime(  ) );
      if (dtendNew.getType(  ) == DTEnd.DATETIME) {
        timeEndtime = new java.sql.Time( dtendNew.get(  ).getTime(  ) );
      }
    }

    if ((classx = (Class)properties.getFirst( Names.strProp_Class ))
        != null) {
      bIsPublic = classx.getEnum(  ) == Class.PUBLIC;
    }

    if ((url = (URL)properties.getFirst( Names.strProp_URL )) != null) {
      strLink = url.get(  );
    }

    if ((xprop =
         (XProp)properties.getFirst( makeXName( XNames.strProp_Creator ) ))
         != null)
    {
      creator = new User( xprop.get( true ) );
    }

    if ((xprop = (XProp)properties.getFirst(
                makeXName( XNames.strProp_Cost ) ))
         != null) {
      strCost = xprop.get( true );
    }

    location = (Location)properties.getFirst( Names.strProp_Location );
    uclocation = LocationToUWCalLocation( location , user);
                                        /* works for null location */

    xprop = (XProp)properties.getFirst( makeXName( XNames.strProp_Sponsor ) );
    ucsponsor = XPropToUWCalSponsor( xprop , user );
                                        /* works for null xprop */

    iter = properties.get( Names.strProp_Categories );
    vuckeyword = CategoriesEnumToUWCalKeywordVector( iter , user );
                                        /* works for null iter */

/*???other properties into some catchall*/

    uckeywords = events.getKeywords(  );
    uclocations = events.getLocations(  );
    ucsponsors = events.getSponsors(  );

/*???checkadd's need to be able to do updates*/
    checkaddKeywords( uckeywords , vuckeyword , bSaveToDB );
    checkaddLocation( uclocations , uclocation , bSaveToDB );
    checkaddSponsor( ucsponsors , ucsponsor , bSaveToDB );

    event = new Event(Eventid, strShortdesc, strLongdesc, dateStartdate,
                      timeStarttime, dateEnddate, timeEndtime, bIsPublic,
                      strLink, creator, strCost, NoRecurrence.NO_RECUR,
                      uclocation , ucsponsor,
                      0,
                      new Timestamp(new java.util.Date().getTime()),
                      new Timestamp(new java.util.Date().getTime()));

    for (idx = 0 ; idx < vuckeyword.size(  ) ; idx++) {
      event.addKeyword( ((Keyword)vuckeyword.elementAt( idx )).getId(  ) );
    }

/*???checkaddEvent() (or update), areEventsEqual()*/
    events.add( event , bSaveToDB );

    return event;
  }


  /**
     Convert this icalendar into a UWCal events object, and save the events
     to the UWCal database.

     @param user User to create events for
     @exception ICalendarException If there's a problem with the VEvent
     @exception ItemException If the data in the event is bad
     @exception SQLException If there's a problem saving to the database
   */
  public
  EventsI
  convertToUWCalEvents  (User	user	)
  throws ICalendarException, ItemException, SQLException
  {
    return convertToUWCalEvents( user , true );
  }


  /**
     Convert this icalendar into a UWCal events object.

     @param user User to create events for
     @param saveToDB Save events to db?
     @exception ICalendarException If there's a problem with the VEvent
     @exception ItemException If the data in the event is bad
     @exception SQLException If there's a problem saving to the database
   */
  public
  EventsI
  convertToUWCalEvents  (User		user		,
			 boolean	saveToDB	)
  throws ICalendarException, ItemException, SQLException
  {
    EventsI	eventsi;
    Iterator	iter;

    if (user == null) {
      throw new IllegalArgumentException(getName() + ": user is null");
    }

    if (!isComplete()) {
      throw new IncompleteComponentException( getName(  ) +
		": convertToUWCalEvents()" +
		": attempt to use an incomplete ICalendar." );
    }

/*???no place for ICalendar properties in Events */
    if        (user != null && !user.isGuest(  )) {
      eventsi = new PersonalEvents(user, false);
    } else {
      eventsi = PublicEvents.getPublicEvents(  );
//???right now users don't have keywords/sponsors and setting them is not allowed
      /*void*/eventsi.setKeywords(new Keywords(user, true));
      /*void*/eventsi.setSponsors(new Sponsors(user, true));
    }

    iter = m_components.get( Names.strComp_VEvent );

    while (iter.hasNext(  )) {
      /*(void)*/convertToUWCalEvent(
		(VEvent)iter.next(  ) , eventsi , user , saveToDB );
    }

    return eventsi;
  }


/*----------------------------------------------------------------------------
 *                                              Object Protected Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- UWCal Support - Saving         */
  /*
   * xref: UWCalKeywordIdEnumToCategories()
   * xref: CategoriesEnumToUWCalKeywordVector()
   */
  protected
  boolean  areKeywordsEqual  (Keyword   uckOld  ,
                              Keyword   uckNew  )
  {
    /*Note: id not supplied by ical/ct,
      but attempt made to supply it to ical/ct through xparam;
      currently ct not returning it;
      i.e. currently this is never true*/
    /* if ids are not equal and neither is unassigned... */
    if (uckOld.getId(  ) != uckNew.getId(  ) &&
        uckOld.getId(  ) != UNASSIGNED_ID &&
        uckNew.getId(  ) != UNASSIGNED_ID) {
      return false;                                             /*RETURN*/
    }

    /*Note: class not supplied by ical/ct,
      and currently no attempt to supply to or extract from ical stream;
      if class is not found (as is currenly always so), it is set to private*/
    if (uckOld.isPublic(  ) != uckNew.isPublic(  )) {
      return false;                                             /*RETURN*/
    }
    /*Note: creator not supplied by ical/ct,
      and currently no attempt to supply to or extract from ical stream;
      if creator is not found (as is currenly always so),
      it is set to user's id*/
    /* if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (uckOld.getCreator(  ) != uckNew.getCreator(  ) &&
        (uckOld.getCreator(  ) != null || uckNew.getCreator(  ) != null) &&
        !uckOld.getCreator(  ).equals( uckNew.getCreator(  ) )) {
      return false;                                             /*RETURN*/
    }

/*???longdesc: not supplied by ical/ct (null),
  currently no attempt to supply to or extract from ical stream*/

    /* if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (uckOld.getWord(  ) != uckNew.getWord(  ) &&
        (uckOld.getWord(  ) != null || uckNew.getWord(  ) != null) &&
        !uckOld.getWord(  ).equals( uckNew.getWord(  ) )) {
      return false;                                             /*RETURN*/
    }

    return true;                                                /*RETURN*/
  }


  /**
    @exception ItemAccessException if the sponsor cannot be added to the cache
   */
  protected
  void  checkaddKeywords  (Keywords     uckeywordsOld   ,
                           Vector       vuckeywordNew   ,
                           boolean      bSaveToDB       )
  throws SQLException, ItemAccessException
  {
    Iterator		iterOld;
    boolean             bMatch;
    Keyword             uckeywordOld    = null;         // set for compiler
    Keyword             uckeywordNew;
    int                 idx;

    if (uckeywordsOld == null) {
      throw new NullPointerException( getName(  ) +
                ": uckeywordsOld (Keywords) is null" );
    }

    if (vuckeywordNew == null) {
      return;                                                   /*RETURN*/
    }

    for (idx = 0 ; idx < vuckeywordNew.size(  ) ; idx++) {
      uckeywordNew = (Keyword)vuckeywordNew.elementAt( idx );
      bMatch = false;
      iterOld = uckeywordsOld.elements(  );
      for ( ; iterOld.hasNext(  ) && !bMatch ; ) {
        uckeywordOld = (Keyword)iterOld.next(  );
        if (areKeywordsEqual( uckeywordOld , uckeywordNew )) {
          bMatch = true;
        }
      }
      if (bMatch) {
        uckeywordNew.setId( uckeywordOld.getId(  ) );
      } else {
        uckeywordsOld.add( uckeywordNew , bSaveToDB );
      }
    }
  }


  /*
   * xref: UWCalLocationToLocation()
   * xref: LocationToUWCalLocation()
   */
  protected
  boolean  areLocationsEqual
                (edu.washington.cac.calendar.data.Location      uclOld  ,
                edu.washington.cac.calendar.data.Location       uclNew  )
  {
    /*Note: id not supplied by ical/ct,
      but attempt made to supply it to ical/ct through xparam;
      currently ct not returning it;
      i.e. currently this is never true*/
    if (uclOld.getId(  ) != uclNew.getId(  ) &&
        uclOld.getId(  ) != UNASSIGNED_ID &&
        uclNew.getId(  ) != UNASSIGNED_ID) {
      return false;                                             /*RETURN*/
    }

    /*Note: class not supplied by ical/ct,
      but attempt made to supply it to ical/ct through xparam;
      currently ct not returning it;
      if class is not found (as is currenly always so), set to private*/
    if (uclOld.isPublic(  ) != uclNew.isPublic(  )) {
      return false;                                             /*RETURN*/
    }
    /*Note: creator not supplied by ical/ct,
      but attempt made to supply it to ical/ct through xparam;
      currently ct not returning it;
      if creator is not found (as is currenly always so), set to user's id*/
    /* if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (uclOld.getCreator(  ) != uclNew.getCreator(  ) &&
        (uclOld.getCreator(  ) != null || uclNew.getCreator(  ) != null) &&
        !uclOld.getCreator(  ).equals( uclNew.getCreator(  ) )) {
      return false;                                             /*RETURN*/
    }

    /*Note: these are supplied together (one per line) as the value and
      also individually in xparams;
      currently ct returns them together as the value,
      but not individually as xparams*/
    /* (for each) if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (uclOld.getAddress(  ) != uclNew.getAddress(  ) &&
        (uclOld.getAddress(  ) != null || uclNew.getAddress(  ) != null) &&
        !uclOld.getAddress(  ).equals( uclNew.getAddress(  ) ) ||
        uclOld.getSubaddress(  ) != uclNew.getSubaddress(  ) &&
        (uclOld.getSubaddress(  ) != null ||
         uclNew.getSubaddress(  ) != null) &&
        !uclOld.getSubaddress(  ).equals( uclNew.getSubaddress(  ) ) ||
        uclOld.getLink(  ) != uclNew.getLink(  ) &&
        (uclOld.getLink(  ) != null || uclNew.getLink(  ) !=null) &&
        !uclOld.getLink(  ).equals( uclNew.getLink(  ) )) {
      return false;                                             /*RETURN*/
    }

    return true;                                                /*RETURN*/
  }


  /**
    @exception ItemAccessException if the sponsor cannot be added to the cache
   */
  protected
  void  checkaddLocation
        (Locations     uclocationsOld  ,
         edu.washington.cac.calendar.data.Location      uclocationNew   ,
         boolean                                        bSaveToDB       )
  throws SQLException, ItemAccessException
  {
    Iterator		iterOld;
    boolean             bMatch;
    edu.washington.cac.calendar.data.Location   uclocationOld   = null;
                                                        // set for compiler

    if (uclocationsOld == null) {
      throw new NullPointerException( getName(  ) +
                ": uclocationsOld (Locations) is null" );
    }

    if (uclocationNew == null) {
      return;                                                   /*RETURN*/
    }

    bMatch = false;
    iterOld = uclocationsOld.elements(  );
    for ( ; iterOld.hasNext(  ) && !bMatch ; ) {
      uclocationOld =
        (edu.washington.cac.calendar.data.Location)iterOld.next(  );
      if (areLocationsEqual( uclocationOld , uclocationNew )) {
        bMatch = true;
      }
    }
    if (bMatch) {
      uclocationNew.setId( uclocationOld.getId(  ) );
    } else {
      uclocationsOld.add( uclocationNew , bSaveToDB );
    }
  }


  /*
   * xref: UWCalSponsorToXProp()
   * xref: XPropToUWCalSponsor()
   */
  protected
  boolean  areSponsorsEqual  (Sponsor   ucsOld  ,
                              Sponsor   ucsNew  )
  {
    /*Note: sponsor not supplied by ical/ct,
      but attempt made to supply it to ical/ct through xprop and xparams;
      currently ct not returning it;
      i.e. currently this method never returns true*/

    if (ucsOld.getId(  ) != ucsNew.getId(  ) &&
        ucsOld.getId(  ) != UNASSIGNED_ID &&
        ucsNew.getId(  ) != UNASSIGNED_ID) {
      return false;                                             /*RETURN*/
    }

    if (ucsOld.isPublic(  ) != ucsNew.isPublic(  )) {
      return false;                                             /*RETURN*/
    }
    /* if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (ucsOld.getCreator(  ) != ucsNew.getCreator(  ) &&
        (ucsOld.getCreator(  ) != null || ucsNew.getCreator(  ) != null) &&
        !ucsOld.getCreator(  ).equals( ucsNew.getCreator(  ) )) {
      return false;                                             /*RETURN*/
    }

    /* (for each) if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (ucsOld.getPhone(  ) != ucsNew.getPhone(  ) &&
        (ucsOld.getPhone(  ) != null || ucsNew.getPhone(  ) != null) &&
        !ucsOld.getPhone(  ).equals( ucsNew.getPhone(  ) ) ||
        ucsOld.getEmail(  ) != ucsNew.getEmail(  ) &&
        (ucsOld.getEmail(  ) != null || ucsNew.getEmail(  ) != null) &&
        !ucsOld.getEmail(  ).equals( ucsNew.getEmail(  ) ) ||
        ucsOld.getLink(  ) != ucsNew.getLink(  ) &&
        (ucsOld.getLink(  ) != null || ucsNew.getLink(  ) != null) &&
        !ucsOld.getLink(  ).equals( ucsNew.getLink(  ) )) {
      return false;                                             /*RETURN*/
    }

    /* if not the same (string) object and
       at least one object is not null and
       (so it is possible that) the strings are not the same... */
    if (ucsOld.getName(  ) != ucsNew.getName(  ) &&
        (ucsOld.getName(  ) != null || ucsNew.getName(  ) != null) &&
        !ucsOld.getName(  ).equals( ucsNew.getName(  ) )) {
      return false;                                             /*RETURN*/
    }

    return true;                                                /*RETURN*/
  }


  /**
    @exception ItemAccessException if the sponsor cannot be added to the cache
   */
  protected
  void  checkaddSponsor  (Sponsors      ucsponsors      ,
                          Sponsor       ucsponsor       ,
                          boolean       bSaveToDB       )
  throws SQLException, ItemAccessException
  {
    Iterator	iter;
    boolean     bMatch;

    if (ucsponsors == null) {
      throw new NullPointerException( getName(  ) +
                ": ucsponsors (Sponsors) is null" );
    }

    if (ucsponsor == null) {
      return;                                                   /*RETURN*/
    }

    iter = ucsponsors.elements(  );
    bMatch = false;
    for ( ; iter.hasNext(  ) && !bMatch ; ) {
      if (areSponsorsEqual( (Sponsor)iter.next(  ) , ucsponsor )) {
        bMatch = true;
      }
    }
    if (!bMatch) {
      ucsponsors.add( ucsponsor , bSaveToDB );
    }
  }


/*-------------------------------------- UWCal Support - Misc           */

  protected
  boolean
  isSponsor  (Sponsor   ucsponsor       )
  {
    return ucsponsor.getId(  ) != Sponsor.NO_SPONSOR_ID &&
        ucsponsor.getId(  ) != Sponsor.UNKNOWN_SPONSOR_ID &&
        ucsponsor.getId(  ) != Sponsor.DELETED_SPONSOR_ID;
  }


  protected
  String
  publicToClass  (boolean       bPublic )
  {
    return bPublic ? Names.strValue_Public : Names.strValue_Private;
  }


  protected
  boolean
  classToPublic  (String        strClass        )
  {
    return strClass.equalsIgnoreCase( Names.strValue_Public );
  }


  /**
   * Converts an ending date/time or a start date/time and duration
   * to an ending date/time.
   */
  protected
  DTEnd
  endstartdurationToEnd  (DTEnd         dtend           ,
                          DTStart       dtstart         ,
                          Duration      duration        )
  {
    GregorianCalendar   gc;
    int			Dur;
    DTEnd               dtendNew        = null;

    if (dtend != null) {
//???clone
      dtendNew = new DTEnd(  );
      /*void*/dtendNew.set( dtend.get(  ) , dtend.getType(  ) );
    } else if (dtstart != null && duration != null) {
      gc = new GregorianCalendar(  );
      /*void*/gc.setTime( dtstart.get(  ) );
      try {
	Dur = duration.get(  );
      } catch (NoValueException	e	) {
	Dur = 0;
      }
      /*void*/gc.add( GregorianCalendar.SECOND , Dur );

      dtendNew = new DTEnd(  );
      /*void*/dtendNew.set( gc.getTime(  ) , DTEnd.DATETIME );
    }

    return dtendNew;
  }


/*-------------------------------------- UWCal Support - Conversion     */

  /*
   * xref: CategoriesEnumToUWCalKeywordVector(), areKeywordsEqual()
   */
  protected
  Categories
  UWCalKeywordIdEnumToCategories  (Iterator	iterId          ,
                                   Keywords     uckeywords      )
  {
    Integer             intId;
    Keyword             uckeyword;
    String              strWord;
    XParam              xparam;
    Categories          categories      = null;
/*???getLongdesc,getCreator,isPublic / Description,Creator,Class*/
/*???requires either a list of each or putting each keyword in a diff category prop*/

    if (iterId != null && iterId.hasNext(  ) &&
        uckeywords != null) {
      categories = new Categories(  );
      xparam = new XParam( makeXName( XNames.strParam_Id ) );
      try {
        categories.addParameter( xparam );
      } catch (ICalendarException       e       ) {
        ;       /* ignore: XParam valid: should not be possible */
      }

      for ( ; iterId.hasNext(  ) ; ) {
        try {
          intId = (Integer)iterId.next(  );
          uckeyword = uckeywords.getKeyword( intId.intValue(  ) );
          strWord = uckeyword.getWord(  );
          /* any exception should have happen by this point */
          categories.addText( strWord , true );
          xparam.addText( intId.toString(  ) );
        } catch (Exception      e       ) {
          ;     /* ignore any entries with errors */
        }
      }
    }

    return categories;
  }


  /**
   Convert a set of ICalendar categories to a set of UWCal keywords

   @param iterCategories The set of categories
   @param user The user to convert for
   @exception CaldataException if there is bad data in the categories
   @see UWCalKeywordIdEnumToCategories(), areKeywordsEqual()
   */
  protected Vector
      CategoriesEnumToUWCalKeywordVector(Iterator iterCategories,
                                         User user)
      throws CaldataException
  {
    Categories          categories;
    Iterator		iterValues, iterIds;
    String              strValue;
    int                 Id;
    Keyword             uckeyword;
    Vector      vuckeyword      = new Vector(  );       /* of Keyword */

/*???getLongdesc,getCreator,isPublic / Description,Creator,Class*/
/*???requires either a list of each or putting each keyword in a diff category prop*/
/*???creator,class for now or if not found, assign as user and private*/

    for ( ; iterCategories.hasNext(  ) ; ) {
      categories = (Categories)iterCategories.next(  );
      iterValues = categories.getList(  );
      try {
        iterIds = ((XParam)categories.getParameters(  ).
                   getFirst( makeXName( XNames.strParam_Id ) )).getList(  );
      } catch (Exception        e       ) {
        /* anything goes wrong, no id */
        iterIds = null;
      }
      for ( ; iterValues.hasNext(  ) ; ) {
        strValue = categories.checkUnescapeText(
			(String)iterValues.next(  ) );
        try {
	  /*Note: should not have been quoted, but just incase...*/
          Id = Integer.parseInt(
			XParam.checkUnquoteText( (String)iterIds.next(  ) ) );
        } catch (Exception      e       ) {
          /* null iterIds or anything goes wrong, no id */
          Id = UNASSIGNED_ID;
        }

        uckeyword = new Keyword( Id , strValue , null , user, false );
        vuckeyword.addElement( uckeyword );
      }
    }

    return vuckeyword;
  }


  /*
   * xref: LocationToUWCalLocation(), areLocationsEqual()
   */
  protected
  Location
  UWCalLocationToLocation
        (edu.washington.cac.calendar.data.Location      uclocation      )
  throws Exception
  {
    XParam      xparam;
    Location    location        = null;

    if (uclocation != null && uclocation.isLocation(  )) {
      location = new Location(  );
      location.set(
                (uclocation.getAddress(  ) == null
                 ? "" : uclocation.getAddress(  )) +
                mc_strLocationTokenDelim +
                (uclocation.getSubaddress(  ) == null
                 ? "" : uclocation.getSubaddress(  )) +
                mc_strLocationTokenDelim +
                (uclocation.getLink(  ) == null
                 ? "" : uclocation.getLink(  )) , true );

      xparam = new XParam( makeXName( XNames.strParam_Id ) ,
                           Integer.toString( uclocation.getId(  ) ) );
      location.addParameter( xparam );

      if (uclocation.getAddress(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Address ) );
        xparam.addText( uclocation.getAddress(  ) , true );
        location.addParameter( xparam );
      }
      if (uclocation.getSubaddress(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Address2 ) );
        xparam.addText( uclocation.getSubaddress(  ) , true );
        location.addParameter( xparam );
      }
      if (uclocation.getLink(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_URL ) );
        xparam.addText( uclocation.getLink(  ) , true );
        location.addParameter( xparam );
      }

      if (uclocation.getCreator(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Creator ) );
        xparam.addText( uclocation.getCreator(  ).getNameDB(  ) , true );
        location.addParameter( xparam );
      }

      xparam = new XParam( makeXName( XNames.strParam_Class ) );
      xparam.addText( publicToClass( uclocation.isPublic(  ) ) );
      location.addParameter( xparam );
    }

    return location;
  }


  /*
   Convert an iCalendar location to a UWCal location

   @param location iCalendar location
   @param user User to convert for
   @exception CaldataException If the location has bad data
   @see UWCalLocationToLocation(), areLocationsEqual()
   */
  protected edu.washington.cac.calendar.data.Location
      LocationToUWCalLocation(Location location, User user)
      throws CaldataException
  {
    String              strSrc;
    ParsePosition       ppSrc;
    String              strToken;
    int                 cDelim;
    XParam              xparam;
    int                 Id              = UNASSIGNED_ID;
    String              strAddress      = null;
    String              strSubaddress   = null;
    String              strLink         = null;
    User creator = user;
    boolean             bPublic         = false;
    edu.washington.cac.calendar.data.Location   uclocation      = null;

    if (location != null) {
      strSrc = location.get( true );
      ppSrc = new ParsePosition( 0 );
      cDelim = 0;
      for ( ;
           null != (strToken = nextToken(
                                strSrc , ppSrc , mc_strLocationTokenDelim )) ;
           ) {
        if (strToken.equals( mc_strLocationTokenDelim )) {
          cDelim++;
        } else {
          switch (cDelim) {
          case 0:
            strAddress = strToken;
            break;
          case 1:
            strSubaddress = strToken;
            break;
          case 2:
            strLink = strToken;
            break;
          default:
            break;
          }
        }
      }

      /* if was not possibly generated by us... */
      if (cDelim != 2) {
        strAddress = location.get( true );
        strSubaddress = null;
        strLink = null;
      }

      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Id ) ))
            != null) {
	  /*Note: should not have been quoted, but just incase...*/
          Id = Integer.parseInt( xparam.getFirstText( true ) );
        }
      } catch (NullPointerException     e       ) {
        ;       /* ignore and Id unset */
      } catch (NoValueException e       ) {
        ;       /* ignore and Id unset */
      } catch (ICalendarException       e       ) {
        ;       /* ignore and Id unset: should not be possible... */
      }

      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Address ) ))
            != null) {
          if (!strAddress.equals( xparam.getFirstText( true ) )) {
            strAddress = null;
          }
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }
      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Address2 ) ))
            != null) {
          if (!strSubaddress.equals( xparam.getFirstText( true ) )) {
            strSubaddress = null;
          }
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }
      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_URL ) ))
            != null) {
          if (!strLink.equals( xparam.getFirstText( true ) )) {
            strLink = null;
          }
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Creator ) ))
            != null)
        {
          creator = new User( xparam.getFirstText( true ) );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      try {
        if ((xparam = (XParam)location.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Class ) ))
            != null) {
	  /*Note: should not have been quoted, but just incase...*/
          bPublic = classToPublic( xparam.getFirstText( true ) );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      uclocation = new edu.washington.cac.calendar.data.Location(
        Id , strAddress , strSubaddress , strLink , creator, bPublic );
    }

    return uclocation;
  }


  /*
   * xref: XPropToUWCalSponsor(), areSponsorsEqual()
   */
  protected
  XProp
  UWCalSponsorToXProp  (Sponsor ucsponsor       )
  throws Exception
  {
    XProp       xprop           = null;
    XParam      xparam          = null;

    if (isSponsor( ucsponsor )) {
      xprop = new XProp( makeXName( XNames.strProp_Sponsor ) ,
                         ucsponsor.getName(  ) );

      xparam = new XParam( makeXName( XNames.strParam_Id ) ,
                           Integer.toString( ucsponsor.getId(  ) ) );
      xprop.addParameter( xparam );
      if (ucsponsor.getPhone(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Phone ) );
        xparam.addText( ucsponsor.getPhone(  ) , true );
        xprop.addParameter( xparam );
      }
      if (ucsponsor.getEmail(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Email ) );
        xparam.addText( ucsponsor.getEmail(  ) , true );
        xprop.addParameter( xparam );
      }
      if (ucsponsor.getLink(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_URL ) );
        xparam.addText( ucsponsor.getLink(  ) , true );
        xprop.addParameter( xparam );
      }
      if (ucsponsor.getCreator(  ) != null) {
        xparam = new XParam( makeXName( XNames.strParam_Creator ) );
        xparam.addText( ucsponsor.getCreator(  ).getNameDB(  ) , true );
        xprop.addParameter( xparam );
      }
      xparam = new XParam( makeXName( XNames.strParam_Class ) );
      xparam.addText( publicToClass( ucsponsor.isPublic(  ) ) );
      xprop.addParameter( xparam );
    }

    return xprop;
  }


  /*
   Convert an iCalendar sponsor to a UWCal sponsor

   @param xprop iCalendar xproperties possibly containing a sponsor
   @param user User to do conversion for
   @exception CaldataException if there's bad data
   @see UWCalSponsorToXProp(), areSponsorsEqual()
   */
  protected Sponsor XPropToUWCalSponsor(XProp xprop, User user)
      throws CaldataException
  {
    XParam      xparam;
    String      strName         = null;
    int         Id              = UNASSIGNED_ID;
    String      strPhone        = null;
    String      strEmail        = null;
    String      strURL          = null;
    User creator = user;
    boolean     bPublic         = false;
    Sponsor     ucsponsor       = null;

    if (xprop != null) {
      strName = xprop.get( true );

      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Id ) ))
            != null) {
	  /*Note: should not have been quoted, but just incase...*/
          Id = Integer.parseInt( xparam.getFirstText( true ) );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }
      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Phone ) ))
            != null) {
          strPhone = xparam.getFirstText( true );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }
      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Email ) ))
            != null) {
          strEmail = xparam.getFirstText( true );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }
      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_URL ) ))
            != null) {
          strURL = xparam.getFirstText( true );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Creator ) ))
            != null)
        {
          creator = new User( xparam.getFirstText( true ) );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      try {
        if ((xparam = (XParam)xprop.getParameters(  ).getFirst(
                makeXName( XNames.strParam_Class ) ))
            != null) {
	  /*Note: should not have been quoted, but just incase...*/
          bPublic = classToPublic( xparam.getFirstText( true ) );
        }
      } catch (Exception        e       ) {
        ;       /* ignore and unset */
      }

      ucsponsor = new Sponsor( Id , strName , strPhone , strEmail ,
                               strURL , creator, bPublic );
    }

    return ucsponsor;
  }


/*-------------------------------------- UWCal Support - addUWCalEvent  */

  protected
  void
  addUWCalEvent  (Event         event   ,
                  EventsI	events  )
  throws Exception      /* NullPointerException */
  {
    VEvent      vevent          = new VEvent(  );
    Class       classx          = new Class(  );
/*unused
    Created     created         = new Created(  );
*/
    Description description     = new Description(  );
    DTStart     dtstart         = new DTStart(  );
/*ununsed
    Geo         geo             = new Geo(  );
*/
/*unused
    LastModified        lastmodifled    = new LastModified(  );
*/
    Location    location        = null;
/*unused
    Organizer   organizer       = new Organizer(  );
*/
/*unused:
    Priority    priority        = new Priority(  );
*/
    DTStamp     dtstamp         = new DTStamp(  );
/*unused
    Sequence    sequence        = new Sequence(  );
    Status      status          = new Status(  );
*/
    Summary     summary         = new Summary(  );
/*unused
    Transp      transp          = new Transp(  );
*/
    UID         uid             = new UID(  );
    URL         url             = new URL(  );
/*unimplemented
recurid
*/
    DTEnd       dtend           = new DTEnd(  );
/*unimplemented
duration
attach
*/
/*unused
    Attendee    attendee        = new Attendee(  );
*/
    Categories  categories      = null;
/*unused
    Comment     comment         = new Comment(  );
    Contact     contact         = new Contact(  );
*/
/*unimplemented
exdate, exrule, rstatus, related
*/
/*unused
    Resources   resources       = new Resources(  );
*/
/*unimplemented
rdate, rrule
*/
    XProp       xprop;

    if (event == null) {
      throw new NullPointerException( getName(  ) +
                ": event (Event) is null" );
    }
    if (events == null) {
      throw new NullPointerException( getName(  ) +
                ": events (Events) is null" );
    }

    classx.setEnum( event.isPublic(  ) ? Class.PUBLIC : Class.PRIVATE );
    vevent.addProperty( classx );

/*
    created.set( ... );
    vevent.addProperty( created );
*/

    if (event.getLongdesc(  ) != null) {
      description.set( event.getLongdesc(  ) , true );
      vevent.addProperty( description );
    }

    if (event.startCalendar(  ) != null) {
      dtstart.set( event.startCalendar(  ).getTime(  ) ,
                   event.getStarttime(  ) != null
                   ? DTStart.DATETIME : DTStart.DATE );
      vevent.addProperty( dtstart );
    }

/*
    geo.set( ... );
    vevent.addProperty( geo );

    lastmodified.set( ... );
    vevent.addProperty( lastmodified );
*/

    if ((location = UWCalLocationToLocation( event.getLocation( events ) ))
        != null) {
      vevent.addProperty( location );
    }

/*
    organizer.set( ... );
    vevent.addProperty( organizer );
*/

/*
    priority.set( ... );
    vevent.addProperty( priority );
*/

    dtstamp.set( new java.util.Date(  ) , dtstamp.DATETIME );
    vevent.addProperty( dtstamp );

/*
    sequence.set( ... );
    vevent.addProperty( sequence );
    status.set( ... );
    vevent.addProperty( status );
*/

    if (event.getShortdesc(  ) != null) {
      summary.set( event.getShortdesc(  ) , true );
      vevent.addProperty( summary );
    }

/*
    transp.set( ... );
    vevent.addProperty( transp );
*/

/*???to string-valued id when avail*/
    uid.set( Integer.toString( event.getId(  ) ) , true );
    vevent.addProperty( uid );
    if (event.getLink(  ) != null) {
      url.set( event.getLink(  ) , true );
      vevent.addProperty( url );
    }

/*
recurid
*/

    if (event.endCalendar(  ) != null) {
      dtend.set( event.endCalendar(  ).getTime(  ) ,
                 event.getEndtime(  ) != null
                 ? DTEnd.DATETIME : DTEnd.DATE );
      vevent.addProperty( dtend );
    }

/*
duration
attach
*/
/*
    attendee.set( ... );
    vevent.addProperty( attendee );
*/

    if ((categories = UWCalKeywordIdEnumToCategories(
                        event.getKeywords(  ) , events.getKeywords(  ) ))
        != null) {
      vevent.addProperty( categories );
    }

/*
    comment.set( ... );
    vevent.addProperty( comment );
    contact.set( ... );
    vevent.addProperty( contact );
*/
/*
exdate, exrule, rstatus, related
*/
/*
    resources.set( ... );
    vevent.addProperty( resources );
*/
/*
rdate, rrule
*/

    if (event.getCreator() != null) {
      xprop = new XProp( makeXName( XNames.strProp_Creator ) );
      /*void*/xprop.set( event.getCreator(  ).getNameDB(  ) , true );
      vevent.addProperty( xprop );
    }

    if (event.getCost(  ) != null) {
      xprop = new XProp( makeXName( XNames.strProp_Cost ) );
      /*void*/xprop.set( event.getCost(  ) , true );
      vevent.addProperty( xprop );
    }

    if ((xprop = UWCalSponsorToXProp( event.getSponsor( events ) ))
        != null) {
      vevent.addProperty( xprop );
    }

    addComponent( vevent );
  }


/*----------------------------------------------------------------------------
 *                                              Class Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Miscellaneous                       */

  static public
  String
  makeXName  (String    strName )
  {
    return "X-" + mc_strVendorId + "-" + strName;
  }


/*----------------------------------------------------------------------------
 *                                              Class Private Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Miscellaneous                  */

  /*
   * Return: next token if any, otherwise null.
   */
  static private
  String
  nextToken  (String            strSrc          , // string being parsed
              ParsePosition     ppSrc           , // current location
              String            strDelim        )
  {
    int         iBeg;
    int         iEnd;   // actually, one past
    String      strToken;

    iBeg = ppSrc.getIndex(  );

    if (iBeg >= strSrc.length(  )) {
      return null;                                              /*RETURN*/
    }

    iEnd = strSrc.indexOf( strDelim , iBeg );
    /* if delim found... */
    if (iEnd != -1) {
      /* if token is delim... */
      if (iBeg == iEnd) {
        iEnd = iEnd + strDelim.length(  );
      }
    } else {
      iEnd = strSrc.length(  );
    }

    if (iEnd >= iBeg) {
      strToken = strSrc.substring( iBeg , iEnd );
    } else {
      strToken = null;
    }

    ppSrc.setIndex( iEnd );
    return strToken;                                            /*RETURN*/
  }


/*----------------------------------------------------------------------------
 *                                              Class Public Constants
 *--------------------------------------------------------------------------*/
  final static public  String   mc_strModuleName                = "UWICal";


/*----------------------------------------------------------------------------
 *                                              Class Private Constants
 *--------------------------------------------------------------------------*/

  final static private  String  mc_strVendorId                  = "UWI";

  final static private  String  mc_strLocationTokenDelim        = "\n";

  final static private  int     UNASSIGNED_ID                   = -1;

  final static private  String  mc_strSwitch                    = "-";


/*----------------------------------------------------------------------------
 *                                              Main
 *--------------------------------------------------------------------------*/

  static private
  void
  exit  (int            ExitCode        ,
         Throwable      throwable       )
  {
    if (throwable != null) {
      /*void*/System.err.println( throwable.getLocalizedMessage(  ) );
      /*void*/throwable.printStackTrace(  );
    }
    /*void*/System.exit( ExitCode );
  }


  static private
  void
  exit  (int            ExitCode        ,
         String         strUsage        ,
         String         strMsg          )
  {
    if (strMsg != null) {
      /*void*/System.err.println( strMsg );
    }
    if (strUsage != null) {
      /*void*/System.err.println( strUsage );
    }
    exit( ExitCode , null );
  }


  static public
  void
  main  (String	argv[]	)
  {
    ICalendar   icalendar       = null;
    ProdId      prodid          = new ProdId(  );
    Version     version         = new Version(  );
    EventsI	eventsi;
    User	user		= null;
    ListIterator		li;
    byte[]      abyte           = new byte[100000];
    int         cbyte           = 0;
    int         status;
    String      strInput;
    int         ExitCode        = 1;
  /* Command Line Related */
    int         iarg            = 0;
    String      strProg         = "UWICal";
    String      strUsage        = "usage:\n" +
	"\t" + strProg + " [-v] [-vv] [-b] [-m] [-n] user name start end\n" +
	"\t" + strProg + " [-v] [-vv] [-b] [-m] [-n] -s user name\n";
    boolean     bVerbose        = false;
    boolean     bVVerbose       = false;
    boolean     bBrief          = false;
    boolean     bMime           = false;
    boolean     bNoExec         = false;
    boolean     bStore          = false;
    String	strUser		= null;
    String	strName		= null;
    String      strBeg          = null;
    String      strEnd          = null;

  /* Initialization */
    try {
      /*void*/prodid.set( "C&C/UWICal" );
      /*void*/version.set( "2.0" );
    } catch (Exception  e       ) {
      /*void*/exit( ExitCode , e );
    }
    ExitCode++;

  /* Command line processing */
    for ( ;
         iarg < argv.length &&
           argv[iarg].startsWith( mc_strSwitch ) ;
         iarg++) {
      if        (argv[iarg].equals( mc_strSwitch + "v" )) {
        bVerbose = true;
      } else if (argv[iarg].equals( mc_strSwitch + "vv" )) {
        bVVerbose = true;
      } else if (argv[iarg].equals( mc_strSwitch + "b" )) {
        bBrief = true;
      } else if (argv[iarg].equals( mc_strSwitch + "m" )) {
        bMime = true;
      } else if (argv[iarg].equals( mc_strSwitch + "n" )) {
        bNoExec = true;
      } else if (argv[iarg].equals( mc_strSwitch + "s" )) {
        bStore = true;
      } else {
        /*void*/exit( ExitCode , strUsage , "unknown option" );
      }
    }

    try {
      strUser = argv[iarg++];
      strName = argv[iarg++];
      if (!bStore) {
        strBeg  = argv[iarg++];
        strEnd  = argv[iarg++];
      }
    } catch (ArrayIndexOutOfBoundsException     e       ) {
      /*void*/exit( ExitCode , strUsage , "missing arguments" );
    }
    ExitCode++;
    if (iarg != argv.length) {
      /*void*/exit( ExitCode , strUsage , "extraneous arguments" );
    }
    ExitCode++;

    if (bVVerbose) {
      bVerbose = true;
    }

    if (bVerbose) {
      /*void*/System.out.println( "Program:	" + strProg );
      /*void*/System.out.println( "	Verbose?:	" + bVerbose );
      /*void*/System.out.println( "	VVerbose?:	" + bVVerbose );
      /*void*/System.out.println( "	Mime?:	" + bMime );
      /*void*/System.out.println( "	NoExec?:	" + bNoExec );
      /*void*/System.out.println( "	Store?:	" + bStore );
      /*void*/System.out.println( "	User:	" + strUser );
      /*void*/System.out.println( "	Name:	" + strName );
      if (!bStore) {
        /*void*/System.out.println( "	Begin:	" + strBeg );
        /*void*/System.out.println( "	End:	" + strEnd );
      }
    }

    user = new User( strUser );

    if (!bStore) {
    /* build calendar by uwcal eventsi */
      try {
	if        (!user.isGuest(  )) {
	  eventsi = new PersonalEvents(user, true);
	} else if (!strName.equals( "" )) {
	  eventsi = new FilteredEvents(
			PublicCalendars.getPublicCalendars(  ).
			getCalendar( strName ).getFilter(  ) );
	} else {
	  eventsi = PublicEvents.getPublicEvents(  );
	}
	li = eventsi.manyDaysEvents( new MyCalendar( strBeg ) ,
				     new MyCalendar( strEnd ) ).
			listIterator(  );
        if (bVVerbose) {
          /*void*/System.out.println( "Dump:" );
          /*void*/System.out.println(
                        "======================================" +
                        "=====================================" );
          /*void*/System.out.println(
			new UCEventsDumper( eventsi , "  " ).
			toString( bBrief ) );
          /*void*/System.out.println(
                        "======================================" +
                        "=====================================" );
        }

        icalendar = new UWICal( eventsi );
        /*void*/icalendar.addProperty( prodid );
        /*void*/icalendar.addProperty( version );
      } catch (Exception        e       ) {
        /*void*/exit( ExitCode , e );
      }
    } else {
    /* build calendar by icalendar stream */
      try {
        for (cbyte = 0 ;
             -1 != (status = System.in.read(
                        abyte , cbyte , abyte.length - cbyte )) ;
             cbyte += status) {
          ;     // null statement
        }
      } catch (java.io.IOException      e       ) {
        /*void*/exit( ExitCode , e );
      }

      strInput = new String( abyte , 0 , cbyte );
      if (bVVerbose) {
        /*void*/System.out.println( "bytes read: " + cbyte );
        /*void*/System.out.println( "======================================" +
                                    "=====================================" );
        System.out.print( strInput );
        /*void*/System.out.println( "======================================" +
                                    "=====================================" );
      }

      if (bMime) {
        strInput = Utilities.decode( strInput );
      } else {
        strInput = Utilities.unfoldLines( strInput );
      }

      try {
        icalendar = new UWICal( strInput );
      } catch (Exception        e       ) {
        /*void*/exit( ExitCode , e );
      }
    }
    ExitCode++;

    if (bVVerbose) {
      /*void*/System.out.println( "complete?: " + icalendar.isComplete(  ) );
      /*void*/System.out.println( "======================================" +
                                  "=====================================" );
    }
    if (!bStore || bVVerbose) {
      if (bMime) {
        /*void*/System.out.print( Utilities.encode( icalendar ) );
      } else {
        /*void*/System.out.print( Utilities.foldLines( icalendar ) );
      }
    }
    if (bVVerbose) {
      /*void*/System.out.println( "======================================" +
                                  "=====================================" );
    }

  /* extract calendar as uwcal eventsi */
    if (bStore || bVVerbose) {
      try {
	eventsi = ((UWICal)icalendar).convertToUWCalEvents(
			user , bStore && !bNoExec );

        if (bVVerbose) {
          /*void*/System.out.println( "Dump:" );
          /*void*/System.out.println(
                        "======================================" +
                        "=====================================" );
          /*void*/System.out.println(
			new UCEventsDumper( eventsi , "  " ).
			toString( bBrief ) );
          /*void*/System.out.println(
                        "======================================" +
                        "=====================================" );
        }
      } catch (Exception        e       ) {
        /*void*/exit( ExitCode , e );
      }
    }
    ExitCode++;
  }

}


/* Note: update class header too */
/* Log:
 *  0.10  2003/10/13, 2003/11/24, 2003/11/25  slh
 *        update for change in DTStamp
 *        update for change in Duration/DurationProp
 *  ...
 *  0.08  2003/06/10, 2003/06/23 - 2003/06/24, 2003/07/01 ... 2003/07/29  slh
 *        update for separate personal and public events objects
 *        use escaping/unescaping option of Text/TextListProp object
 *        use quoting/unquoting option of Text/TextListParam object
 *        move dumping to instrumentation/
 *        re-add 0.06 changes
 *  ...
 *  0.06  2003/04/30  slh
 *        re-add: UWICal(ICalendar), UWICal(ProdId,Version,VEvent)
 *        rm extraneous imports
 *  ...
 *  0.03  2002/03/04, 2002/03/06, 2002/03/08  slh
 *        adjusted for new package locations
 *        replace most fully qualified uwcal paths with imports
 *        endstartdurationToEnd(): interface ch: return DTEnd
 *        change some methods access, move some code
 *        comment doc changes
 *  0.02  2002/02/27  slh
 *        add -n
 *        give error for invalid options; add exit()
 *        add constructors (ProdId,Version,Events), (String,String,Events)
 *        redo parsing for reading back locations
 *  0.01  2002/02/26  slh
 *        redid main() including adding/supporting -vv -m -s
 *        rm some test/debug output
 *  0.00  2002/02/12 - 2002/02/14  slh
 *        create from icalendar/ICalendar.java
 */
/*--UWICal.java-------------------------------------------------------------*/
