/*++TimeProp.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.Date;
import  java.util.SimpleTimeZone;
import  java.text.SimpleDateFormat;
import  java.text.ParseException;


/**
 * Base class for iCalendar DATE- and DATE-TIME-valued properties.
 *
 *<PRE>	<!-- for alignment purposes -->&nbsp;
 * Spec:
 *   section:	4.3.4 - 4.3.5 [2445]
 *   allowed:	---
 *</PRE>
 *
 * @author  slh
 * @version  0.20 2003/11/03 slh
 */
abstract public
class  TimeProp
extends  Property
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /*Note: this is the base constructor*/
  /**
   * Create a time property, unset.
   *
   * Since DefaultType is unchecked,
   * this interface should not be passed on in subclasses.
   */
  protected
  TimeProp  (String	strName		,
	     int	DefaultType	,
	     boolean	bAlwaysUTC	)
  {
    super( strName );

    switch (DefaultType) {
    case DATETIME:
    case DATE:
      m_DefaultType = DefaultType;
      break;
    case NONE:
    default:
      throw new IllegalArgumentException( getName(  ) + ".TimeProp()" +
		": illegal/invalid type: " + DefaultType );
    }
    m_bAlwaysUTC = bAlwaysUTC;

    /*void*/setType( m_DefaultType );
  }


  /**
   * Create a time property set to supplied value.
   *
   * Since DefaultType is unchecked,
   * this interface should not be passed on in subclasses.
   */
  protected
  TimeProp  (String	strName		,
	     String	strValue	,
	     int	DefaultType	,
	     boolean	bAlwaysUTC	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName , DefaultType , bAlwaysUTC );

    try {
      /*void*/set( strValue );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( getName(  ) + ".TimeProp()" +
		": " + e.getLocalizedMessage(  ) );
    }
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/

  public
  boolean
  isComplete  (		)
  {
    ValueParam	vp;

    vp = (ValueParam)m_parameters.getFirst( ValueParam.class );
    if (vp == null && m_Type == m_DefaultType) {
      ;		// ok
    } else if (vp == null || m_Type != vp.getEnum(  )) {
      return false;						/*RETURN*/
    }

    return m_date != null && super.isComplete(  );
  }


  /**
   * Get date/time as a Date.
   */
  public
  Date
  get  (		)
  {
    return m_date;
  }


  /**
   * Get date/time value type.
   */
  public
  int
  getType  (		)
  {
    return m_Type;
  }


  public
  boolean
  isUTC  (		)
  {
    return m_bUTC;
  }


  /**
   * Set date/time value type.
   */
  public
  void
  setType  (int		Type	)
  throws IllegalArgumentException
  {
    Collectible		coll;

    switch (Type) {
    case DATETIME:
    case DATE:
      if (m_Type != Type) {
	if (null !=
	    (coll = m_parameters.getFirst( ValueParam.class ))) {
	  /* let throw exception if does,
	     though there should never be a problem */
	  /*void*/m_parameters.remove( coll );
	}
	if (Type != m_DefaultType) {
	  /*void*/addAnyParameter( new ValueParam( Type ) );
	}
      }
      m_Type = Type;
      break;
    case NONE:
    default:
      throw new IllegalArgumentException( getName(  ) + ".setType()" +
		": illegal/invalid type: " + Type );
    }
  }


  /**
   * Set date/time with a date and
   * indicate date/time value type.
   */
  public
  void
  set  (Date	date	,
	int	Type	)
  throws IllegalArgumentException
  {
    /*void*/set( date , Type , m_bAlwaysUTC );
  }


  /**
   * Set date/time with a date and
   * indicate date/time value type.
   */
  public
  void
  set  (Date	date	,
	int	Type	,
	boolean	bUTC	)
  throws IllegalArgumentException
  {
    if (!bUTC && m_bAlwaysUTC) {
      throw new IllegalArgumentException( getName(  ) + ".set()" +
		": value must be a UTC time" );
    }

    /*void*/setType( Type );
    m_bUTC = bUTC;
    m_date = date;
  }


  /**
   * Set date/time and value type with a string.
   */
  public
  void
  set  (String	strValue	)
  throws Exception
  {
    SimpleDateFormat	sdf		= null;
    boolean		fNegative	= false;

    if (!strValue.endsWith( "Z" ) && m_bAlwaysUTC) {
      throw new IllegalArgumentException( getName(  ) + ".set()" +
		": value must be a UTC time: " + strValue );
    }

    m_bUTC = false;
    if        (strValue == null) {
      m_date = null;
      return;							/*RETURN*/
    } else if (strValue.endsWith( "Z" )) {
      /*void*/setType( DATETIME );
      m_bUTC = true;
      sdf = new SimpleDateFormat( mc_strFmtTimeUTC );
    } else if (strValue.indexOf( "T" ) != -1) {
      /*void*/setType( DATETIME );
      sdf = new SimpleDateFormat( mc_strFmtTime );
    } else if (strValue.length(  ) == mc_strFmtDate.length(  )) {
      /*void*/setType( DATE );
      sdf = new SimpleDateFormat( mc_strFmtDate );
    } else {
      /*void*/setType( NONE );
      throw new ParseException( getName(  ) + ".set()" +
		": unrecognized time format: " + strValue , 0 );
    }

    if (m_bUTC) {
      /*void*/sdf.setTimeZone( new SimpleTimeZone( 0 , "Z" ) );
    }

    m_date = sdf.parse( strValue );
    if (m_date == null) {
      throw new ParseException( getName(  ) + ".set()" +
		": unparsable time: " + strValue , 0 );
    }
								/*RETURN*/
  }


  public
  int
  getDefaultType  (		)
  {
    return m_DefaultType;
  }


  public
  boolean
  isAlwaysUTC  (		)
  {
    return m_bAlwaysUTC;
  }


  /**
   * Get date/time as a string.
   */
  public
  String
  getString  (		)
  {
    SimpleDateFormat	sdf;

    switch (m_Type) {
    case DATETIME:
      if (m_bUTC) {
	sdf = new SimpleDateFormat( mc_strFmtTimeUTC );
      } else {
	sdf = new SimpleDateFormat( mc_strFmtTime );
      }
      break;
    case DATE:
      sdf = new SimpleDateFormat( mc_strFmtDate );
      break;
    case NONE:
    default:
      return null;						/*RETURN*/
    }

    if (m_bUTC) {
      /*void*/sdf.setTimeZone( new SimpleTimeZone( 0 , "Z" ) );
    }

    return m_date != null ? sdf.format( m_date ) : null;	/*RETURN*/
  }


/*----------------------------------------------------------------------------
 *						Class Public Constants
 *--------------------------------------------------------------------------*/

  final static public  int	NONE		= ValueParam.NONE;
  final static public  int	DATETIME	= ValueParam.DATETIME;
  final static public  int	DATE		= ValueParam.DATE;


/*----------------------------------------------------------------------------
 *						Class Private Constants
 *--------------------------------------------------------------------------*/

  static private final String	mc_strFmtTime		= "yyyyMMdd'T'HHmmss";
  static private final String	mc_strFmtTimeUTC	=
					mc_strFmtTime + "'Z'";
  static private final String	mc_strFmtDate		= "yyyyMMdd";


/*----------------------------------------------------------------------------
 *						Object Protected Variables
 *--------------------------------------------------------------------------*/

  protected  Date		m_date		= null;
  protected  int		m_Type		= NONE;
  protected  boolean		m_bUTC		= false;
  /*Note: these should only be set in a constructor and
    then early on before any set*()s;
    otherwise the ValueParam may be not added (correctly)
  */
  protected  int		m_DefaultType	= DATETIME;
  protected  boolean		m_bAlwaysUTC	= false;


/*----------------------------------------------------------------------------
 *						Main
 *--------------------------------------------------------------------------*/

  static public
  void
  main  (String	argv[]	)
  {
    Created		created		= new Created(  );
    DTStart		dtstart		= new DTStart(  );
    LastModified	lastmodified	= new LastModified(  );
    DTStamp		dtstamp		= new DTStamp(  );
    DTEnd		dtend		= new DTEnd(  );
    RecurrenceId	recurrenceid	= new RecurrenceId(  );
    Completed		completed	= new Completed(  );
    Due			due		= new Due(  );

    /*void*/created.set( new java.util.Date(  ) , TimeProp.DATETIME );
    /*void*/System.out.println( created.isComplete(  ) );
    /*void*/System.out.println( created );
    /*void*/dtstart.set( new java.util.Date(  ) , TimeProp.DATETIME );
    /*void*/System.out.println( dtstart.isComplete(  ) );
    /*void*/System.out.println( dtstart );
    /*void*/lastmodified.set( new java.util.Date(  ) , TimeProp.DATETIME );
    /*void*/System.out.println( lastmodified.isComplete(  ) );
    /*void*/System.out.println( lastmodified );
    /*void*/dtstamp.set( new java.util.Date(  ) , TimeProp.DATETIME );
    /*void*/System.out.println( dtstamp );
    /*void*/System.out.println( dtend.getDefaultType(  ) );
    /*void*/System.out.println( new ValueParam( dtend.getDefaultType(  ) ) );
    /*void*/dtend.set( new java.util.Date(  ) , ValueParam.DATE );
    /*void*/System.out.println( dtend.isComplete(  ) );
    /*void*/System.out.println( dtend );
    try {
      /*void*/System.out.println(
		"Note: should throw IllegalArgumentException:" );
      /*void*/recurrenceid.set( new java.util.Date(  ) , ValueParam.BINARY );
      /*void*/System.out.println( recurrenceid.isComplete(  ) );
      /*void*/System.out.println( recurrenceid );
    } catch (Exception	e	) {
      /*void*/System.out.println( "\t" + e );
    }
    try {
      /*void*/System.out.println(
		"Note: should throw IllegalArgumentException:" );
      /*void*/completed.set( "20021210T235959" );
      /*void*/System.out.println( completed.isComplete(  ) );
      /*void*/System.out.println( completed );
    } catch (Exception	e	) {
      /*void*/System.out.println( "\t" + e );
    }
    try {
      /*void*/due.set( "20021210T235959Z" );
      /*void*/System.out.println( due.isComplete(  ) );
      /*void*/System.out.println( due );
    } catch (Exception	e	) {
      /*void*/e.printStackTrace(  );
    }
    try {
      /*void*/System.out.println( "Note: isComplete() should be false:" );
      /*void*/due.addParameter( new ValueParam( ValueParam.DATE ) );
      /*void*/System.out.println( due.isComplete(  ) );
      /*void*/System.out.println( due );
    } catch (Exception	e	) {
      /*void*/e.printStackTrace(  );
    }
  }

}


/* Log:
 *  0.20  2003/10/20, 2003/10/29, 2003/11/03  slh
 *        redo constructors
 *        adjust for ValueTypeParam -> ValueParam
 *        use *.class instead of Names.*
 *        rm set(Date...) methods w/o a type param
 *        misc
 *  0.11  2003/06/23, 2003/07/09  slh
 *        isUTC(), isAlwaysUTC()
 *        fixed not-quite-right utc stuff
 *        setType(): bugfix: was allowing multiple VALUE params
 *  0.10  2002/12/10  slh
 *        redo constructors
 *        drop toString() (let super do with getString())
 *        isComplete()
 *        set(String): handle/allow null
 *        shuffle
 *  0.02  2002/03/05 - 2002/03/06  slh
 *        adjusted for new package location
 *        add default type support
 *        mc_strFmt*: make private as was intended
 *        more javadoc comments
 *  0.01  2001/08/14, 2001/08/20  slh
 *        set(String)
 *        move duration to own property type
 *  0.00  2001/05/08  slh
 *        create
 */
/*--TimeProp.java-----------------------------------------------------------*/
