/*++TextProp.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.StringTokenizer;


/**
 * Base class for iCalendar TEXT-valued properties
 * (and base class for some other base classes).
 *
 *<PRE>	<!-- for alignment purposes -->&nbsp;
 * Spec:
 *   section:	4.3.11 [2445]
 *   allowed:	---
 *</PRE>
 *
 * Methods affected by Config ProcessText flag:
 *   TextProp(String,String)
 *   get()
 *   set(String)
 * Methods not affected by Config ProcessText flag include:
 *   getString()
 *
 * @author  slh
 * @version  0.20 2003/07/30 slh
 */
abstract public
class  TextProp
extends  Property
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /* this is the base constructor */
  /**
   * Create an TEXT property, unset.
   */
  protected
  TextProp  (String	strName	)
  {
    super( strName );
  }


  /**
   * Create a TEXT property set to supplied value.
   */
  protected
  TextProp  (String	strName		,
	     String	strValue	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    try {
      /*void*/set( strValue );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( getName(  ) +
		": " + e.getLocalizedMessage(  ) );
    }
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Primary Methods		*/

  public
  boolean
  isComplete  (		)
  {
    return m_strValue != null && super.isComplete(  );
  }


  /**
   * @param  bUnescape	remove any quoting
   * @return  value as a String
   */
  /* uses getString(), in support of possible subclasses
   */
  public
  String
  get  (boolean	bUnescape	)
  {
    if (!bUnescape) {
      return getString(  );					/*RETURN*/
    } else {
      return checkUnescapeText( getString(  ) );		/*RETURN*/
    }
  }


  public
  String
  get  (		)
  {
    return get( Config.mc_bProcessTexts );
  }


  /**
   * Set value.
   *
   * @param  strValue	value to which to set
   * @param  bEscape	escape invalid text, if possible
   */
  public
  void
  set  (String	strValue	,
	boolean	bEscape		)
  throws Exception	/* to support subclasses */
  {
    if (!bEscape) {
      m_strValue = checkText( strValue );
    } else {
      m_strValue = checkEscapeText( strValue );
    }
  }


  public
  void
  set  (String	strValue	)
  throws Exception	/* to support subclasses */
  {
    /*void*/set( strValue , Config.mc_bProcessTexts );
  }


  public
  String
  getString  (		)
  {
    return m_strValue;
  }


/*----------------------------------------------------------------------------
 *						Class Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Utility/Support Methods	*/

  /**
   * Check if text is valid.
   *
   * @param  strText	text to be examined
   * @return  strText
   */
  static public
  String
  checkText  (String	strText	)
  {
    return checkEscapeText( strText , false );
  }


  /**
   * Check if text needs escaping and do so if so.
   *
   * @param  strText	text to be examined/escaped
   * @return  escaped text if needs escaping, otherwise original String
   */
  static public
  String
  checkEscapeText  (String	strText	)
  {
    return checkEscapeText( strText , true );
  }


  /**
   * Check if text needs escaping and do so if so.
   *
   * @param  strText	text to be examined/escaped
   * @param  bEscape	escape text if needed (otherwise throw exception)
   * @return  escaped text if needs escaping, otherwise original String
   */
  static public
  String
  checkEscapeText  (String	strText	,
		    boolean	bEscape	)
  {
    StringTokenizer	st;
    String		strToken;
    String		strOut		= "";

    if (strText == null) {
      return strText;						/*RETURN*/
    }

    st = new StringTokenizer( strText , Strings.strTextComplement , true );
    for ( ; st.hasMoreTokens(  ) ; ) {
      strToken = st.nextToken(  );
      /* if char to be escaped ... */
      if        (!strToken.equals( escapeValueToSequence( strToken ) )) {
	if (bEscape) {
	  strOut += escapeValueToSequence( strToken );
	}
      /* ... else if disallowed inescapable text ... */
      } else if (-1 != Strings.strTextComplement.indexOf( strToken )) {
	throw new IllegalArgumentException( "TextProp.checkEscapeText()" +
			": invalid text: " + strText );
      /* ... else allowed text ... */
      } else {
	if (bEscape) {
	  strOut += strToken;
	}
      }
    }

    return !bEscape || strText.length(  ) == strOut.length(  )
		? strText : strOut;				/*RETURN*/
  }


  /**
   * Check if value is escaped and remove escapes if so.
   *
   * @param  strText	text to be examined/unescaped
   * @return  unescaped value if needs unescaping, otherwise original String
   */
  static public
  String
  checkUnescapeText  (String	strText	)
  {
    StringTokenizer	st;
    String		strToken;
    String		strOut		= "";

    if (-1 == strText.indexOf( Strings.strPropEscape )) {
      return strText;						/*RETUIRN*/
    } else {
      st = new StringTokenizer( strText , Strings.strTextComplement , true );
      for ( ; st.hasMoreTokens(  ) ; ) {
	strToken = st.nextToken(  );
	/* if not escape ... */
	if (!strToken.equals( Strings.strPropEscape )) {
	  strOut += strToken;
	/* ... escape ... */
	} else {
	  if (!st.hasMoreTokens(  )) {
	    throw new IllegalArgumentException(
			"TextProp.checkUnescapeText()" +
			": expected code after escape char" );
	  }
	  strToken = st.nextToken( Strings.strEscapeCodeComplement );
	  strOut += escapeCodeToValue( strToken );
	}
      }

      return strOut;						/*RETURN*/
    }
  }


  static public
  String
  escapeCodeToValue  (String	strCode	)
  {
    int		idx;

    for (idx = 0 ;
	 idx < Strings.astrPropEscapeCode.length &&
	   idx < Strings.astrPropEscapeValue.length ;
	 idx++) {
      if (Strings.astrPropEscapeCode[idx].equals( strCode )) {
	return Strings.astrPropEscapeValue[idx];		/*RETURN*/
      }
    }

    return strCode;						/*RETURN*/
  }


  /*Note: will return input if not escapable (or they are the same...)*/
  static public
  String
  escapeValueToSequence  (String	strValue	)
  {
    int		idx;

    for (idx = 0 ;
	 idx < Strings.astrPropEscapeValue.length &&
	   idx < Strings.astrPropEscapeCode.length ;
	 idx++) {
      if (Strings.astrPropEscapeValue[idx].equals( strValue )) {
	return Strings.strPropEscape + Strings.astrPropEscapeCode[idx];
								/*RETURN*/
      }
    }

    return strValue;						/*RETURN*/
  }


/*----------------------------------------------------------------------------
 *						Object Protected Variables
 *--------------------------------------------------------------------------*/

  protected  String	m_strValue	= null;

}


/* Log:
 *  0.20  2003/07/14 - 2003/07/15, 2003/07/30  slh
 *        get(): option to unescape
 *        set(): option to escape if nec
 *        check{|Unescape|Escape}Text()
 *  0.10  2002/12/10  slh
 *        redo constructors
 *        isComplete()
 *        set(): allow null
 *        drop toString() (let super do with getString())
 *  0.00  2001/04/24 - 2001/04/25, 2001/05/08, 2001/05/15 - 2001/05/17  slh
 *        create
 */
/*--TextProp.java-----------------------------------------------------------*/
