/*++TextParam.java++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.StringTokenizer;
import  java.util.NoSuchElementException;


/**
 * Base class for iCalendar TEXT-valued parameters
 * (and base class for some other base classes).
 *
 *<PRE>	<!-- for alignment purposes -->&nbsp;
 * Spec:
 *   section:	4.1 [2445]
 *   allowed:	---
 *</PRE>
 *
 * Methods affected by Config ProcessText flag:
 *   TextParam(String,String)
 *   get()
 *   set(String)
 * Methods not affected by Config ProcessText flag include:
 *   getString()
 *
 * @author  slh
 * @version  0.20 2003/07/30 slh
 */
abstract public
class  TextParam
extends  Parameter
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /* this is the base constructor */
  /**
   * Create an TEXT parameter, unset.
   */
  protected
  TextParam  (String	strName	)
  {
    super( strName );
  }


  /**
   * Create a TEXT parameter set to supplied value.
   */
  protected
  TextParam  (String	strName		,
	      String	strValue	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    try {
      /*void*/set( strValue );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( getName(  ) +
		": " + e.getLocalizedMessage(  ) );
    }
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Primary Methods		*/

  public
  boolean
  isComplete  (		)
  {
    return m_strValue != null;
  }


  /**
   * @param  bUnquote	remove any quoting
   * @return  value as a String
   */
  /* uses getString(), in support of possible subclasses
   */
  public
  String
  get  (boolean	bUnquote	)
  {
    if (!bUnquote) {
      return getString(  );					/*RETURN*/
    } else {
      return checkUnquoteText( getString(  ) );		/*RETURN*/
    }
  }


  public
  String
  get  (		)
  {
    return get( Config.mc_bProcessTexts );
  }


  /**
   * Set value.
   *
   * @param  strValue	value to which to set
   * @param  bQuote	quote invalid text, if possible
   */
  public
  void
  set  (String	strValue	,
	boolean	bQuote		)
  throws Exception	/* to support subclasses */
  {
    if (!bQuote) {
      m_strValue = checkText( strValue );
    } else {
      m_strValue = checkQuoteText( strValue , false );
    }
  }


  public
  void
  set  (String	strValue	)
  throws Exception	/* to support subclasses */
  {
    /*void*/set( strValue , Config.mc_bProcessTexts );
  }


  public
  String
  getString  (		)
  {
    return m_strValue;
  }


/*----------------------------------------------------------------------------
 *						Class Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- Utility/Support Methods	*/

  /**
   * Check that text is valid.
   */
  static public
  String
  checkText  (String	strText	)
  throws Exception
  {
    StringTokenizer	st;

    if (strText == null) {
      return strText;						/*RETURN*/
    }

    st = new StringTokenizer( strText , Strings.strSafeComplement , true );
    /* if quoted ... */
    if (st.nextToken(  ).equals( Strings.strParamValueQuote )) {
      try {
	/* actual value */
	/*(void)*/st.nextToken( Strings.strQSafeComplement );
	/* closing quote */
	if (!st.nextToken(  ).equals( Strings.strParamValueQuote )) {
	  throw new IllegalArgumentException( "TextParam.checkText()" +
			": expected: '" + Strings.chParamValueQuote + "'" );
	}
      } catch (NoSuchElementException	e	) {
	throw new IllegalArgumentException( "TextParam.checkText()" +
			": invalid text: " + strText );
      }
    }
    /* ... else not quoted (and found value) ... */
    /* and nothing else */
    if (st.hasMoreElements(  )) {
      throw new IllegalArgumentException( "TextParam.checkText()" +
		": invalid text: " + strText );
    }

    return strText;						/*RETURN*/
  }


  /**
   * Check if value needs quoting and do so if so.
   *
   * @param  strText	text to be examined/quoted
   * @param  bForce	force quoting?
   * @return  quoted value if needs quoting, otherwise original String
   */
  static public
  String
  checkQuoteText  (String	strText	,
		   boolean	bForce	)
  {
    StringTokenizer	st;

    if (strText != null &&
	!strText.startsWith( Strings.strParamValueQuote )) {
      st = new StringTokenizer( strText , Strings.strSafeComplement , true );
      if (1 != st.countTokens(  ) || bForce) {
	st = new StringTokenizer(
			strText , Strings.strQSafeComplement , true );
	if (1 == st.countTokens(  )) {
	  return Strings.strParamValueQuote + strText +
		Strings.strParamValueQuote;			/*RETURN*/
	} else {
	  throw new IllegalArgumentException( "TextParam.checkQuoteText()" +
			": contains illegal character(s): " + strText );
	}
      }
    }

    return strText;						/*RETURN*/
  }


  /**
   * Check if value is quoted and remove quotes if so.
   *
   * @param  strText	text to be examined/unquoted
   * @return  unquoted value if needs unquoting, otherwise original String
   */
  static public
  String
  checkUnquoteText  (String	strText	)
  {
//???guess should do something if one and not the other or invalid text
    if (strText != null &&
	strText.startsWith( Strings.strParamValueQuote ) &&
	strText.endsWith( Strings.strParamValueQuote )) {
      return strText.substring( 1 , strText.length(  ) - 1 );	/*RETURN*/
    }

    return strText;						/*RETURN*/
  }


/*----------------------------------------------------------------------------
 *						Object Protected Variables
 *--------------------------------------------------------------------------*/

  protected  String	m_strValue	= null;

}


/* Log:
 *  0.20  2003/07/14 - 2003/07/15, 2003/07/30  slh
 *        get(): option to unquote
 *        set(): option to quote if nec
 *        check{|Unquote|Quote}Text()
 *  0.10  2002/12/10  slh
 *        redo constructors
 *        isComplete()
 *        set(): allow null
 *        drop toString() (let super do with getString())
 *  0.00  2001/05/16 - 2001/05/17  slh
 *        create
 */
/*--TextParam.java----------------------------------------------------------*/
