/*++TextListProp.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.Iterator;
import  java.util.StringTokenizer;
import  java.util.Vector;


/**
 * Base class for iCalendar TEXT-valued properties
 * that allow a delimiter-separated list of texts.
 *
 * Methods affected by Config ProcessText flag:
 *   TextListProp(String,String)
 *   getFirstText()
 *   set(String)
 *   addText(String)
 *   addTexts(String)
 * Methods not affected by Config ProcessText flag include:
 *   getList()
 *   getFirst()
 *   set(Iterator)
 *   add(Object)
 *   getString()
 *
 * Subclassing:
 * override AddDirect() and AddTextDirect()
 * to make a class with different constraints on individual members of list.
 *
 * @author  slh
 * @version  0.10 2003/07/30 slh
 */
abstract public
class  TextListProp
extends  TextProp
{

/*----------------------------------------------------------------------------
 *                                              Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/

  /**
   * Create a TextList property unset.
   */
  /* Note: this is the base constructor */
  protected
  TextListProp  (String strName )
  {
    super( strName );

    m_vector = new Vector(  );
  }


  /**
   * Create a TextList property set to supplied value.
   */
  /* Convenience */
  protected
  TextListProp  (String	strName		,
		 String	strValues	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    try {
      /*void*/set( strValues );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( e.getLocalizedMessage(  ) );
    }
  }


  /**
   * Create a TextList property unset and
   * set minimum and maximum number of elements.
   */
  /* Convenience */
  protected
  TextListProp  (String strName ,
                 int    Min     ,
                 int    Max     )
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    /*void*/setMinMax( Min , Max );
  }


  /**
   * Create a TextList property unset and
   * set the delimiter and the minimum and maximum number of elements.
   */
  /* Convenience */
  protected
  TextListProp  (String	strName		,
		 String	strDelim	,
		 int	Min		,
		 int	Max		)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    /*void*/setDelim( strDelim );
    /*void*/setMinMax( Min , Max );
  }


/*----------------------------------------------------------------------------
 *                                              Object Public Methods
 *--------------------------------------------------------------------------*/

  public
  boolean
  isComplete  (         )
  {
    /*Note: second clause should always be true*/
    return m_Min <= m_vector.size(  ) &&
                (m_Max == UNBOUNDED || m_vector.size(  ) <= m_Max);
  }


  /**
   * @return  count of values in list
   */
  public
  int
  getCount  (           )
  {
    return m_vector.size(  );
  }


  /***
   * Get Iterator of correct object class for class (String here).
   * @return  iterator of correct object class for class (String here)
   */
  public
  Iterator
  getList  (            )
  {
    return m_vector.iterator(  );
  }


  /**
   * @return  first value as object (which is a String here)
   */
  /* Convenience */
  public
  Object
  getFirst  (           )
  throws NoValueException
  {
    Iterator            enum;

    enum = m_vector.iterator(  );

    if (enum.hasNext(  )) {
      return enum.next(  );
    } else {
      throw new NoValueException( getName(  ) + ": no values" );
    }
  }


  /**
   * @return  first value as a String
   */
  /* Convenience */
  public
  String
  getFirstText  (boolean	bUnescape	)
  throws NoValueException
  {
    /* probably work for all classes */
    String	strText;

    strText = getFirst(  ).toString(  );

    if (bUnescape) {
      strText = checkUnescapeText( strText );
    }

    return strText;
  }


  /**
   * @return  first value as a String
   */
  /* Convenience */
  public
  String
  getFirstText  (		)
  throws NoValueException
  {
    return getFirstText( Config.mc_bProcessTexts );
  }


  /**
   * @param  strValues	comma-separated list of values
   * @param  bEscape	escape invalid text, if possible
   */
  public
  void
  set  (String  strValues       ,
	boolean	bEscape		)
  throws Exception
  {
    /*Note: letting calls take care of null cases*/

    /*void*/m_vector.removeAllElements(  );

    /*void*/addTexts( strValues , bEscape );
  }


  /**
   * @param  strValues	comma-separated list of values
   */
  public
  void
  set  (String  strValues       )
  throws Exception
  {
    /*void*/set( strValues , Config.mc_bProcessTexts );
  }


  /**
    @param enum     Iterator of correct object class for class (String here).
   */
  /* convenience */
  public void set(Iterator enum) throws Exception
  {
    /*Note: letting calls take care of null cases*/

    m_vector.removeAllElements(  );

    for (; enum.hasNext(); ) {
      add(enum.next());
    }
  }


  /**
   * @param  object	object of correct class for class (String here)
   */
  public
  void
  add  (Object  object  )
  throws Exception      /* TooManyValuesException, NullPointerException */
  {
    if (object == null) {
      throw new NullPointerException(
                getName(  ) +
                ": object (Object) is null" );
    }

    if (m_Max == UNBOUNDED || m_vector.size(  ) < m_Max) {
      /*Note: addDirect() appropriate for current subclass is called*/
      /*void*/addDirect( object );
    } else {
      throw new TooManyValuesException(
                getName(  ) +
                ": attempt to add more than maximum allowed values" );
    }
  }


  public
  void
  addText  (String	strText	,
	    boolean	bEscape	)
  throws Exception      /* TooManyValuesException, NullPointerException */
  {
    if (strText == null) {
      throw new NullPointerException(
                getName(  ) +
                ": strText (String) is null" );
    }

    if (m_Max == UNBOUNDED || m_vector.size(  ) < m_Max) {
      /*Note: addTextDirect() appropriate for current subclass is called*/
      if (bEscape) {
	strText = checkEscapeText( strText , false );
      }
      /*void*/addTextDirect( strText );
    } else {
      throw new TooManyValuesException(
                getName(  ) +
                ": attempt to add more than maximum allowed values" );
    }
  }


  public
  void
  addText  (String      strText )
  throws Exception      /* TooManyValuesException, NullPointerException */
  {
    /*void*/addText( strText , Config.mc_bProcessTexts );
  }


  /**
   * @param  strTexts	comma-separated list of values
   * @param  bEscape	escape invalid text, if possible
   */
  /* Convenience+ */
  public
  void
  addTexts  (String	strTexts	,
	     boolean	bEscape		)
  throws Exception      /* TooManyValuesException */
  {
    StringTokenizer	st;
    String		strToken;
    String		strText		= "";
    boolean		bInEscape	= false;

    if (strTexts == null) {
      throw new NullPointerException( getName(  ) +
		": strTexts (String) is null" );
    }

    st = new StringTokenizer(
		strTexts , m_strDelim + Strings.strPropEscape , true );
    for ( ; st.hasMoreTokens(  ) ; ) {
      strToken = st.nextToken(  );
      if (strToken.equals( m_strDelim ) && !bInEscape) {
	/*void*/addText( strText , bEscape );
	strText = "";
      } else {
	strText += strToken;
	if (bInEscape) {
	  bInEscape = false;
	/* ... else at escape char (and not already in an escape) ... */
	} else if (strToken.equals( Strings.strPropEscape )) {
	  bInEscape = true;
	}
      }
    }

    /* ... add final text */
    /*void*/addText( strText , bEscape );
  }


  /**
   * @param  strTexts	comma-separated list of values
   */
  /* Convenience+ */
  public
  void
  addTexts  (String	strTexts	)
  throws Exception      /* TooManyValuesException */
  {
    /*void*/addTexts( strTexts , Config.mc_bProcessTexts );
  }


  public
  String
  getString  (          )
  {
/*IMPLEMENT: more effeciently*/
    int         idx;
    String      str             = "";

    for (idx = 0 ; idx < m_vector.size(  ) ; idx++) {
      if (idx != 0) {
	str += m_strDelim;
      }
      str += m_vector.elementAt( idx );
    }

    return str;
  }


/*----------------------------------------------------------------------------
 *                                              Class Public Constants
 *--------------------------------------------------------------------------*/
  final static public  int	UNBOUNDED	= -1;


/*----------------------------------------------------------------------------
 *                                              Object Protected Methods
 *--------------------------------------------------------------------------*/

  /**
   * Add text after null and value count checks.
   * Should be overridden by any subclasses using other value types.
   */
  protected
  void
  addDirect  (Object    object  )
  throws Exception
  {
    /* probably work for all classes, though not sensible for all that work */
    /*void*/addTextDirect( (String)object );
  }


  /**
   * Add text after null and value count checks.
   * Should be overridden by any subclasses using other value types.
   */
  protected
  void
  addTextDirect  (String        strText )
  throws Exception
  {
    /*void*/m_vector.addElement( checkText( strText ) );
  }


  protected
  void
  setDelim  (String	strDelim	)
  {
    m_strDelim = strDelim;
  }


  protected
  void
  setMinMax  (int	Min	,
	      int	Max	)
  {
    /* if caller wants to set them to something goofy, let them */
    m_Min = Min;
    m_Max = Max;
  }


/*----------------------------------------------------------------------------
 *                                              Class Private Constants
 *--------------------------------------------------------------------------*/

  /*Assert: m_vector never null after construction*/
  protected  Vector	m_vector	= null;
  /*Assert: these never change after construction*/
  protected  int	m_Min		= 0;
  protected  int	m_Max		= UNBOUNDED;
  protected  String	m_strDelim	= ",";


/*----------------------------------------------------------------------------
 *						Main
 *--------------------------------------------------------------------------*/

  static public
  void
  main  (String         argv[]  )
  {
    Iterator    enum;
    Categories	categories	= new Categories( "fish,are, good" );
    Categories	categories2	= new Categories(  );

    try {
      /*void*/System.out.println( categories.toString(  ) );
      for (enum = categories.getList(  ) ; enum.hasNext(  ) ; ) {
	/*void*/System.out.println( "\t" + (String)enum.next(  ) );
      }

      /*void*/categories.addTexts( "xxx,yyy,zzz" );
      /*void*/System.out.println( categories.toString(  ) );

      /*void*/categories.set( "aaa,bbb,ccc" );
      /*void*/System.out.println( categories.toString(  ) );

      /*void*/System.out.println( categories2.isComplete(  ) );
      /*void*/categories2.set( "1.0;2.0" );
      /*void*/System.out.println( categories2 );
      /*void*/categories2.addTexts( "3.0" );
      /*void*/System.out.println( categories2 );
    } catch (Exception	e	) {
      /*void*/e.printStackTrace(  );
    }
  }


}


/* Log:
 *  0.20  2003/07/15, 2003/07/23, 2003/07/30  slh
 *        getFirstText(): option to unquote
 *        addText(): option to quote if nec
 *        addTextDirect(): check value
 *  0.10  2002/11/06  slh
 *        make delim settable (at construction)
 *        make min/max only settable at construction
 *        redo constructors
 *        comments
 *  0.01  2001/05/22 - 2001/05/23  slh
 *        getFirst(), getFirstText()
 *        getTextList() -> getList()
 *        getCount()
 *        min/max
 *        object vars were decl static
 *  0.00  2001/05/15 - 2001/05/16  slh
 *        create
 */
/*--TextListProp.java-------------------------------------------------------*/
