/*++TextListParam.java++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.Iterator;
import  java.util.StringTokenizer;
import  java.util.Vector;
import  java.util.NoSuchElementException;


/**
 * Base class for TEXT-valued parameters
 * that allow a delimiter-separated list of texts.
 *
 *<PRE>	<!-- for alignment purposes -->&nbsp;
 * Spec:
 *   section:	4.1 [2445]
 *   allowed:	---
 *</PRE>
 *
 * Methods affected by Config ProcessText flag:
 *   TextListParam(String,String)
 *   getFirstText()
 *   set(String)	which uses addTexts()
 *   add(String)
 *   addTexts(String)	but probably does nothing for you
 * Methods not affected by Config ProcessText flag include:
 *   getList()
 *   getFirst()
 *   set(Iterator)
 *   add(Object)
 *   getString()
 *
 * Subclassing:
 * override AddDirect() and AddTextDirect()
 * to make a class with different constraints on individual members of list.
 *
 * @author  slh
 * @version  0.20 2003/07/30 slh
 */
abstract public
class  TextListParam
extends  TextParam
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /**
   * Create a TextList parameter unset.
   */
  /* Note: this is the base constructor */
  protected
  TextListParam  (String	strName	)
  {
    super( strName );

    m_vector = new Vector(  );
  }


  /**
   * Create a TextList parameter set to supplied value.
   */
  /* Convenience */
  protected
  TextListParam  (String	strName		,
		  String	strValues	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    try {
      /*void*/set( strValues );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( e.getLocalizedMessage(  ) );
    }
  }


  /**
   * Create a TextList parameter unset and
   * set minimum and maximum number of elements.
   */
  protected
  TextListParam  (String	strName	,
		  int		Min	,
		  int		Max	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );

    /*void*/setMinMax( Min , Max );
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/

  public
  boolean
  isComplete  (		)
  {
    /*Note: second clause should always be true*/
    return m_Min <= m_vector.size(  ) &&
		(m_Max == UNBOUNDED || m_vector.size(  ) <= m_Max);
  }


  /**
   * @return  count of values in list
   */
  public
  int
  getCount  (		)
  {
    return m_vector.size(  );
  }


  /***
   * Get Iterator of correct object class for class.
   * @return  iterator of correct object class for class
   */
  public
  Iterator
  getList  (		)
  {
    return m_vector.iterator(  );
  }


  /**
   * @return  first value as object
   */
  /* Convenience */
  public
  Object
  getFirst  (		)
  throws NoValueException
  {
    Iterator		iter;

    iter = m_vector.iterator(  );

    if (iter.hasNext(  )) {
      return iter.next(  );
    } else {
      throw new NoValueException( getName(  ) + ": no values" );
    }
  }


  /**
   * @param  bUnquote	remove any quoting
   * @return  first value as a String
   */
  /* Convenience */
  public
  String
  getFirstText  (boolean	bUnquote	)
  throws NoValueException
  {
    /* probably work for all classes */
    String	strText;

    strText = getFirst(  ).toString(  );

    if (bUnquote && strText != null) {
      strText = checkUnquoteText( strText );
    }

    return strText;
  }


  /**
   * @return  first value as a String
   */
  /* Convenience */
  public
  String
  getFirstText  (		)
  throws NoValueException
  {
    return getFirstText( Config.mc_bProcessTexts );
  }


  /**
   * @param  strValues	comma-separated list of values
   * @param  bQuote	quote otherwise invalid text, if possible
   */
  public
  void
  set  (String	strValues	,
	boolean	bQuote		)
  throws Exception
  {
    /*Note: letting calls take care of null cases*/

    /*void*/m_vector.removeAllElements(  );

    /*void*/addTexts( strValues , bQuote );
  }


  public
  void
  set  (String	strValues	)
  throws Exception
  {
    /*void*/set( strValues , Config.mc_bProcessTexts );
  }


  /**
   * @param  iter	iterator of correct object class for class
   */
  /* convenience */
  public
  void
  set  (Iterator	iter	)
  throws Exception
  {
    /*Note: letting calls take care of null cases*/

    m_vector.removeAllElements(  );

    for ( ; iter.hasNext(  ) ; ) {
      add( iter.next(  ) );
    }
  }


  /**
   * @param  object	object of correct class for class
   */
  public
  void
  add  (Object	object	)
  throws Exception	/* TooManyValuesException, NullPointerException */
  {
    if (object == null) {
      throw new NullPointerException( getName(  ) +
		": object (Object) is null" );
    }

    if (m_Max == UNBOUNDED || m_vector.size(  ) < m_Max) {
      /*Note: addDirect() appropriate for current subclass is called*/
      addDirect( object );
    } else {
      throw new TooManyValuesException( getName(  ) +
		": attempt to add more than maximum allowed values" );
    }
  }


  /**
   * Add a single text value.
   *
   * @param  strText	single text value to be added
   * @param  bQuote	quote otherwise invalid text, if possible
   */
  public
  void
  addText  (String	strText	,
	    boolean	bQuote	)
  throws Exception	/* TooManyValuesException, NullPointerException */
  {
    if (strText == null) {
      throw new NullPointerException( getName(  ) +
		": strText (String) is null" );
    }

    if (m_Max == UNBOUNDED || m_vector.size(  ) < m_Max) {
      /*Note: addTextDirect() appropriate for current subclass is called*/
      if (bQuote && strText != null) {
	strText = checkQuoteText( strText , false );
      }
      addTextDirect( strText );
    } else {
      throw new TooManyValuesException( getName(  ) +
		": attempt to add more than maximum allowed values" );
    }
  }


  public
  void
  addText  (String	strText	)
  throws Exception	/* TooManyValuesException, NullPointerException */
  {
    /*void*/addText( strText , Config.mc_bProcessTexts );
  }


  /**
   * @param  strTexts	comma-separated list of values
   * @param  bQuote	quote otherwise invalid text, if possible
   */
  /* Convenience+ */
  public
  void
  addTexts  (String	strTexts	,
	     boolean	bQuote		)
  throws Exception	/* TooManyValuesException */
  {
    /*Note: lets constructor catch nulls*/
    StringTokenizer	st;
    String		strToken;
    String		strText;

    if (strTexts == null) {
      throw new NullPointerException( getName(  ) +
		": strTexts (String) is null" );
    }

    st = new StringTokenizer( strTexts , Strings.strSafeComplement , true );
    for ( ; st.hasMoreTokens(  ) ; ) {
      strToken = st.nextToken(  );
      strText = strToken;
      if (strToken.equals( Strings.strParamValueQuote )) {
	try {
	  strToken = st.nextToken( Strings.strQSafeComplement );
	  strText += strToken;
	  strToken = st.nextToken(  );
	  if (!strToken.equals( Strings.strParamValueQuote )) {
	    throw new IllegalArgumentException( getName(  ) +
			": expected: '" + Strings.chParamValueQuote + "'" );
	  }
	  strText += strToken;
	} catch (NoSuchElementException	e	) {
	  throw new IllegalArgumentException( getName(  ) +
			": invalid text: " + strTexts );
	}
      }
      /*void*/addText( strText , bQuote );
      if (st.hasMoreTokens(  )) {
	strToken = st.nextToken(  );
	if (!strToken.equals( Strings.strParamValueSeparator )) {
	  throw new IllegalArgumentException( getName(  ) +
			": expected: '" +
			Strings.chParamValueSeparator + "'" );
	}
      }
    }
  }


  public
  void
  addTexts  (String	strTexts	)
  throws Exception	/* TooManyValuesException */
  {
    /*void*/addTexts( strTexts , Config.mc_bProcessTexts );
  }


  public
  String
  getString  (		)
  {
/*IMPLEMENT: more effeciently*/
    int		idx;
    String	str		= "";

    for (idx = 0 ; idx < m_vector.size(  ) ; idx++) {
      if (idx != 0) {
	str += Strings.chParamValueSeparator;
      }
      str += m_vector.elementAt( idx );
    }

    return str;
  }


/*----------------------------------------------------------------------------
 *						Class Public Constants
 *--------------------------------------------------------------------------*/

  final static public  int	UNBOUNDED	= -1;


/*----------------------------------------------------------------------------
 *						Object Protected Methods
 *--------------------------------------------------------------------------*/

  /**
   * Add text after null and value count checks.
   * Should be overridden by any subclasses using other value types.
   */
  protected
  void
  addDirect  (Object	object	)
  throws Exception
  {
    /* probably work for all classes, though not sensible for all that work */
    addTextDirect( (String)object );
  }


  /**
   * Add text after null and value count checks.
   * Should be overridden by any subclasses using other value types.
   */
  protected
  void
  addTextDirect  (String	strText	)
  throws Exception
  {
    m_vector.addElement( checkText( strText ) );
  }


  protected
  void
  setMinMax  (int	Min	,
	      int	Max	)
  {
    /* if caller wants to set them to something goofy, let them */
    m_Min = Min;
    m_Max = Max;
  }


/*----------------------------------------------------------------------------
 *						Object Protected Variables
 *--------------------------------------------------------------------------*/

  /*Assert: m_vector never null after construction*/
  protected  Vector	m_vector	= null;
  /*Assert: these never change after construction*/
  protected  int	m_Min		= 0;
  protected  int	m_Max		= UNBOUNDED;

}


/* Log:
 *  0.20  2003/06/18, 2003/06/23, 2003/07/01, 2003/07/14, 2003/07/30  slh
 *        getFirstText(): option to unquote
 *        addText(): option to quote if nec
 *        addTexts(): properly parse quoted strings
 *        addTextDirect(): check value
 *        rm settable delim feature
 *  0.10  2002/11/06  slh
 *        (resync'ing w/TextListProp:)
 *        make delim settable (at construction)
 *        make min/max only settable at construction
 *        redo constructors
 *        comments
 *        getCount()
 *  0.01  2001/05/21 - 2001/05/22  slh
 *        getFirst(), getFirstText()
 *        getTextList() -> getList()
 *        min/max
 *        object vars were decl static
 *  0.00  2001/05/16 - 2001/05/17  slh
 *        create
 */
/*--TextListParam.java------------------------------------------------------*/
