/*++EnumProp.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.util.HashMap;
import  java.util.Map;
import  java.util.Iterator;


/**
 * Base class for iCalendar text-valued properies
 * with (mostly) enumerable values.
 * <P>
 * Subclassing:
 * Override <CODE>configEnum()</CODE>
 * to create and populate <CODE>m_hmValue</CODE>
 * with enumerated values expressed as <CODE>Integer</CODE> keys and
 * <CODE>String</CODE> values and
 * to set <CODE>m_strDefault</CODE> (if there is a default value).
 * If iana-tokens and/or x-names are to be allowed,
 * use <CODE>IANATOKEN</CODE> and/or <CODE>XNAME</CODE> as keys and
 * nulls as values.
 * <CODE>NONE</CODE> is always allowed.
 * <CODE>configEnum()</CODE> will be called from all the constructors.
 *
 * @author  slh
 * @version  0.21 2003/04/29 slh
 */
abstract public
class  EnumProp
extends  TextProp
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /**
   * Create an enumerated property unset.
   */
  /* this is the base constructor */
  protected
  EnumProp  (String	strName	)
  {
    super( strName );
    /*void*/configEnum(  );

    if (m_strDefault != null) {
      try {
	/*void*/set( m_strDefault );
      } catch (Exception	e	) {
	throw new InternalError( e.getLocalizedMessage(  ) );
      }
    }
  }


  /**
   * Create an enumerated property set to supplied value.
   */
  public
  EnumProp  (String	strName		,
	     String	strValue	)
  {
    this( strName );

    try {
      /*void*/set( strValue );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( e.getLocalizedMessage(  ) );
    }
  }


  /**
   * Create an enumerated property set to supplied value.
   */
  public
  EnumProp  (String	strName	,
	     int	iValue	)
  {
    this( strName );

    try {
      /*void*/set( iValue );
    } catch (Exception	e	) {
      throw new IllegalArgumentException( e.getLocalizedMessage(  ) );
    }
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/
/*-------------------------------------- get/set			*/

  /**
   * Get id value.
   */
  public
  int
  getEnum  (		)
  {
    return m_iValue;
  }


  /**
   * Set by id value.
   *
   * Note: will only allow <CODE>NONE</CODE> value.
   */
  public
  void
  setEnum  (int	iValue	)
  throws Exception
  {
    String	str;

    if (iValue == NONE) {
      m_iValue = iValue;
      /*void*/super.set( null );
    } else if (null != (str = intToString( iValue ))) {
      m_iValue = iValue;
      /*void*/super.set( str );
    } else {
      throw new IllegalArgumentException(
		getName(  ) + ": invalid int: " + iValue );
    }
  }


  /**
   * Set by id value.
   *
   * Note: just a wrapper for <CODE>setEnum(int)</CODE>.
   */
  public
  void
  set  (int	iValue	)
  throws Exception
  {
    /*void*/setEnum( iValue );
  }


  /**
   * Set by string value.
   */
  public
  void
  set  (String	strValue	)
  throws Exception
  {
    if        (isNone( strValue )) {
      m_iValue = NONE;
    } else if (Config.isIANAToken( strValue ) &&
	       m_hmValue != null &&
	       m_hmValue.containsKey( new Integer( IANATOKEN ) )) {
      m_iValue = IANATOKEN;
    } else if (Config.isXName( strValue ) &&
	       m_hmValue != null &&
	       m_hmValue.containsKey( new Integer( XNAME ) )) {
      m_iValue = XNAME;
    /*Note: must use stringToInt instead of contains for case-insensitivity*/
    } else if (NONE != stringToInt( strValue )) {
      m_iValue = stringToInt( strValue );
    } else {
      throw new IllegalArgumentException(
		getName(  ) + ": invalid string: \"" + strValue + "\"" );
    }
    /* either valid or exception was thrown */
    /*void*/super.set( strValue );
  }


/*-------------------------------------- isComplete()			*/

  public
  boolean
  isComplete  (		)
  {
    return m_iValue != NONE;
  }


/*-------------------------------------- Other Tests			*/

  /**
   * Is value None?
   */
  public
  boolean
  isNone  (String	strValue	)
  {
    return strValue == null;
  }


/*-------------------------------------- String/int Converters		*/

  /**
   * Translates an integer enumeration to a string.
   *
   * @return  corresponding string if one of enum, null otherwise.
   */
  public
  String
  intToString  (int	iValue	)
  {
    if (m_hmValue == null) {
      return null;						/*RETURN*/
    }

    return (String)m_hmValue.get( new Integer( iValue ) );	/*RETURN*/
  }


  /**
   * Translates string to integer enumeration.
   *
   * @return  corresponding integer if one of enum,
   *          <CODE>NONE</CODE> otherwise.
   */
  public
  int
  stringToInt  (String	strValue	)
  {
    Iterator	iter;
    Map.Entry	entry;
    String	str;

    if (m_hmValue == null) {
      return NONE;						/*RETURN*/
    }

    if (strValue == null) {
      return NONE;						/*RETURN*/
    }

    for (iter = m_hmValue.entrySet(  ).iterator(  ) ; iter.hasNext(  ) ;  ) {
      entry = (Map.Entry)iter.next(  );
      str = (String)entry.getValue(  );
      if (strValue.equalsIgnoreCase( str )) {
	return ((Integer)entry.getKey(  )).intValue(  );	/*RETURN*/
      }
    }

    return NONE;						/*RETURN*/
  }


/*----------------------------------------------------------------------------
 *						Object Protected Methods
 *--------------------------------------------------------------------------*/

  /**
   * Configures enumeration by populating <CODE>m_hmValue</CODE>.
   */
  protected
  void
  configEnum  (		)
  {
    m_hmValue = new HashMap(  );
    m_hmValue.put( new Integer( IANATOKEN ) , null );
    m_hmValue.put( new Integer( XNAME ) , null );

    /* no default */
  }

  
/*----------------------------------------------------------------------------
 *						Class Public Constants
 *--------------------------------------------------------------------------*/

  final static public  int	NONE		= 0;
  final static public  int	IANATOKEN	= -1;
  final static public  int	XNAME		= -2;


/*----------------------------------------------------------------------------
 *						Object Protected Variables
 *--------------------------------------------------------------------------*/

  protected  int	m_iValue	= NONE;

  protected  HashMap	m_hmValue	= null;
  protected  String	m_strDefault	= null;


}


/* Log:
 *  0.21  2003/04/29  slh
 *        (new year, new version...)
 *        use Config.isIANAToken/isXName()
 *        intToString/stringToInt(): protected -> public
 *  0.20  2002/11/13, 2002/12/02  slh
 *        EnumProp replacement: table driven enum values
 *  0.10  2002/10/23, 2002/10/29  slh
 *        made class public (e.g. for use in cap package)
 *        made class abstract (wasn't already because ...?)
 *  0.00  2001/04/24 - 2001/04/25, 2001/05/08  slh
 *        create
 */
/*--EnumProp.java-----------------------------------------------------------*/
