/*++DurationProp.java+++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Log:
 * see eof
 */


package  edu.washington.cac.calendar.icalendar;


import  java.text.DecimalFormat;
import  java.text.NumberFormat;
import  java.text.ParsePosition;


/**
 * Base class for iCalendar DURATION-valued properties.
 *
 *<PRE>	<!-- for alignment purposes -->&nbsp;
 * Spec:
 *   section:	4.3.6 [2445]
 *   allowed:	---
 *</PRE>
 *
 * @author  slh
 * @version  0.20 2003/10/08 slh
 */
abstract public
class  DurationProp
extends  IntegerProp
{

/*----------------------------------------------------------------------------
 *						Constructors
 *--------------------------------------------------------------------------*/

  /*Note: attempting to use the default constructor is wrong*/


  /* this is the base constructor */
  /**
   * Create an DURATION-valued property, unset.
   */
  protected
  DurationProp  (String	strName	)
  {
    super( strName );
  }


  /**
   * Create a DURATION-valued property set to supplied value.
   */
  protected
  DurationProp  (String	strName		,
		 String	strValue	)
  {
    /*Note: letting calls handle any nulls*/

    this( strName );
    /*Note: do separate set() so that super version does not get called*/
    /*void*/set( strValue );
  }


  /**
   * Create an DURATION-valued property set to supplied value.
   */
  protected
  DurationProp  (String		strName	,
		 Integer	IValue	)
  {
    super( strName , IValue );
  }


  /**
   * Create an DURATION-valued property set to supplied value.
   */
  protected
  DurationProp  (String	strName	,
		 int	iValue	)
  {
    super( strName , iValue );
  }


/*----------------------------------------------------------------------------
 *						Object Public Methods
 *--------------------------------------------------------------------------*/

  public
  void
  set  (String	strValue	)
  {
    NumberFormat	nf		= DecimalFormat.getNumberInstance(  );
    ParsePosition	pp		= new ParsePosition( 0 );
    Number		number;
    int			Negative	= 1;
    int			Duration	= 0;
    String		strLast		= null;

    /*void*/nf.setParseIntegerOnly( true );
    /*void*/pp.setIndex( 0 );

    if        (strValue.startsWith( "+" , pp.getIndex(  ) )) {
      /*void*/pp.setIndex( pp.getIndex(  ) + 1 );
    } else if (strValue.startsWith( "-" , pp.getIndex(  ) )) {
      Negative = -1;
      /*void*/pp.setIndex( pp.getIndex(  ) + 1 );
    }
    if (!strValue.startsWith( "P" , pp.getIndex(  ) )) {
      throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a 'P': " + strValue );
    }
    /*void*/pp.setIndex( pp.getIndex(  ) + 1 );

   /* week or day portion */
    if (!strValue.startsWith( "T" , pp.getIndex(  ) )) {
      number = nf.parse( strValue , pp );
      if (number == null) {
	throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a number: " + strValue );
      }
      Duration += number.intValue(  ) *
			getMultiplierWD( strValue , pp , null );
      if        (strValue.startsWith( "W" , pp.getIndex(  ) )) {
	if (strValue.length(  ) != pp.getIndex(  ) + 1) {
	  throw new IllegalArgumentException( getName(  ) +
			": illegal duration format; not expecting a 'W': " +
			 strValue );
	}
      } else if (!strValue.startsWith( "D" , pp.getIndex(  ) )) {
	throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a 'D': " + strValue );
      }
      /*void*/pp.setIndex( pp.getIndex(  ) + 1 );
    }

   /* time portion */
    if (strValue.startsWith( "T" , pp.getIndex(  ) )) {
      /*void*/pp.setIndex( pp.getIndex(  ) + 1 );

      number = nf.parse( strValue , pp );
      if (number == null) {
	throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a number: " + strValue );
      }
      Duration += number.intValue(  ) *
			getMultiplierHMS( strValue , pp , strLast );
      strLast = strValue.substring( pp.getIndex(  ) , pp.getIndex(  ) + 1 );
      /*void*/pp.setIndex( pp.getIndex(  ) + 1 );

      if (strValue.length(  ) != pp.getIndex(  )) {
	number = nf.parse( strValue , pp );
	if (number == null) {
	  throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a number: " + strValue );
	}
	Duration += number.intValue(  ) *
			getMultiplierHMS( strValue , pp , strLast );
	strLast = strValue.substring( pp.getIndex(  ) , pp.getIndex(  ) + 1 );
	/*void*/pp.setIndex( pp.getIndex(  ) + 1 );
      }

      if (strValue.length(  ) != pp.getIndex(  )) {
	number = nf.parse( strValue , pp );
	if (number == null) {
	  throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a number: " + strValue );
	}
	Duration += number.intValue(  ) *
			getMultiplierHMS( strValue , pp , strLast );
	strLast = strValue.substring( pp.getIndex(  ) , pp.getIndex(  ) + 1 );
	/*void*/pp.setIndex( pp.getIndex(  ) + 1 );
      }
    }

    if (strValue.length(  ) != pp.getIndex(  )) {
      throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; extraneous text: " + strValue );
    }

    /*void*/set( Negative * Duration );
  }

  
  public
  String
  getString  (		)
  {
    int		Duration;
    String	strDuration	= "";
    int		Tmp;

    try {
      Duration = get(  );
    } catch (NoValueException	e	) {
      return null;						/*RETURN*/
    }

    if (Duration < 0) {
      strDuration += "-";
      Duration = -Duration;
    }
    strDuration += "P";

    if (Duration % mc_SecsPerWeek == 0) {
      strDuration += Duration / mc_SecsPerWeek + "W";
      Duration = 0;
    }	/* and done */

    if (Duration >= mc_SecsPerDay) {
      Tmp = Duration / mc_SecsPerDay;
      strDuration += Tmp + "D";
      Duration -= Tmp * mc_SecsPerDay;
    }
    if (Duration != 0) {
      strDuration += "T";
    }
    if (Duration >= mc_SecsPerHour) {
      Tmp = Duration / mc_SecsPerHour;
      strDuration += Tmp + "H";
      Duration -= Tmp * mc_SecsPerHour;
    }
    if (Duration >= mc_SecsPerMinute) {
      Tmp = Duration / mc_SecsPerMinute;
      strDuration += Tmp + "M";
      Duration -= Tmp * mc_SecsPerMinute;
    }
    if (Duration > 0) {
      strDuration += Duration + "S";
      Duration = 0;
    }

    return strDuration;						/*RETURN*/
  }


/*----------------------------------------------------------------------------
 *						Object Private Methods
 *--------------------------------------------------------------------------*/

  private
  int
  getMultiplierWD  (String		strValue	,
		    ParsePosition	pp		,
		    String		strLast		)	/* not used */
  {
    if        (strValue.startsWith( "W" , pp.getIndex(  ) )) {
      return mc_SecsPerWeek;
    } else if (strValue.startsWith( "D" , pp.getIndex(  ) )) {
      return mc_SecsPerDay;
    } else {
      throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected a 'W' or 'D': " +
		strValue );
    }
  }


  private
  int
  getMultiplierHMS  (String		strValue	,
		     ParsePosition	pp		,
		     String		strLast		)
  throws IllegalArgumentException
  {
    if        (strValue.startsWith( "H" , pp.getIndex(  ) ) &&
	       strLast == null) {
      return mc_SecsPerHour;
    } else if (strValue.startsWith( "M" , pp.getIndex(  ) ) &&
	       (strLast == null || strLast.startsWith( "H" ))) {
      return mc_SecsPerMinute;
    } else if (strValue.startsWith( "S" , pp.getIndex(  ) )) {
      return 1;
    } else {
      throw new IllegalArgumentException( getName(  ) +
		": illegal duration format; expected an 'H', 'M' or 'S': " +
		strValue );
    }
  }


/*----------------------------------------------------------------------------
 *						Class Protected Constants
 *--------------------------------------------------------------------------*/

  static protected final  int	mc_SecsPerMinute	= 60;
  static protected final  int	mc_SecsPerHour		=
							60 * mc_SecsPerMinute;
  static protected final  int	mc_SecsPerDay		= 24 * mc_SecsPerHour;
  static protected final  int	mc_SecsPerWeek		= 7 * mc_SecsPerDay;


/*----------------------------------------------------------------------------
 *						Main
 *--------------------------------------------------------------------------*/

  static public
  void
  main  (String	argv[]	)
  {
    Duration	duration	= new Duration(  );
    int		Value;
    String	strValue;

    try {
      Value = 3872;
      /*void*/System.out.println( "int: " + Value );
      /*void*/duration.set( Value );
      /*void*/System.out.println( duration );
      /*void*/strValue = "-P11W";
      /*void*/System.out.println( "string: " + strValue );
      /*void*/duration.set( strValue );
      /*void*/System.out.println( duration );
      /*void*/strValue = "-P13DT5H4M37S";
      /*void*/System.out.println( "string: " + strValue );
      /*void*/duration.set( strValue );
      /*void*/System.out.println( duration );
      /*void*/strValue = "PT37S";
      /*void*/System.out.println( "string: " + strValue );
      /*void*/duration.set( strValue );
      /*void*/System.out.println( duration );
      /*void*/System.out.println(
		"Note: next set will not match in display," +
		" but will match in value" );
      /*void*/strValue = "P7D";
      /*void*/System.out.println( "string: " + strValue );
      /*void*/duration.set( strValue );
      /*void*/System.out.println( duration );
    } catch (IllegalArgumentException	e	) {
      /*void*/System.out.println(
		"IllegalArgumentException:" + e.getLocalizedMessage(  ) );
      /*void*/e.printStackTrace(  );
    } catch (Exception	e	) {
      /*void*/System.out.println(
		"other exception:" + e.getLocalizedMessage(  ) );
      /*void*/e.printStackTrace(  );
    }

    try {
      /*void*/System.out.println(
		"Note: next set will throw IllegalArgumentException" );
      /*void*/strValue = "P7D3H";
      /*void*/System.out.println( "string: " + strValue );
      /*void*/duration.set( strValue );
      /*void*/System.out.println( duration );
    } catch (IllegalArgumentException	e	) {
      /*void*/System.out.println( e.getLocalizedMessage(  ) );
    }
  }

}


/* Log:
 *  0.20  2003/10/08  slh
 *        deriving from IntegerProp
 *        new contructors: int and Integer variants added
 *        isComplete(), get(), set(int): rm; rely on super versions
 *        set(String): redo exceptions; use DecimalFormat.getInstance()
 *        main(): more tests
 *  0.10  2002/12/10  slh
 *        redo constructors
 *        drop toString() (let super do with getString())
 *        isComplete()
 *  0.00  2001/08/20 - 2001/08/21  slh
 *        create from TimeProp.java
 */
/*--DurationProp.java-------------------------------------------------------*/
