/*

 @author Greg Barnes, Cameron Mallory, modified for SQL Server by Clay Gravelle

      This file is used to create the tables and indexes for the UW Calendar database.
      It also loads some sample data for testing purposes.   
      After creating the calendardb database, use SQL Analyzer to load and run this file.

      SQL Server specific issues:
      1) SQL Server does not support date or time data types.
         Possible solution: Use DATENAME function to create computed columns.
      2) SQL Server does not support substrings on indexes.
         Possible solution: Create index on computed column.
*/

USE calendardb
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'events' AND type = 'U')
   DROP Table events
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'recurrence_dates' AND type = 'U')
   DROP Table recurrence_dates
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'recurrence_rules' AND type = 'U')
   DROP Table recurrence_rules
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'bywhat' AND type = 'U')
   DROP Table bywhat
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'event_keywords' AND type = 'U')
   DROP Table event_keywords
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'keywords' AND type = 'U')
   DROP Table keywords
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'locations' AND type = 'U')
   DROP Table locations
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'sponsors' AND type = 'U')
   DROP Table sponsors
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'auth' AND type = 'U')
   DROP Table auth
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'users' AND type = 'U')
   DROP Table users
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'eventrefs' AND type = 'U')
   DROP Table eventrefs
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'master_details' AND type = 'U')
   DROP Table master_details
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'subscriptions' AND type = 'U')
   DROP Table subscriptions
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'calendars' AND type = 'U')
   DROP Table calendars
GO

IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'keyword_attrs' AND type = 'U')
   DROP Table keyword_attrs
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'lastmods' AND type = 'U')
   DROP Table lastmods
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'adminGroups' AND type = 'U')
   DROP Table adminGroups
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'adminGroupMembers' AND type = 'U')
   DROP Table adminGroupMembers
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'authprefs' AND type = 'U')
   DROP Table authprefs
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'authprefKeywords' AND type = 'U')
   DROP Table authprefKeywords
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'authprefLocations' AND type = 'U')
   DROP Table authprefLocations
GO
IF EXISTS (SELECT name FROM sysobjects
   WHERE name = 'authprefSponsors' AND type = 'U')
   DROP Table authprefSponsors
GO


CREATE TABLE events (
   eventid int IDENTITY(1,1) primary key NOT NULL,   /* key for most tables */
   lastmod timestamp NOT NULL,         /* updated automatically on update */
   seq int NOT NULL,            /* number of changes to this record */
   publicf char(1) NOT NULL,         /* T or F */
   created datetime NOT NULL,      
   startdate datetime NOT NULL,         /* SQL Server doesn't support date type */
   starttime datetime NULL,         /* SQL Server doesn't support time type */
   enddate datetime NOT NULL,
   endtime datetime NULL,
   shortdesc varchar(255) NOT NULL,
   longdesc ntext NULL,
   longdesc1 ntext NULL,
   longdesc2 ntext NULL,
   link varchar(255) NULL,            /* URL */
            /* T = tentative, F = confirmed, C = Cancelled, P = postponed */
   status char(1) NOT NULL,         /* DEFAULT should be 'F' */
   locationid int NOT NULL,         /* key in location table */
   sponsorid int NOT NULL,            /* key in sponsor table */
   creator varchar(20) NOT NULL,         /* UWNetID */
    /* N = no recurrence, M = master record for an expanded recurrence */
   recurring_status CHAR(1) NOT NULL DEFAULT('N'),

   cost varchar(255) NULL            /* not in iCal */
)
GO

CREATE INDEX epu on events(publicf)
CREATE INDEX esd on events(startdate)
CREATE INDEX eed on events(enddate)
CREATE INDEX esh on events(shortdesc)         /* Could limit to char(10) by indexing a view */
CREATE INDEX est on events(status)
CREATE INDEX eli on events(locationid)
CREATE INDEX esi on events(sponsorid)
CREATE INDEX ecr on events(creator)
GO

# all instances of expanded recurrences
CREATE TABLE master_details (
  detailid int IDENTITY(1,1) primary key NOT NULL,   # id of an instance
  master_eventid int NOT NULL,                      # from events table
  startdate datetime NOT NULL,
  starttime datetime,
  enddate datetime NOT NULL,
  endtime datetime,
);
GO

CREATE INDEX mdme on master_details(master_eventid)
CREATE INDEX mdsd on master_details(startdate)
CREATE INDEX mded master_details(enddate)
CREATE INDEX mdse on master_details(startdate, enddate)
GO

CREATE TABLE recurrence_dates (
   eventid int NOT NULL,            /* from events table */
   type char(1) NOT NULL,            /* (R)ecurrence or (E)xclusion
            for specific dates:  recurstart, or recurstart + recurtime
            (set recurend = recurstart; recurendtime = NULL)
             for periods:  recurstart + recurtime (through) recurend + recurendtime */
   recurstartdate datetime NOT NULL,      /* cannot be null if recurendtime isn't null */
   recurstarttime datetime NULL,         /* for end of period ( == recurstart if no period) */
   recurenddate datetime NOT NULL,         /* cannot be null if designating a period */
   recurendtime datetime NULL 
)
GO

CREATE INDEX rde on recurrence_dates(eventid)
CREATE INDEX rdrsd on recurrence_dates(recurstartdate)
CREATE INDEX rdred on recurrence_dates(recurenddate)
GO

CREATE TABLE recurrence_rules (
   ruleid int IDENTITY(1,1) primary key NOT NULL,   /* key for recurrence rules tables */
   eventid int NOT NULL,            /* (R)ecurrence or (E)xclusion */
            /* S(econdly), M(inutely), H(ourly), D(aily), W(eekly), (m)O(nthly), Y(early) */
   type char(1) NOT NULL,
   freq char(1) NOT NULL,
   multiplier int default 1 NOT NULL,      /* number of freq* units between occur. */
   [count] int NULL,            /* recur set number of times */
   untildate datetime NOT NULL,         /* recur until date */
   untiltime datetime NULL,
   wkst char(2) default('SU') NOT NULL      /* week start: dow (SU/MO/TU/WE/TH/FR/SA) */
)
GO

CREATE INDEX rrud on recurrence_rules(untildate)
GO

CREATE TABLE bywhat (
   ruleid int NOT NULL,            /* from recurrence_rules table */
         /* S(econd), M(inute), H(our), (month)D(ay), Y(earday), W(eekno), (m)O(nth), (d)A(y), (set)P(os) */
   type char(1) NOT NULL,
   number int NOT NULL    /* type = S, M:   0-59
                  type = H:    0-23
                  type = D:    -31 - -1, 1-31
                  type = Y, P: -366 - -1, 1-366
                  type = W:    -53 - -1, 1-53
                  type = O:    1-12
                  type = A:    0-53 * 10 + 1-7
                    where 0-53 encodes week (0 = null) and 1-7
                 encodes day of week (Sunday = 1, Saturday = 7)
            */
)
GO

CREATE INDEX bwri on bywhat(ruleid)
GO

CREATE TABLE event_keywords (            /* (references to) keywords for the event */
   eventid int NOT NULL ,
   keywordid int NOT NULL             /* key in keyword table */
)
GO

CREATE INDEX ekei on event_keywords(eventid)
CREATE INDEX ekki on event_keywords(keywordid)
GO

CREATE TABLE keywords (
   keywordid int IDENTITY(1,1) primary key NOT NULL,   /* keywords themselves */
   word varchar(255) NOT NULL,
   longdesc ntext NULL,               /* further description */
   longdesc1 ntext NULL,
   creator varchar(20) NOT NULL,            /* UWNetID */
   publicf char(1) NOT NULL default('t')         /* T or F */
)
GO

CREATE INDEX kww on keywords(word)            /* should be keyed on 1st char(10) */
CREATE INDEX kwc on keywords(creator)
CREATE INDEX kwp on keywords(publicf)
GO

CREATE TABLE locations (
   locationid int IDENTITY (1,1) primary key NOT NULL,   /* location for event */
            /* on-campus: building + room
                      off-campus:  name of place
                   nowhere (e.g., radio program): name of station, URL, etc.
            */
   address varchar(255) NOT NULL,
             /* on-campus: UW, plus any other necessary info
               off-campus: address, plus any other necessary info
                 */
   subaddress varchar(255) NULL,
   link varchar(255) NULL,               /* URL */
   creator varchar(20) NOT NULL,            /* UWNetID */
   publicf char(1) NOT NULL             /* T or F */
)
GO

CREATE INDEX la on locations(address)
CREATE INDEX lc on locations(creator)
CREATE INDEX lp on locations(publicf)
GO

CREATE TABLE sponsors (                  /* sponsor of event */
   sponsorid int IDENTITY(1,1) primary key NOT NULL,
   [name] varchar(255) default('') NOT NULL,
   phone varchar(255) NULL,
   email varchar(255) NULL,
   link varchar(255) NULL,               /* URL */
   creator varchar(20) NOT NULL,            /* UWNetID */
   publicf char(1) NOT NULL default('T')         /* T or F */
)
GO

CREATE INDEX sn on sponsors([name])
CREATE INDEX sc on sponsors(creator)
CREATE INDEX sp on sponsors(publicf)
GO

CREATE TABLE auth (
   userid int NOT NULL,
   usertype varchar(1) NULL,
   lastname varchar(255) NULL,
   firstname varchar(255) NULL,
   phone varchar(255) NULL,
   email varchar(255) NULL,
   department varchar(255) NULL
)
GO

CREATE TABLE users (               /* users who have actually logged in */
   userid int IDENTITY(1,1) primary key NOT NULL,
   username varchar(20) NOT NULL,
   created datetime NULL,            /* when the user first logged in */
   last_logon datetime NULL,         /* when the user last logged in */
   last_access datetime NULL,         /* when the user last read an object */
   last_modify datetime NULL          /* when the user last modified an object */
)
GO

CREATE INDEX uu on users(username)
GO

CREATE TABLE eventrefs (      /* (references to) events for a user */
   eventid int NOT NULL,
   userid int NOT NULL,
   purpose char(1) NOT NULL default('E')         /* T or F */
)
GO

CREATE INDEX ere on eventrefs(eventid)
CREATE INDEX eru on eventrefs(userid)
GO

CREATE TABLE calendars (
  calendarid int IDENTITY(1,1) primary key NOT NULL,
  name VARCHAR(200), 
  type CHAR(1) NOT NULL,
  ref_num  int NOT NULL,
  parent int NOT NULL DEFAULT(0)
);

CREATE TABLE subscriptions (    -- references to calendars for a user
  userid int NOT NULL,
  calendarid int NOT NULL,
);

CREATE INDEX suu on subscriptions(calendarid)
CREATE INDEX suc on subscriptions(userid)
GO



/* useful values for location and sponsors */
SET IDENTITY_INSERT locations ON         /* allow explicit insert into locationid */
go
insert into locations (locationid, address, creator, publicf) values (1, 'none', '007', 'T')
insert into locations (locationid, address, creator, publicf) values (2, 'unknown', '007', 'T')
insert into locations (locationid, address, subaddress, creator, publicf) values
 (3, 'deleted', 'Used as a replacement when the owner of a public location used by others deletes the location', '007', 'T')
GO
SET IDENTITY_INSERT locations OFF
GO
SET IDENTITY_INSERT sponsors ON         /* allow explicit insert into sponsorid */
go
insert into sponsors (sponsorid, [name], creator, publicf) values (1, 'none', '007', 'T');
insert into sponsors (sponsorid, [name], creator, publicf) values (2, 'unknown', '007', 'T');
insert into sponsors (sponsorid, [name], phone, creator, publicf) values
 (3, 'deleted', 'Used as a replacement when the owner of a public sponsor used by others deletes the sponsor', '007', 'T')

/* sponsors for the 'all calendars' page */
insert into sponsors (sponsorid, name, creator, publicf) values (1667, 'Benefits Office', '007', 'T');
insert into sponsors (sponsorid, name, creator, publicf) values (1673, 'Recruiting & Candidate Services', '007', 'T');
insert into sponsors (sponsorid, name, creator, publicf) values (1672, 'Training & Development', '007', 'T');
insert into sponsors (sponsorid, name, creator, publicf) values (1668, 'Work/Life Office', '007', 'T');

GO
SET IDENTITY_INSERT sponsors OFF
GO
SET IDENTITY_INSERT keywords ON         /* allow explicit insert into keywordid */
GO
/* default keywords for official public events */
insert into keywords (keywordid, word, creator, publicf) values (35, 'Academic Calendar', '007other', 'T');
insert into keywords (keywordid, word, creator, publicf) values (34, 'Holidays', '007other', 'T');
insert into keywords (keywordid, word, creator, publicf) values (33, 'Alerts', '007other', 'T');
insert into keywords (keywordid, word, creator, publicf) values (1, 'Benefits', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (2, 'Commencement', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (3, 'Concerts', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (24, 'Conferences', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (4, 'Dance', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (22, 'Deadlines', '007unofficial', 'T');
insert into keywords (keywordid, word, creator, publicf) values (5, 'Drama', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (6, 'Exhibits', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (7, 'Films', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (25, 'Forums', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (8, 'Health Sciences', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (9, 'Hearings', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (10, 'Lectures', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (29, 'Meetings', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (11, 'Open Houses', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (12, 'Operas', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (13, 'Radio', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (14, 'Readings', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (15, 'Sales', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (16, 'Seminars', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (17, 'Special Events', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (18, 'Sports', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (19, 'Television', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (20, 'Tours', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (21, 'Training', '007main', 'T');
insert into keywords (keywordid, word, creator, publicf) values (26, 'Workshops', '007main', 'T');
GO
SET IDENTITY_INSERT keywords OFF
GO

/* bootstrap users for pubevents auth table and class meetings */
insert into users (userid, username) values (2, '007-ts');
insert into users (userid, username) values (3, '1user');
insert into users (userid, username) values (4, 'caladmin');
insert into users (userid, username) values (5, 'calcontent');
insert into users (userid, username) values (6, 'calowner');
insert into users (userid, username) values (7, 'calowner2');
insert into users (userid, username) values (8, 'calowner3');

insert into auth (userid, usertype, lastname)
    values (3, 'A', 'dummy user to get things started');
insert into auth (userid, usertype, lastname)
    values (4, 'A', 'dummy admin user');
insert into auth (userid, usertype, lastname)
    values (5, 'C', 'dummy calendar content admin user');
insert into auth (userid, usertype, lastname)
    values (6, 'P', 'dummy public events owner user');
insert into auth (userid, usertype, lastname)
    values (7, 'P', 'dummy public events owner user');
insert into auth (userid, usertype, lastname)
    values (8, 'P', 'dummy public events owner user');

-- keyword_attrs - store attributes of keywords
--     for example, flag alert categories
--  kaid          unique key
--  kakeywordid:  id of keyword entry
--  kaname:       attribute name
--  kaval:        attribute value

CREATE TABLE keyword_attrs (
  kaid INTEGER NOT NULL IDENTITY,
  kakeywordid INTEGER NOT NULL,
  kaname VARCHAR(255) NOT NULL,
  kaval VARCHAR(255) NOT NULL
);

CREATE INDEX kwaid on keyword_attrs(kakeywordid);
CREATE INDEX kwanm on keyword_attrs(kaname);
CREATE INDEX kwavl on keyword_attrs(kaval);

-- table to store lastmod timestamp - indexed and searched by name
CREATE TABLE lastmods (
  name VARCHAR(255) NOT NULL,	-- key
  lastmod TIMESTAMP NOT NULL
);

CREATE INDEX lmnm on lastmods(name);

insert into lastmods values ('pubevents', '2003-07-03 13:12:29.0');

CREATE TABLE adminGroups (
  groupname VARCHAR(255) NOT NULL,
  description LONGVARCHAR,
  groupOwner VARCHAR(20) NOT NULL,
  ownerid VARCHAR(20) NOT NULL
);

CREATE INDEX agnm on adminGroups(groupname);
CREATE INDEX aggo on adminGroups(groupOwner);
CREATE INDEX agoi on adminGroups(ownerid);

insert into adminGroups values ('grp1', 'Demo group with 2 members',
                                'caladmin', 'agrp_grp1');
insert into adminGroups values ('grp2', 'Demo group with 2 members',
                                'calowner3', 'agrp_grp2');

CREATE TABLE adminGroupMembers (
  groupname VARCHAR(255),
  userid VARCHAR(20) NOT NULL
);

CREATE INDEX agmnm on adminGroupMembers(groupname);
CREATE INDEX agmui on adminGroupMembers(userid);

insert into adminGroupMembers values ('grp1', 'caladmin');
insert into adminGroupMembers values ('grp1', 'calowner');
insert into adminGroupMembers values ('grp2', 'calowner2');
insert into adminGroupMembers values ('grp2', 'calowner3');

-- Preferences for authorized users
CREATE TABLE authprefs (
  userid INTEGER NOT NULL,
  autoaddKeywords CHAR(1) NOT NULL,  -- DEFAULT 'T',
  autoaddLocations CHAR(1) NOT NULL, -- DEFAULT 'T',
  autoaddSponsors CHAR(1) NOT NULL   -- DEFAULT 'T'
);

CREATE INDEX apui on authprefs(userid);

-- Preferred keywords for authorized users
CREATE TABLE authprefKeywords (
  userid INTEGER NOT NULL,
  keywordid INTEGER NOT NULL
);

CREATE INDEX apkui on authprefKeywords(userid);

-- Preferred locations for authorized users
CREATE TABLE authprefLocations (
  userid INTEGER NOT NULL,
  locationid INTEGER NOT NULL
);

CREATE INDEX aplui on authprefLocations(userid);

-- Preferred sponsors for authorized users
CREATE TABLE authprefSponsors (
  userid INTEGER NOT NULL,
  sponsorid INTEGER NOT NULL
);

CREATE INDEX apsui on authprefSponsors(userid);

insert into auth (userid, usertype, lastname) values (3, 'A', 'dummy user to get things started');
GO
