package edu.washington.cac.calfacade.shared;

import java.io.Serializable;

/** Value object to represent a calendar user.
 *
 *   @author Mike Douglass douglm@rpi.edu
 *  @version 1.0
 */
public class UserVO implements Serializable {
  private String name;  // null for guest
  private String nameApp;
  private int id;

  private static final int invalidId = -1;

  /** Value from edu.washington.cac.calfacade.shared.UserAuth
      Only affects PublicEvents administration
   */
  private int rights;

  /** True if this user is trying to acces public events.
   */
  private boolean publicEvents;

  /** Create a guest user vo
   */
  public UserVO() {
    this(null, null, 0, 0, false);
  }

  /** Create a user vo usually based on a single string name from an object,
   * e.g., an event.
   *
   * @param  name            String user id for database
   */
  public UserVO(String name) {
    this(name, name, 0, 0, false);
  }

  /** Create a user vo by specifying all its fields
   *
   * @param  name            String user id for database
   * @param  nameApp         Application (external) user name
    */
  public UserVO(String name,
                String nameApp,
                int id,
                int rights,
                boolean publicEvents) {
    this.name = name;
    this.nameApp = nameApp;
    this.id = id;
    this.rights = rights;
    this.publicEvents = publicEvents;
  }

  public void setId(int val) {
    id = val;
  }

  public int getId() {
    return id;
  }

  public void setName(String val) {
    name = val;
  }

  public String getName() {
    return name;
  }

  public void setNameApp(String val) {
    nameApp = val;
  }

  public String getNameApp() {
    return nameApp;
  }

  public boolean isGuest() {
    return name == null;
  }

  /** Set the rights this user has
   *
   * @param val    int access rights value as defined by
   *               edu.washington.cac.calfacade.shared.UserAuth
   */
  public void setRights(int val) {
    rights = val;
  }

  /** Get the rights this user has
   *
   * @return int   value as defined by
   *               edu.washington.cac.calfacade.shared.UserAuth
   */
  public int getRights() {
    return rights;
  }

  /** Show if this user accessing public events
   */
  public void setPublicEvents(boolean val) {
    publicEvents = val;
  }

  /** Is this user accessing public events
   */
  public boolean getPublicEvents() {
    return publicEvents;
  }

  /** Check for priv user
   *
   * @return boolean    true for super user
   */
  public boolean isSuperUser() {
    return (rights & UserAuth.superUser) != 0;
  }

  /** Check for alert user
   *
   * @return boolean    true for alert user
   */
  public boolean isAlertUser() {
    return (rights & UserAuth.alertUser) != 0;
  }

  /** Check for public events owner user
   *
   * @return boolean    true for public events owner user
   */
  public boolean isOwnerUser() {
    return (rights & UserAuth.publicEventUser) != 0;
  }

  /** Check for content admin user
   *
   * @return boolean    true for content admin user
   */
  public boolean isContentAdminUser() {
    return (rights & UserAuth.contentAdminUser) != 0;
  }

  public String toString() {
    return "UserVO{name=" + name + ", nameApp=" + nameApp +
                   ", id=" + id + ", rights=" + rights + "}";
  }

  /** Check for equality
   *
   * @param  o          Object to compare.
   * @return true       if object represents same user.
   */
  public boolean equals(Object o) {
    if (!(o instanceof UserVO)) {
      return false;
    }

    UserVO that = (UserVO)o;

    if (name == null) {
      return that.name == null;
    }

    return name.equals(that.name);
  }

  public int hashCode() {
    int hc = 1;

    if (name != null) {
      hc = name.hashCode();
    }

    return hc;
  }
}
