package edu.washington.cac.calfacade.shared;

import java.io.Serializable;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * Recurrence information for an event.
 *
 * Currently, all recurrences are expanded in the Events object.
 * If an event is recurring, it is either the 'master' instance of the
 * recurring event, or not.
 *
 * The id of the master instance is used when adding public eventrefs
 * to a user's personal calendar, and when removing a recurring event.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @author Greg Barnes
 * @version 1.0
 */
public class RecurrenceVO {
  /** masterId when the master id is not known or doesn't exist */
  public static final int NO_MASTER = -1;

  /** True if this is the master instance
   */
  private boolean master;

  /** True if this event recurs
   */
  private boolean recurs;

  /** id of the master event of the recurrence */
  private int masterId;

  /** All known instances of the recurrence */
  private SortedSet instances;

  /** Constructor
   *
   * @param masterId   int id of the master event of the recurrence
   */
  public RecurrenceVO(boolean master,
                      boolean recurs,
                      int masterId,
                      SortedSet instances) {
    this.master = master;
    this.recurs = recurs;
    this.masterId = masterId;
    this.instances = instances;
  }

  public void setMaster(boolean val) {
    master = val;
  }

  /** Is this a master recurrence?
   *
   * @return boolean    true if this is a master recurrence?
   */
  public boolean isMaster() {
    return master;
  }

  public void setRecurs(boolean val) {
    recurs = val;
  }

  /** Is this a recurring event?
   *
   * @return boolean    true if this is a recurring event?
   */
  public boolean getRecurs() {
    return recurs;
  }

  /** Set the id of the master event
   *
   * @param val     int id of the master event
   */
  public void setMasterId(int val) {
    masterId = val;
  }

  /**  Get the id of the master id of this recurrence (assuming it exists)
    @return int  id of the master id of this recurrence (assuming it exists)
   */
  public int getMasterId() {
    return masterId;
  }

  /**  Add an instance to the set of instances
   *
   * @param val   EventVO instance to add
   */
  public void addInstance(EventVO val) {
    if (instances == null) {
      instances = new TreeSet();
    }

    instances.add(val);
  }

  /** Get the set of instances for this event
   *
   * @return SortedSet   set of instances for this event
   */
  public SortedSet getInstances() {
    return instances;
  }
}
