package edu.washington.cac.calfacade.shared;

import java.io.Serializable;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.sql.Time;

/** Representation of the MyCalendar uwcal class.
 *
 * This object is intended to allow applications to interact with the
 * calendar back end. It does not represent the internal stored structure of a
 * MyCalendar object.
 *
 *   @author Mike Douglass douglm@rpi.edu
 *  @version 1.0
 */
public class MyCalendarVO implements Serializable {
  /** Time and date as a Calendar object.
   */
  private Calendar calendar;

  /** Days of the week indexed by day of the week number
   */
  private static final String[] dayNames = new String[7]; // indexed from 0
  private static final String[] shortDayNames = new String[7]; // indexed from 0

  /** Days of week ajusted with the first day of the week at index 0
   * This is used for presentation.
   */
  private static final String[] dayNamesAdjusted; // indexed from 0
  private static final String[] shortDayNamesAdjusted = new String[7]; // indexed from 0

  private static final int firstDayOfWeek;
  private static final int lastDayOfWeek;
  private static final int numberDaysInWeek;

  static {
    SimpleDateFormat formatter = new SimpleDateFormat("EEEE");
    Calendar c = Calendar.getInstance();
    ArrayList dow = new ArrayList();

    firstDayOfWeek = c.getFirstDayOfWeek();

    /** Get the number of days in a week. Is this ever anything other than 7?
     */
    numberDaysInWeek = c.getMaximum(Calendar.DAY_OF_WEEK) -
                       c.getMinimum(Calendar.DAY_OF_WEEK) + 1;

    lastDayOfWeek = (numberDaysInWeek + firstDayOfWeek - 1) % numberDaysInWeek;

    for (int i = 0; i < 7; i++) {
      c.set(Calendar.DAY_OF_WEEK, i);

      dayNames[i] = formatter.format(c.getTime());
      dow.add(dayNames[i]);
    }

    if (firstDayOfWeek != 0) {
      /* Rotate the days of the week so that the first day is at the beginning
       */

      int fdow = firstDayOfWeek;
      while (fdow > 0) {
        Object o = dow.remove(0);
        dow.add(o);

        fdow--;
      }
      dayNamesAdjusted = (String[])dow.toArray(new String[dow.size()]);
    } else {
      dayNamesAdjusted = dayNames;
    }

    for (int i = 0; i < 7; i++) {
      shortDayNames[i] = dayNames[i].substring(0, 2);
      shortDayNamesAdjusted[i] = dayNamesAdjusted[i].substring(0, 2);
    }
  }

  /** Time and date as a Date object.
   */
  private Date time;

  /** Create a MyCalendarVO object representing the current date and time.
   */
  public MyCalendarVO() {
    this(Calendar.getInstance());
  }

  /** Create a MyCalendarVO object representing a particular date and time.
   *
   * @param calendar     Calendar object representing the date and time.
   */
  public MyCalendarVO(Calendar calendar) {
    this.calendar = calendar;
    time = calendar.getTime();
  }

  /** Create a MyCalendarVO object representing a particular date and,
   * optionally, time.
   *
   * @param date   Non-null Date object.
   * @param time   Time object. If null the MyCalendarVO object has no time.
   */
  public MyCalendarVO(Date date, Time time) {
    this.time = time;
    calendar = Calendar.getInstance();

    if (time == null) {
      calendar.setTime(date);
    } else {
      calendar.setTime(time);

      Calendar cal2 = Calendar.getInstance();
      cal2.setTime(date);

      calendar.set(cal2.get(Calendar.YEAR), cal2.get(Calendar.MONTH),
                   cal2.get(Calendar.DATE));
    }
  }

  /** Create a MyCalendarVO object representing a particular date.
   * A <code>MyCalendar</code> created using this constructor will
   * not have an associated time.
   *
   * @param val   String date in question, represented as an
   *            8 characters of the form <code>YYYYMMDD</code>
   */
  public MyCalendarVO(String val) throws CalFacadeException {
    this(Calendar.getInstance());

    if (!CheckData.checkDateString(val)) {
      throw new CalFacadeException("Bad date: " + val);
    }

    calendar.set(CheckData.yearNum(val),
                 CheckData.monthNum(val),
                 CheckData.dayNum(val), 0, 0, 0);
    calendar.set(Calendar.MILLISECOND, 0);

    /* gb:  The next line is to workaround some nasty JDK behavior.

       if you set() as above, then set the DAY_OF_WEEK (as
       getFirstDayOfThisWeek() and other methods below do), the DAY_OF_MONTH
       we entered above is thrown out when determining the new DAY_OF_MONTH
       in favor of the unset WEEK_OF_MONTH (or WEEK_OF_YEAR).

       To avoid this, we make the following unnecessary call to get
       the WEEK_OF_YEAR value, which should recompute all the other 'fields'
       of the CALENDAR.
      */
    calendar.get(Calendar.WEEK_OF_YEAR);
  }

  /** Return Calendar object representing this object
   *
   * @return Calendar    object representing this object
   */
  public Calendar getCalendar() {
    return calendar;
  }

  /** Return Date object representing this object
   *
   * @return Date    object representing this object
   */
  public Date getTime() {
    return calendar.getTime();
  }

  /** Test whether the object represents a particular time
   * (as well as a particular date)
   *
   * @return Does the object represent a particular time,
   * as well as a particular date?
   */
  public boolean getHasTime() {
    return time != null;
  }

  public long getTimeInMillis() {
    return calendar.getTimeInMillis();
  }

  /** ===================================================================
   *                Components of the date
   *  =================================================================== */

  /** Get the year for this object.
   *
   * @return int    year for this object
   */
  public int getYear() {
    return calendar.get(Calendar.YEAR);
  }

  /** Get the number of the month for this object.
   * Note:  The first month is number 1.
   *
   * @return int    month number for this object
   */
  public int getMonth() {
    return calendar.get(Calendar.MONTH) + 1;
  }

  /** Get the day of the month for this object.
   *
   * @return int    day of the month for this object
   */
  public int getDay() {
    return calendar.get(Calendar.DATE);
  }

  /** Get the hour of the (24 hour) day for this object.
   *
   * @return int    hour of the day for this object
   */
  public int getHour24() {
    return calendar.get(Calendar.HOUR_OF_DAY);
  }

  /** Get the hour of the day for this object.
   *
   * @return int    hour of the day for this object
   */
  public int getHour() {
    return calendar.get(Calendar.HOUR);
  }

  /** Get the minute of the hour for this object.
   *
   * @return int    minute of the hour for this object
   */
  public int getMinute() {
    return calendar.get(Calendar.MINUTE);
  }

  /** Get the am/pm value
   *
   * @return int   am/pm for this object.
   */
  public int getAmPm() {
    return calendar.get(Calendar.AM_PM);
  }

  /** Get the day of the week for this object.
   *
   * @return int    day of the week
   */
  public int getDayOfWeek() {
    return getDayOfWeek(calendar);
  }

  /** Get the day name for this object.
   *
   * @return String    day name for this object
   */
  public String getDayName() {
    return dayNames[getDayOfWeek()];
  }

  /** Get the day of the week for a Calendar object
   *
   * @param c The date to evaluate
   * @return The day of the week for a Calendar
   */
  public int getDayOfWeek(Calendar c) {
    return c.get(Calendar.DAY_OF_WEEK) % numberDaysInWeek;
  }

  /** Get the long month name for this object.
   *
   * @return String    month name for this object
   */
  public String getMonthName() {
    return getComponent(DateFormat.MONTH_FIELD, DateFormat.LONG);
  }

  /** Get a four-digit representation of the year
   *
   * @return String   four-digit representation of the year for this object.
   */
  public String getFourDigitYear() {
    return String.valueOf(getYear());
  }

  /** Get a two-digit representation of the month of year
   *
   * @return String   two-digit representation of the month of year for
   *                  this object.
   */
  public String getTwoDigitMonth() {
    return getTwoDigit(getMonth());
  }

  /** Get a two-digit representation of the day of the month
   *
   * @return String   two-digit representation of the day of the month for
   *                  this object.
   */
  public String getTwoDigitDay() {
    return getTwoDigit(getDay());
  }

  /** Get a two-digit representation of the 24 hour day hour
   *
   * @return String   two-digit representation of the hour for this object.
   */
  public String getTwoDigitHour24() {
    return getTwoDigit(getHour24());
  }

  /** Get a two-digit representation of the hour
   *
   * @return String   two-digit representation of the hour for this object.
   */
  public String getTwoDigitHour() {
    return getTwoDigit(getHour());
  }

  /** Get a two-digit representation of the minutes
   *
   * @return String   two-digit representation of the minutes for this object.
   */
  public String getTwoDigitMinute() {
    return getTwoDigit(getMinute());
  }

  /** ===================================================================
   *                Get various representations of date/time
   *  =================================================================== */

  /**  Get a short String representation of the time of day
   *
   * @return String        Short representation of the time of day
   *            represented by this object.
   *            If there is no time, returns a zero length string.
   */
  public String getTimeString() {
    return getTimeString(DateFormat.getTimeInstance(DateFormat.SHORT));
  }

  /** Get a <code>String</code> representation of the time of day
   *
   * @param df      DateFormat format for the result
   * @return String  time of day, formatted per df or zero length String for
   *                  no time value
   */
  public String getTimeString(DateFormat df) {
    if (time == null) {
      return "";
    }

    return df.format(getTime());
  }

  /**  Get a short String representation of the date
   *
   * @return String        Short representation of the date
   *            represented by this object.
   */
  public String getDateString() {
    return getDateString(DateFormat.SHORT);
  }

  /**  Get a long String representation of the date
   *
   * @return String        long representation of the date
   *            represented by this object.
   */
  public String getLongDateString() {
    return getDateString(DateFormat.LONG);
  }

  /**  Get a full String representation of the date
   *
   * @return String        full representation of the date
   *            represented by this object.
   */
  public String getFullDateString() {
    return getDateString(DateFormat.FULL);
  }

  /** Get a <code>String</code> representation of the date
   *
   * @param style       int DateFormat style for the result
   * @return String  date formatted per df
   */
  public String getDateString(int style) {
    return getDateString(DateFormat.getDateInstance(style));
  }

  /** Get a <code>String</code> representation of the date
   *
   * @param df      DateFormat format for the result
   * @return String  date formatted per df
   */
  public String getDateString(DateFormat df) {
    return df.format(getTime());
  }

  /** Get an eight-digit String representation of the date
   *
   * @return String  date in the form <code>YYYYMMDD</code>
   */
  public String getDateDigits() {
    return getFourDigitYear() + getTwoDigitMonth() + getTwoDigitDay();
  }

  /** ===================================================================
   *                Adding and subtracting time
   *  =================================================================== */

  /** Get a calendar some time later or earlier than this one
   *
   * @param unit Units to add/subtract, <i>e.g.</i>,
   *          <code>Calendar.DATE</code> to add days
   * @param amount Number of units to add or subtract.  Use negative
   *            numbers to subtract
   * @return MyCalendarVO    new object corresponding to this
   *                  object +/- the appropriate number of units
   */
  public MyCalendarVO addTime(int unit, int amount) {
    return new MyCalendarVO(add(calendar, unit, amount));
  }

  /**  Get a MyCalendarVO object one day earlier.
   *
   * @return MyCalendarVO    equivalent to this object one day earlier.
   */
  public MyCalendarVO getYesterday() {
    return addTime(Calendar.DATE, -1);
  }

  /**  Get a MyCalendarVO object one day later.
   *
   * @return MyCalendarVO    equivalent to this object one day later.
   */
  public MyCalendarVO getTomorrow() {
    return addTime(Calendar.DATE, 1);
  }

  /**  Get a MyCalendarVO object one week earlier.
   *
   * @return MyCalendarVO    equivalent to this object one week earlier.
   */
  public MyCalendarVO getPrevWeek() {
    return addTime(Calendar.WEEK_OF_YEAR, -1);
  }

  /**  Get a MyCalendarVO object one week later.
   *
   * @return MyCalendarVO    equivalent to this object one week later.
   */
  public MyCalendarVO getNextWeek() {
    return addTime(Calendar.WEEK_OF_YEAR, 1);
  }

  /**  Get a MyCalendarVO object one month earlier.
   *
   * @return MyCalendarVO    equivalent to this object one month earlier.
   */
  public MyCalendarVO getPrevMonth() {
    return addTime(Calendar.MONTH, -1);
  }

  /**  Get a MyCalendarVO object one month later.
   *
   * @return MyCalendarVO    equivalent to this object one month later.
   */
  public MyCalendarVO getNextMonth() {
    return addTime(Calendar.MONTH, 1);
  }

  /**  Get a MyCalendarVO object one year earlier.
   *
   * @return MyCalendarVO    equivalent to this object one year earlier.
   */
  public MyCalendarVO getPrevYear() {
    return addTime(Calendar.YEAR, -1);
  }

  /**  Get a MyCalendarVO object one year later.
   *
   * @return MyCalendarVO    equivalent to this object one year later.
   */
  public MyCalendarVO getNextYear() {
    return addTime(Calendar.YEAR, 1);
  }

  /** ===================================================================
   *                Relative times
   *  =================================================================== */

  /** Get the first day of the year for this object
   *
   * @return MyCalendarVO    representing the first day of the year for this
   *             object, e.g. if this object represents February 5, 2000,
   *             returns a MyCalendarVO object representing January 1, 2000.
   */
  public MyCalendarVO getFirstDayOfThisYear() {
    Calendar firstDay = (Calendar)calendar.clone();
    firstDay.set(Calendar.DAY_OF_YEAR,
                 calendar.getMinimum(Calendar.DAY_OF_YEAR));
    return new MyCalendarVO(firstDay);
  }

  /** Get the last day of the year for this object
   *
   * @return MyCalendarVO    epresenting the last day of the year for this
   *             object, e.g. if this object represents February 5, 2000,
   *             returns a MyCalendarVO object representing December 31, 2000.
   */
  public MyCalendarVO getLastDayOfThisYear() {
    return getFirstDayOfThisYear().addTime(Calendar.YEAR, 1).
               addTime(Calendar.DATE, -1);
  }

  /**  Get the first day of the month for this object
   *
   * @return MyCalendarVO    representing the first day of the month for this
   *             object, e.g. if this object represents February 5, 2000,
   *             returns a MyCalendarVO object representing February 1, 2000.
   */
  public MyCalendarVO getFirstDayOfThisMonth() {
    Calendar firstDay = (Calendar)calendar.clone();
    firstDay.set(Calendar.DAY_OF_MONTH,
                 calendar.getMinimum(Calendar.DAY_OF_MONTH));
    return new MyCalendarVO(firstDay);
  }

  /** Get the last day of the month for this object
   *
   * @return MyCalendarVO    representing the last day of the month for this
   *             object, e.g. if this object represents February 5, 2000,
   *             returns a MyCalendarVO object representing February 29, 2000.
   */
  public MyCalendarVO getLastDayOfThisMonth() {
    return getFirstDayOfThisMonth().addTime(Calendar.MONTH, 1).
               addTime(Calendar.DATE, -1);
  }

  /**  Get the first day of the week for this object
   *
   * @return MyCalendarVO    representing the first day of the week for this
   *             object.
   */
  public MyCalendarVO getFirstDayOfThisWeek() {
    Calendar firstDay = (Calendar)calendar.clone();
    firstDay.set(Calendar.DAY_OF_WEEK,
                 calendar.getMinimum(Calendar.DAY_OF_WEEK));
    return new MyCalendarVO(firstDay);
  }

  /** Get the last day of the week for this object
   *
   * @return MyCalendarVO    representing the last day of the week for this
   *             object.
   */
  public MyCalendarVO getLastDayOfThisWeek() {
    return getFirstDayOfThisWeek().addTime(Calendar.WEEK_OF_YEAR, 1).
               addTime(Calendar.DATE, -1);
  }

  /** ===================================================================
   *                Comparisons
   *  =================================================================== */

  public boolean isSameDate(MyCalendarVO that) {
    if (that.getDay() != getDay()) {
      return false;
    }

    if (that.getMonth() != getMonth()) {
      return false;
    }

    if (that.getYear() != getYear()) {
      return false;
    }

    return true;
  }

  /** ===================================================================
   *                Useful methods
   *  =================================================================== */

  /** Get the names of the day of the week
   *
   * @return String[] day names
   */
  public static String[] getDayNames() {
    return dayNames;
  }

  /** Get the short names of the day of the week
   *
   * @return String day names
   */
  public static String[] getShortDayNames() {
    return shortDayNames;
  }

  /** Get the names of the day of the week
   * Elements have been adjusted so that the first day of the week is at 0
   *
   * @return String[] day names
   */
  public static String[] getDayNamesAdjusted() {
    return dayNamesAdjusted;
  }

  /** Get the short names of the day of the week
   * Elements have been adjusted so that the first day of the week is at 0
   *
   * @return String day names
   */
  public static String[] getShortDayNamesAdjusted() {
    return shortDayNamesAdjusted;
  }

  /** Return the dayOfWeek regarded as the first day
   *
   * @return int   firstDayOfWeek
   */
  public static int getFirstDayOfWeek() {
    return firstDayOfWeek;
  }

  /** Return the dayOfWeek regarded as the last day
   *
   * @return int   lastDayOfWeek
   */
  public static int getLastDayOfWeek() {
    return lastDayOfWeek;
  }

  public String toString() {
    return getDateString() + " " + getTimeString();
  }

  /** ===================================================================
   *                Private methods
   *  =================================================================== */

  /** Get a two-digit representation of a one to two-digit number
   *
   * @param  i         int one or two digit number
   * @return String    two-digit representation of the number
   */
  private static String getTwoDigit(int i) {
    if (i < 10) {
      return "0" + i;
    }

    return String.valueOf(i);
  }

  /** Get a calendar some time later or earlier than this one
   *
   * @param c    The calendar to start with
   * @param unit The unit of time to add or subtract, <i>e.g.</i>,
   *           <code>MONTH</code>.  For possible values, see the constants in
   *           <code>java.util.Calendar</code>
   * @param amount The number of units to add or subtract.  A positive
   *              value means to add, a negative value to subtract
   * @return Calendar    equivalent to c some time later or earlier
   */
  private static Calendar add(Calendar c, int unit, int amount) {
    Calendar newc = (Calendar)c.clone();
    newc.add(unit, amount);
    return newc;
  }

  /** Get a String representation of a particular time
   *  field of the object.
   *
   * @param field The field to be returned,
   *        <i>e.g.</i>, <code>MONTH_FIELD</code>.  For possible values, see
   *       the constants in <code>java.text.DateFormat</code>
   * @param dateFormat The style of <code>DateFormat</code> to use,
   *            <i>e.g.</i>, <code>SHORT</code>.  For possible values, see
   *           the constants in <code>java.text.DateFormat</code>.
   * @return A <code>String</code> representation of a particular time
   *             field of the object.
   */
  private String getComponent(int field, int dateFormat) {
    FieldPosition f = new FieldPosition(field);
    StringBuffer s = DateFormat.
        getDateTimeInstance(dateFormat, dateFormat).
        format(getTime(), new StringBuffer(), f);
    return s.substring(f.getBeginIndex(), f.getEndIndex());
  }
}
