package edu.washington.cac.calfacade.shared;

import java.io.Serializable;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Enumeration;
import java.util.Vector;

/** An Event in UWCal. This value object does no consistency or validity
 * checking. It does not modify the dates.
 *
 *  @version 1.0
 */
public class EventVO implements Serializable {
  /** Some constants. These are copies of the constants in the data objects.
   *  I want to remove all references to data classes in the facade. We
   *  should probably move those constants to here or to a global
   *  constant defs class.
   */

  public final static int NoLocationId = 1;
  public final static int UnknownLocationId = 2;

  /** Unique ID of a location that has been deleted (An event can
   *   use a location created by another user, and if the user
   *   deletes the location, the 'deleted' location can be used
   *   to indicate this event to this user.)
   */
  public final static int DeletedLocationId = 3;

  public final static int DefaultLocationId = NoLocationId;

  public final static int NoSponsorId = 1;
  public final static int UnknownSponsorId = 2;
  public final static int DeletedSponsorId = 3;

  public final static int DefaultSponsorId = NoSponsorId;

  private int eventid;
  private String shortdesc;

  /** This should be set by the back end code.
   */
  public static final int maxLongdescLength = 500;

  private String longdesc;
  private String longdesc1;
  private Date startdate;
  private Time starttime;
  private Date enddate;
  private Time endtime;
  private String link, cost;

  /** User who created this event */
  private UserVO creator;

  private boolean isPublic;
  private int locationid = DefaultLocationId;
  private int sponsorid = DefaultSponsorId;

  private int seq;

  private Timestamp lastmod;

  private Timestamp created;

  /** Information about whether this event is a recurring event, and
   * if it is, how it recurs.
   *  null means no rcurrence
   */
  private RecurrenceVO recur;

  /** A vector of KeywordVO objects
   */
  private Vector keywords = null;

  /** This may or may not be set to populate fully
   */
  private SponsorVO sponsor;

  /** This may or may not be set to populate fully
   */
  private LocationVO location;

  /** editable is set at retrieval to indicate an event owned by the current
   * user. This only has significance for the personal calendar.
   */
  private boolean editable;

  public final static int kindEntry = 0;
  public final static int kindAdded = 1;
  public final static int kindUndeletable = 2;

  /** kind tells us something about where the event comes from
   * user. This only has significance for the personal calendar.
   *  0 - actual event entry
   *  1 - 'added' event - from a reference
   *  2 - from a subscription
   */
  private int kind;

  public EventVO() {
    created = new Timestamp(System.currentTimeMillis());
  }

  /** Create an event by specifying all its fields

      @param eventid Unique ID for the event
      @param shortdesc    String  Short description
      @param longdesc     String Long description
      @param longdesc1    String another long description
      @param startdate    Date start Date
      @param starttime    Time starting Time (can be null)
      @param enddate      Date end Date (can be null)
      @param endtime      Time end Time (can be null)
      @param isPublic     boolean true if the event is viewable by everyone
      @param link         String URL with more info
      @param creator      UserVO creator of the event
      @param cost         String Cost to attend
      @param recur        RecurrenceVO information
      @param locationid   int unique ID of the event's location
      @param sponsorid    int unique ID of the event's sponsor
      @param seq          int sequence number
      @param lastmod      Timestamp last modification time
      @param created      Timestamp creation time
    */
  public EventVO(int eventid,
               String shortdesc,
               String longdesc,
               String longdesc1,
               Date startdate,
               Time starttime,
               Date enddate,
               Time endtime,
               boolean isPublic,
               String link,
               UserVO creator,
               String cost,
               RecurrenceVO recur,
               int locationid,
               int sponsorid,
               int seq,
               Timestamp lastmod,
               Timestamp created) {
    this.eventid = eventid;
    this.shortdesc = shortdesc;
    this.longdesc = longdesc;
    this.longdesc1 = longdesc1;
    this.startdate = startdate;
    this.starttime = starttime;
    this.enddate = enddate;
    this.endtime = endtime;
    this.isPublic = isPublic;
    this.link = link;
    this.creator = creator;
    this.cost = cost;
    this.recur = recur;
    this.locationid = locationid;
    this.sponsorid = sponsorid;
    this.seq = seq;
    this.lastmod = lastmod;
    this.created = created;
  }

  /**
    Set the id for this event
    @param eventid Event id
   */
  public void setId(int eventid) {
    this.eventid = eventid;
  }

  /** Get the event's unique id
   *
   * @return int    the event's unique id
   */
  public int getId() {
    return eventid;
  }

  /** Set the event's short description
   *
   * @param val    String event's short description
   */
  public void setShortdesc(String val) {
    shortdesc = val;
  }

  /** Get the event's short description
   *
   * @return String   event's short description
   */
  public String getShortdesc() {
    return shortdesc;
  }

  /** Set the event's long description
   *
   * @param val    String event's short description
   */
  public void setLongdesc(String val) {
    longdesc = val;
  }

  /** Get the event's long description
   *
   *  @return String   event's long description
   */
  public String getLongdesc() {
    return longdesc;
  }

  /** Set the event's other long description
   *
   * @param val    String event's short description
   */
  public void setLongdesc1(String val) {
    longdesc1 = val;
  }

  /** Get the event's other long description
   *
   *  @return String   event's long description
   */
  public String getLongdesc1() {
    return longdesc1;
  }

  /** Set the event's starting date (not time)
   *
   *  @param  val   Date event's starting date (not time)
   */
  public void setStartdate(Date val) {
    startdate = val;
  }

  /** Get the event's starting date (not time)
   *
   *  @return The event's starting date (not time)
   */
  public Date getStartdate() {
    return startdate;
  }

  /** Set the event's starting time (not date)
   *
   *  @param  val   Time event's starting time (not date)
   */
  public void setStarttime(Time val) {
    starttime = val;
  }

  /** Get the event's starting time (not date)
   *
   *  @return The event's starting time (not date)
   */
  public Time getStarttime() {
    return starttime;
  }

  /** Set the event's ending date (not time)
   *
   *  @param  val   Date event's ending date (not time)
   */
  public void setEnddate(Date val) {
    enddate = val;
  }

  /** Get the event's ending date (not time)
   *
   * @return The event's ending date (not time)
   */
  public Date getEnddate() {
    return enddate;
  }

  /** Set the event's ending time (not date)
   *
   *  @param  val   Time event's ending time (not date)
   */
  public void setEndtime(Time val) {
    endtime = val;
  }

  /** Get the event's ending time (not date)
   *
   * @return Time    The event's ending time (not date)
   */
  public Time getEndtime() {
    return endtime;
  }

  /**
    Set recurrence information for this event
    @param val       RecurrenceVO information
   */
  public void setRecurrence(RecurrenceVO val) {
    recur = val;
  }

  /** Get recurrence information for this event
   *
   * @return ReccurrenceVO information
   */
  public RecurrenceVO getRecurrence() {
    return recur;
  }

  /**
    Is this a recurring event?
    @return Is this a recurring event?
   */
  public boolean isRecurring() {
    if (recur == null) {
      return false;
    }

    return recur.getRecurs();
  }

  /** Set the event public flag
   *  @param val    boolean true if the event is public
   */
  public void setPublic(boolean val) {
    isPublic = val;
  }

  /** Get the event public flag
   *
   *  @return boolean    true if the event is public
   */
  public boolean getPublic() {
    return isPublic;
  }

  /** Is the event public?
   *  @return boolean    true if the event is public
   */
  public boolean isPublic() {
    return isPublic;
  }

  /** Set the event's URL
   *
   *  @param link   string URL
   */
  public void setLink(String link) {
    this.link = link;
  }

  /** Get the event's URL
   *
   * @return the event's URL
   */
  public String getLink() {
    return link;
  }

  /** Set the creator
   *
   * @param val      UserVO creator of the event
   */
  public void setCreator(UserVO val) {
    creator = val;
  }

  /** Return the creator of the event
   *
   * @return UserVO       Event's creator
   */
  public UserVO getCreator() {
    return creator;
  }

  /** Set the event's cost
   *
   * @param val    String event's cost
   */
  public void setCost(String val) {
    cost = val;
  }

  /** Get the event's cost
   *
   * @return String   the event's cost
   */
  public String getCost() {
    return cost;
  }

  /** Set the unique id of the event's location id
   *
   * @param val  int unique id of the event's location
   */
  public void setLocationid(int val) {
    locationid = val;
  }

  /**
      Get the unique id of the event's location
      @return the unique id of the event's location
    */
  public int getLocationid() {
    return locationid;
  }

  /** Set the unique id of the event's sponsor id
   *
   * @param val  int unique id of the event's sponsor
   */
  public void setSponsorid(int val) {
    sponsorid = val;
  }

  /**
    Get the unique id of the event's sponsor
    @return the unique id of the event's sponsor
   */
  public int getSponsorid() {
    return sponsorid;
  }

  /** Set the seq for this event
   *
   * @param val   sequence number
   */
  public void setSeq(int val) {
    seq = val;
  }

  /** Get the events seq
   *
   * @return int    the events seq
   */
  public int getSeq() {
    return seq;
  }

  public void setLastmod(Timestamp val) {
    lastmod = val;
  }

  public Timestamp getLastmod() {
    return lastmod;
  }

  public void setCreated(Timestamp val) {
    created = val;
  }

  public Timestamp getCreated() {
    return created;
  }

  /** Set the events sponsor
   *
   * @param val  SponsorVO events sponsor
   */
  public void setSponsor(SponsorVO val) {
    sponsor = val;

    if (val != null) {
      setSponsorid(val.getId());
    }
  }

  /**
    Get the event's sponsor
    @return SponsorVO   event's sponsor
   */
  public SponsorVO getSponsor() {
    return sponsor;
  }

  public void setLocation(LocationVO val) {
    location = val;

    if (val != null) {
      setLocationid(val.getId());
    }
  }

  public LocationVO getLocation() {
    return location;
  }

  public void setEditable(boolean val) {
    editable = val;
  }

  public boolean getEditable() {
    return editable;
  }

  public void setKind(int val) {
    kind = val;
  }

  public int getKind() {
    return kind;
  }

  public void setKeywords(Vector val) {
    keywords = val;
  }

  public Vector getKeywords() {
    return keywords;
  }

  /** Get an <code>Enumeration</code> of the event's keywords
   *
   * @return Enumeration<  of the event's keywords
   */
  public Enumeration enumerateKeywords() {
    if (keywords == null) {
      return null;
    }

    return keywords.elements();
  }

  public void addKeyword(KeywordVO val) {
    if (keywords == null) {
      keywords = new Vector();
    }

    if (!keywords.contains(val)) {
      keywords.addElement(val);
    }
  }

  /** Set the ith keyword.
   *
   * @param  i         int index of keyword
   * @return KeywordVO ith keyword or null.
   */
  public void setKeyword(int i, KeywordVO val) throws Throwable {
    if (keywords == null) {
      if (i > 0) {
        throw new Exception("Keyword index out of range");
      }
      addKeyword(val);
      return;
    }

    if (i >= keywords.size()) {
      throw new Exception("Keyword index out of range");
    }

    keywords.removeElementAt(i);
    keywords.insertElementAt(val, i);
  }

  /** Return the ith keyword or null.
   *
   * @param  i         int index of keyword
   * @return KeywordVO ith keyword or null.
   */
  public KeywordVO getKeyword(int i) {
    if ((keywords == null) || (i >= keywords.size())) {
      return null;
    }

    return (KeywordVO)keywords.elementAt(i);
  }

  /** return the corresponding KeywordVO object if the given id is within
   * the keywords list
   *
   * @param  id    int keyword id
   * @return KeywordVO  keyword value object
   */
  public KeywordVO findKeyword(int id) {
    if (keywords == null) {
      return null;
    }

    Enumeration enum = keywords.elements();

    while (enum.hasMoreElements()) {
      KeywordVO k = (KeywordVO)enum.nextElement();

      if (id == k.getId()) {
        return k;
      }
    }

    return null;
  }
}
