package edu.washington.cac.calfacade.shared;

import java.sql.Date;
import java.sql.Time;
import java.util.Calendar;

/** A class to check various data fields for consistency and correctness
 * Based on DataChecker class by Greg Barnes
 *
 * @author Mike Douglass   douglm@rpi.edu
 * @author Greg Barnes
 * @version 1.1
 */
public class CheckData {
  private static final int DATE_STRING_LENGTH = 8;
  private static final int YEAR_START_INDEX = 0;
  private static final int MONTH_START_INDEX = 4;
  private static final int DAY_START_INDEX = 6;

  /** Used to obtain a number of values we might need.
   */
  private static final Calendar refCal = Calendar.getInstance();
  private static final int maxMonth = refCal.getMaximum(Calendar.MONTH);
  private static final int maxDay = refCal.getMaximum(Calendar.DATE);

  private CheckData() {}  // No instantiation allowed.

  /** Check that a date string, purportedly in YYYYMMDD format, actually is
   * and represents a valid date.
   *
   * Note that not all errors are flagged.  In particular, days that
   * are too large for a given month (e.g., Feb 30) or months too large
   * for a given year (not possible in the Gregorian calendar, but
   * perhaps in others) are not flagged as long as the day/month
   * represent valid values in *some* month/year.  These 'overflow'
   * dates are handled per the explanation in the
   * <code>java.util.Calendar</code> documentation (e.g., Feb 30, 1999
   * becomes Mar 2, 1999).
   *
   *  @param val       String to check. Should be of form yyyymmdd
   *  @return boolean  true for OK
   */
  public static boolean checkDateString(String val) {
    if (val == null || val.length() != DATE_STRING_LENGTH) {
      return false;
    }

    for (int i = 0; i < DATE_STRING_LENGTH; i++) {
      if (val.charAt(i) < '0' || val.charAt(i) > '9') {
        return false;
      }
    }

    if (monthNum(val) < 0 ||
        dayNum(val) < 1 ||
        monthNum(val) > maxMonth ||
        dayNum(val) > maxDay ||
        yearNum(val) < 1) {    // there was no year zero
      return false;
    }

    return true;
  }

    /**
     Extract the year from an eight digit date of the form YYYYMMDD

     @param eightDigitDate The eight digit date
     @return The year number
     */
  public static int yearNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(YEAR_START_INDEX,
       YEAR_START_INDEX + 4));
  }

    /**
     Extract the month number from an eight digit date of the
     form YYYYMMDD.  Following java.util.Date, the first month
     is month number 0

     @param eightDigitDate The eight digit date
     @return The month number
     */
  public static int monthNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(MONTH_START_INDEX,
        MONTH_START_INDEX + 2)) - 1;
  }

    /**
     Extract the day number from an eight digit date of the
     form YYYYMMDD.

     @param eightDigitDate The eight digit date
     @return The day number
     */
  public static int dayNum(String eightDigitDate) {
     return Integer.parseInt(eightDigitDate.substring(DAY_START_INDEX,
       DAY_START_INDEX + 2));
  }

    /**
     Check that a given hour and minute have reasonable values.

     @param hour A supposed hour in the day, using the 12 hour am/pm
      format
     @param minute A supposed minute in an hour
     @exception CaldataException If either value is too high or low
      */
  public static void checkTime(int hour, int minute) throws Exception {
    if (hour > 12 || minute > 59 || hour < 1 || minute < 0) {
      throw new Exception("Bad time: " + hour + ":" + minute);
    }
  }

  /** Check that a given set of start dates and times are consistent.
   *
   * @param startdate     Date starting date
   * @param starttime     Time starting time
   * @param enddate       Date ending date
   * @param endtime       Time ending time
   * @param sb            StringBuffer for error message
   * @return boolean      true for OK false for bad, error message in sb
   */
  public static boolean checkDateTimes(Date startdate,
                                      Time starttime,
                                      Date enddate,
                                      Time endtime,
                                      StringBuffer sb) {
    if (startdate == null) {
      sb.append("Event missing start date");
      return false;
    }

    if (enddate == null) {
      sb.append("End date unspecified");
      return false;
    }

    int comp = enddate.compareTo(startdate);

    if (comp < 0) {
      sb.append("Event ends before it begins");
      return false;
    }

    if ((comp == 0) && (endtime != null)) {
      if (starttime == null) {
        sb.append("Event needs a start time if it has an end time");
        return false;
      }

      // Same day and got a time
      if (endtime.compareTo(starttime) < 0) {
        sb.append("Event ends before it begins");
        return false;
      }
    }

    return true;
  }
/*
    private boolean blank(String s) {
       return s == null || s.trim().length() == 0;
    }

    private void checkImportantString(String s, String displayName)
      throws CaldataException
    {
       if (blank(s)) {
           throw new CaldataException(displayName + " cannot be blank.");
       }
    }

    void checkShortdesc(String shortdesc) throws CaldataException
    {
       checkImportantString(shortdesc, FormsDisplayText.NAME);
    }

    void checkAddress(String address) throws CaldataException
    {
       checkImportantString(address, FormsDisplayText.ADDRESS);
    }

  /**
    compress spaces in a string:  any sequence of whitespace characters
    is converted to a single space.
    @param s The String to compress
    @return s, compressed
   * /
  public static String compressSpace(String s) {
    StringBuffer sb = new StringBuffer(s);
    StringBuffer sb2 = new StringBuffer(s.length());
    boolean prevCharWasSpace = false;

    for (int i = 0; i < sb.length(); i++) {
      if (!Character.isWhitespace(sb.charAt(i))) {
        sb2.append(sb.charAt(i));
        prevCharWasSpace = false;
      } else {
       if (!prevCharWasSpace) {
         sb2.append(' ');
       }

        prevCharWasSpace = true;
      }
    }

    return sb2.toString();
  }

  /**
    normalize a string.  Since our strings will be used on the web,
    right now this means removing any whitespace at the beginning and
    end, and replacing any other sequence of whitespace characters with
    a single space.  If the string is blank when we're finished, return
    null.
       @param s The String to normalize
       @return the string normalized
   * /
  public static String normalize(String s) {
    if (compressSpace(s).trim().length() == 0) {
      return null;
    } else {
      return compressSpace(s).trim();
    }
  }*/
}
