package edu.washington.cac.calfacade.shared;

import edu.washington.cac.calenv.CalEnv;
import java.io.Serializable;
import org.apache.log4j.Logger;

/** This class instantiates the appropriate calintf objects defined by the
 * property defined in CalEnv.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class CalintfFactory implements Serializable {
  /** Obtain a Calintf object for an application.
   *
   * @param initObj     object used to initialise properties, for example, a
   *                    servlet.
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public static Calintf getCalintf(Object initObj,
                                   String user,
                                   int rights,
                                   boolean isGuest,
                                   boolean publicAdmin,
                                   boolean debug) throws CalFacadeException {
    try {
      CalEnv env = CalEnv.getEnv(debug);

      if (env != null) {
        env.initProperties(initObj);
      }

      return getCalintf(env, user, rights, isGuest, publicAdmin, debug);
    } catch (CalFacadeException ce) {
      throw ce;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    }
  }

  /** Obtain a Calintf object.
   *
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  public static Calintf getCalintf(String user,
                                 int rights,
                                 boolean isGuest,
                                 boolean publicAdmin,
                                 boolean debug) throws CalFacadeException {
    try {
      CalEnv env = CalEnv.getEnv(debug);

      return getCalintf(env, user, rights, isGuest, publicAdmin, debug);
    } catch (CalFacadeException ce) {
      throw ce;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    }
  }

  /** Obtain a Calintf object.
   *
   * @param env         Initialised CalEnv object
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   * @return Calintf    object for interactions with back-end.
   */
  private static Calintf getCalintf(CalEnv env,
                                    String user,
                                    int rights,
                                    boolean isGuest,
                                    boolean publicAdmin,
                                    boolean debug) throws Throwable {
    Calintf cal = null;

    String calClassName = env.getCalIntfClassName();
    if (debug) {
      Logger.getLogger("edu.washington.cac.calfacade.shared.CalintfFactory").
               debug("About to get CalIntf class " + calClassName);
    }

    if (calClassName == null) {
      throw new CalFacadeException("No property defined in calenv for " +
                          "getCalIntfClassName()");
    }

    Object o = Class.forName(calClassName).newInstance();

    if (o == null) {
      throw new CalFacadeException("Class " + calClassName +
                          " not found");
    }

    if (!(o instanceof Calintf)) {
      throw new CalFacadeException("Class defined in calenv for " +
                          "getCalIntfClassName() is not a subclass of " +
                          Calintf.class.getName());
    }

    cal = (Calintf)o;

    if (debug) {
      Logger.getLogger("edu.washington.cac.calfacade.shared.CalintfFactory").
               debug("About to initialise CalIntf class " + calClassName);
    }

    cal.init(user, rights, isGuest, publicAdmin, debug);

    return cal;
  }
}

