package edu.washington.cac.calfacade.shared;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.SortedSet;
import java.util.TreeSet;

/** This class acts as an interface to the calendar.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public interface Calintf extends Serializable {
  /** Must be called to initialise the new object.
   *
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   */
  public void init(String user,
                   int rights,
                   boolean isGuest,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException;

  /** Set this up for the given user. Must be called before any other action
   *
   * @param val   String user id - null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   */
  public void setUser(String val, int rights,
                      boolean publicAdmin) throws CalFacadeException;

  /** Returns a value object representing the current user.
   *
   * @return UserVO       representing the current user
   */
  public UserVO getUserVO() throws CalFacadeException;

  /** Get a UserAuth object which allows the application to determine what
   * special rights the user has.
   *
   * <p>Note that the returned object may require a one time initialisation
   * call to set it up correctly.
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, int)
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, Object)
   *
   * @return UserAuth    implementation.
   */
  public UserAuth getUserAuth() throws CalFacadeException;

  /** Get an AdminGroups object which allows the application to handle
   * administrative groups.
   *
   * @return AdminGroups    implementation.
   */
  public AdminGroups getAdminGroups() throws CalFacadeException;

  /** Refresh the users events cache
   */
  public void refreshEvents() throws CalFacadeException;

  /** Update lastmod timestamp.
   *
   * @exception CalFacadeException If there's a database access problem
   */
  public void touch() throws CalFacadeException;

  /** Get the value of the current public events timestamp.
   *
   * @return long       current timestamp value
   */
  public long getPublicLastmod() throws CalFacadeException;

  /** ===================================================================
   *                   Calendars and search
   *  =================================================================== */

  /** Return a list of calendar we can present to the client as a list of
   * available calendars.
   *
   * @return SortedSet   of CalendarVO
   */
  public SortedSet getCalendars() throws CalFacadeException;

  /** Set a search filter using the suppplied search string
   *
   * @param val    String search parameters
   */
  public void setSearch(String val) throws CalFacadeException;

  /** Return the current search string
   *
   * @return  String     search parameters
   */
  public String getSearch() throws CalFacadeException;

  /** Set the calendar we are interested in. This is represented by the id
   * of a calendar.
   *
   * @param  val     id of calendar - 0 for none
   * @return boolean true for OK, false for unknown calendar
   */
  public boolean setCalendar(int val) throws CalFacadeException;

  /** Get the name of the current calendar we are displaying
   *
   * @return String    name of calendar or null for all events
   */
  public String getCalendarTitle() throws CalFacadeException;

  /** This method is called to add a calendar. No consistency checks are
   * carried out at the moment.
   *
   * @param val  String values for db
   */
  public void addCalendarDef(String val) throws CalFacadeException;

  /** Set a filter which returns events starting after a given Timestamp.
   *
   * @param val   Timestamp object.
   */
  public void setLastmodGECalendar(Timestamp val) throws CalFacadeException;

  /** Set a filter which returns events starting after a given date of the
   * form yyyymmdd.
   *
   * @param date   String of form yyyymmdd.
   */
  public void setLastmodGEDateCalendar(String date) throws CalFacadeException;

  /** Set a filter which returns events starting after a given date and time
   * of the form yyyymmddhhmm.
   *
   * @param dateTime   String of form yyyymmddhhmm.
   */
  public void setLastmodGEDateTimeCalendar(String dateTime)
      throws CalFacadeException;

  /** ===================================================================
   *                   Subscriptions
   *  =================================================================== */

  /** Ensure that the users subscriptions match a set of calendars
   *
   * @param crs      TreeSet of Integer calendar ids
   * @return int[]   Number of calendars subscribed to, and number of
   *                 calendars unsubscribed to
   */
  public int[] fixSubscriptions(TreeSet crs) throws CalFacadeException;

  /** Determine if user is subscribed to a calendar
   *
   * @param id       int calendar id
   * @return boolean true if subscribed to calendars
   */
  public boolean getSubscribed(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return all keywords for this user
   *
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO[] getKeywords() throws CalFacadeException;

  /** Add a Keyword to the database. The id will be set in the parameter
   * object.
   *
   * @param val   KeywordVO object to be added
   * @return int  new sponsor id
   */
  public int addKeyword(KeywordVO val) throws CalFacadeException;

  /** Return a keyword with the given id
   *
   * @param id     int id of the keyword
   * @return KeywordVo object representing the keyword in question
   *                     null if it doesn't exist.
   */
  public KeywordVO getKeyword(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Public Keywords
   *  =================================================================== */

  /** Return all public keywords
   *
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO[] getPublicKeywords() throws CalFacadeException;

  /** Return given public keyword
   *
   * @param id     int id of the keyword
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO getPublicKeyword(int id) throws CalFacadeException;

  /** Replace a keyword in the database.
   *
   * @param val   KeywordVO object to be replaced
   */
  public void replaceKeyword(KeywordVO val) throws CalFacadeException;

  /** Delete a keyword with the given id
   *
   * @param id       int id of the keyword
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteKeyword(int id) throws CalFacadeException;

  /** Check to see if a Keyword is referenced.
   *
   * @param id       int id of the Keyword
   * @return boolean true if the Keyword is referenced somewhere
   */
  public boolean checkKeywordRefs(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Keyword attributes
   *  =================================================================== */

  /** Get the keyword attributes entry for a keyword id
   *
   * @param keywordid       int id of the keyword entry
   * @return KeywordAttrsVO keyword attributes
   */
  public KeywordAttrsVO getKeywordAttrs(int keywordid)
      throws CalFacadeException;

  /** Delete keyword attrs entry
   *
   * @param id         int Id of the keyword attrs entry
   */
  public void deleteKeywordAttrs(int id) throws CalFacadeException;

  /** Add a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void addKeywordAttr(int keywordid,
                             String name,
                             String val) throws CalFacadeException;

  /** Remove a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void removeKeywordAttr(int keywordid,
                                String name,
                                String val) throws CalFacadeException;

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  /** Return all locations for this user
   *
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getLocations() throws CalFacadeException;

  /** Add a location to the database. The id will be set in the parameter
   * object.
   *
   * @param val   LocationVO object to be added
   * @return int  new location id
   */
  public int addLocation(LocationVO val) throws CalFacadeException;

  /** Replace a location in the database.
   *
   * @param val   LocationVO object to be replaced
   */
  public void replaceLocation(LocationVO val) throws CalFacadeException;

  /** Return a location with the given id
   *
   * @param id     int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getLocation(int id) throws CalFacadeException;

  /** Delete a location with the given id
   *
   * @param id       int id of the location
   * @return boolean false if it din't exist, true if it was deleted.
   */
  public boolean deleteLocation(int id) throws CalFacadeException;

  /** Check to see if a location is referenced.
   *
   * @param id       int id of the location
   * @return boolean true if the location is referenced somewhere
   */
  public boolean checkLocationRefs(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Public Locations
   *  =================================================================== */

  /** Return public locations - excluding reserved entries.
   *
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getPublicLocations() throws CalFacadeException;

  /** Return public locations - optionally including reserved entries.
   *
   * @param includeReserved boolean true include reserved, false exclude
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getPublicLocations(boolean includeReserved)
      throws CalFacadeException;

  /** Return given public location
   *
   * @param id          int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getPublicLocation(int id) throws CalFacadeException;

  /** Delete given public location
   *
   * @param id          int id of the location
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deletePublicLocation(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return all sponsors - excluding reserved entries.
   *
   * @return SponsorVO[]  array of sponsor value objects
   */
  public SponsorVO[] getSponsors() throws CalFacadeException;

  /** Return all sponsors - optionally including reserved entries.
   *
   * @param includeReserved true to include reserved entries.
   * @return SponsorVO[]    array of sponsor value objects
   */
  public SponsorVO[] getSponsors(boolean includeReserved)
      throws CalFacadeException;

  /** Return a sponsor with the given id
   *
   * @param id     int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getSponsor(int id) throws CalFacadeException;

  /** Add a sponsor to the database. The id will be set in the parameter
   * object.
   *
   * @param val   SponsorVO object to be added
   * @return int  new sponsor id
   */
  public int addSponsor(SponsorVO val) throws CalFacadeException;

  /** Replace a sponsor in the database.
   *
   * @param val   SponsorVO object to be replaced
   */
  public void replaceSponsor(SponsorVO val) throws CalFacadeException;

  /** Delete a sponsor with the given id
   *
   * @param id       int id of the sponsor
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteSponsor(int id) throws CalFacadeException;

  /** Check to see if a sponsor is referenced.
   *
   * @param id       int id of the sponsor
   * @return boolean true if the sponsor is referenced somewhere
   */
  public boolean checkSponsorRefs(int id) throws CalFacadeException;

  /** Return given public sponsor
   *
   * @param id         int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getPublicSponsor(int id) throws CalFacadeException;

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return a single event for the current user
   *
   * @param   eventId   int id of the event
   * @return  EventVO   value object representing event.
   */
  public EventVO getEvent(int eventId) throws CalFacadeException;

  /** Return the events for the given day as an array of value objects
   *
   * @param   date    MyCalendarVO object defining day
   * @return  EventVO[]  one days events or null for no events.
   */
  public EventVO[] getDaysEvents(MyCalendarVO date) throws CalFacadeException;

  /** Add an event to the database. The id will be set in the parameter
   * object.
   *
   * @param val   EventVO object to be added
   * @return int  new event id
   */
  public int addEvent(EventVO val) throws CalFacadeException;

  /** Replace an event wit the same id in the database.
   *
   * @param val   EventVO object to be replaced
   */
  public void replaceEvent(EventVO val) throws CalFacadeException;

  /** Delete an event
   *
   * @param eventid   int id of event to delete
   * @return boolean false if it didn't exist, true if it was deleted.
   * @exception CalFacadeException If there's a database access problem
   */
  public boolean deleteEvent(int eventid) throws CalFacadeException;

  /** ===================================================================
   *                   Public Events
   *  =================================================================== */

  /** Return the public events for the current user. This is an administrative
   * function for the public events admin client.
   *
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents() throws CalFacadeException;

  /** Return the public events for the current user within the given date
   * range. This is an administrative function for the public events admin
   * client.
   *
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents(MyCalendarVO startDate,
                                    MyCalendarVO endDate)
          throws CalFacadeException;

  /* * Return the public events for the given day as an array of value objects
   *
   * @param   date    MyCalendar object defining day
   * @return  EventVO[]  one days events or null for no events.
   * /
  public EventVO[] getDaysPublicEvents(MyCalendarVO date)
      throws CalFacadeException;*/

  /** ===================================================================
   *                   Event keywords
   *  =================================================================== */

  /** Given a eventid and keywordid add an entry to the event keyword table
   *
   * @param eventid   Id of event
   * @param keywordid Id of keyword
   * @exception CalFacadeException If there's a database access problem
   */
  public void addEventKeyword(int eventid, int keywordid)
      throws CalFacadeException;

  /** Given a eventid delete an entry from the event keyword table
   *
   * @param eventid   Id of event
   * @exception CalFacadeException If there's a database access problem
   */
  public void deleteEventKeyword(int eventid) throws CalFacadeException;

  /** ===================================================================
   *                   EventRefs
   *  =================================================================== */

  /** Create an event ref for the current user.
   *
   * @param   eventId   int id of the event to add
   * @return  boolean   false for no such item.
   */
  public boolean addEventRef(int eventId) throws CalFacadeException;

  /** Delete an event ref for the current user.
   *
   * @param   eventId   int id of the event to delete
   * @return  boolean   false for no such item.
   */
  public boolean deleteEventRef(int eventId) throws CalFacadeException;
}

