package edu.washington.cac.calfacade.shared;

import java.io.Serializable;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

/** A calendar in UWCal. This value object does no consistency or validity
 * checking. A calendar db object is a stored filter.
 *
 * This object is intended to allow applications to interact with the
 * calendar back end. It does not represent the internal stored structure of a
 * calendar object.
 *
 *   @author Mike Douglass douglm@rpi.edu
 *  @version 1.0
 */
public class CalendarVO implements Comparable, Serializable {
  private int id;

  /** The internal name of the calendar
   */
  private String name;

  /** A printable title for the calendar
   */
  private String title;

  /** Some sort of description - may be null
   */
  private String description;

  /** This flag indicates that this calendar should be displayed as an
   * 'important' calendar in some sense, that is we display it in the menu
   * of available calendars.
   */
  private boolean important;

  /** true if we display the children
   */
  boolean showChildren;

  /** The children of the filter */
  private SortedSet children;

  public CalendarVO() {
  }

  /** Create a calendar by specifying all its fields

      @param id           Unique ID for the keyword
      @param name         String internal name
      @param title        String printable title
      @param description  String long description
      @param showChildren Display sub-calendars if true
    */
  public CalendarVO(int id,
                    String name,
                    String title,
                    String description,
                    boolean important,
                    boolean showChildren) {
    this.id = id;
    this.name = name;
    this.title = title;
    this.description = description;
    this.important = important;
    this.showChildren = showChildren;
  }

  /**
    Set the id for this keyword
    @param val    int Keyword id
   */
  public void setId(int val) {
    id = val;
  }

  /** Get the kwywords unique id
   *
   * @return int    the keywords unique id
   */
  public int getId() {
    return id;
  }

  /** Set the name
   *
   * @param val    String name
   */
  public void setName(String val) {
    name = val;
  }

  /** Get the name
   *
   * @return String   name
   */
  public String getName() {
    return name;
  }

  /** Set the title
   *
   * @param val    String title
   */
  public void setTitle(String val) {
    title = val;
  }

  /** Get the title
   *
   * @return String   title
   */
  public String getTitle() {
    return title;
  }

  /** Set the description
   *
   * @param val    description
   */
  public void setDescription(String val) {
    description = val;
  }

  /** Get the description
   *
   *  @return String   description
   */
  public String getDescription() {
    return description;
  }

  /** Set the important flag
   *
   *  @param val    true if the calendar is important
   */
  public void setImportant(boolean val) {
    important = val;
  }

  /** Is the keyword public?
   *
   *  @return boolean    true if the keyword is public
   */
  public boolean getImportant() {
    return important;
  }

  /** Should the children of this calendar be displayed?
   *
   * @param val   boolean true if the children of this calendar are to be
   *                 displayed
   */
  public void setShowChildren(boolean val) {
    showChildren = val;
  }

  /** Should the children of this calendar be displayed?
   *
   * @return boolean  true if the children of this calendar are to be displayed
   */
  public boolean getShowChildren() {
    return showChildren;
  }

  /** Add a calendar to the set of children that make up this calendar
   * @param val   CalendarVO to add
   */
  public void addChild(CalendarVO val) {
    if (children == null) {
      children  = new TreeSet();
    }
    children.add(val);
  }

  /**  Set the set of children
   *
   * @param   val   SortedSet children for this filter
   */
  public void setChildren(SortedSet val) {
    children = val;
  }

  /**  Get the set of children
   *
   * @return SortedSet   Calendar children for this filter
   */
  public SortedSet getChildren() {
    return children;
  }

  /** Return the number of children that will be displayed
   *
   * @return int    number of visible children
   */
  public int getNumberVisibleChildren() {
    if (!showChildren || children == null) {
      return 0;
    }

    return children.size();
  }

  /** Return the number of descendents that will be displayed
   *
   * @return int    total number of all visible descendents
   */
  public int getTotalVisibleDescendents() {
    if (!showChildren || children == null) {
      return 0;
    }

    Iterator it = children.iterator();
    int total = 0;

    while (it.hasNext()) {
      total += ((CalendarVO)it.next()).getTotalVisibleDescendents();
    }

    return total;
  }

  /** Comapre this calendar and an object
   *
   * @param  o    object to compare.
   * @return int -1, 0, 1
   */
  public int	compareTo(Object o) {
    if (!(o instanceof CalendarVO)) {
      return -1;
    }

    CalendarVO that = (CalendarVO)o;

    if (id < that.id) {
      return -1;
    }

    if (id > that.id) {
      return 1;
    }

    return 0;
  }

  /** The id does not take part in these comparisons because we're presumably
   * doing this to see if two objects are the same without doing a db retrieval.
   * That is, one of the objects in a comparison may not have the id set.
   *
   * <p>This is relevant for hashcode as the equals method requires that
   * hashcode return equal values for both objects in a comparison that are
   * to be considered equal.
   *
   * <p>As we have both objects to hand in the equals method we can check to
   * see if both ids are non-zero and use them in the comparison.
   */
  public int hashCode() {
    int hc = 1;

    if (id != 0) {
      hc *= id;
    }

    if (name != null) {
      hc *= name.hashCode();
    }

    if (title != null) {
      hc *= title.hashCode();
    }

    if (description != null) {
      hc *= description.hashCode();
    }

    if (important) {
      hc *= 2;
    }

    if (showChildren) {
      hc *= 2;
    }

    return hc;
  }

  public boolean equals(Object obj) {
    if (!(obj instanceof CalendarVO)) {
      return false;
    }

    CalendarVO that = (CalendarVO)obj;

    if ((id != 0) && (that.id != 0)) {
      if (id != that.id) {
        return false;
      }
    }

    if (name == null) {
      if (that.name != null) {
        return false;
      }
    } else if (!name.equals(that.name)) {
      return false;
    }

    if (title == null) {
      if (that.title != null) {
        return false;
      }
    } else if (!title.equals(that.title)) {
      return false;
    }

    if (description == null) {
      if (that.description != null) {
        return false;
      }
    } else if (!description.equals(that.description)) {
      return false;
    }

    if (important != that.important) {
      return false;
    }

    if (showChildren != that.showChildren) {
      return false;
    }

    return true;
  }

  public String toString() {
    StringBuffer sb = new StringBuffer();

    sb.append("CalendarVO(");
    sb.append("name=");
    sb.append(String.valueOf(name));
    sb.append(", title=");
    sb.append(String.valueOf(title));
    sb.append(", description=");
    sb.append(String.valueOf(description));
    sb.append(", showChildren=");
    sb.append(String.valueOf(showChildren));
    sb.append(")");

    return sb.toString();
  }
}
