package edu.washington.cac.calfacade.shared;

/** An interface to handle calendar admin groups.
 *
 * <p>Groups may be stored in a site specific manner so the actual
 * implementation used is a build-time configuration option.
 *
 * <p>Groups do not provide any form of access privileges. That is determined
 * by the user login id.
 *
 * <p>If a user is a member of more than one group the admin client should
 * ask which group they want.
 *
 * <p>Events etc are owned by the group which has it's own owner id. It's
 * important that the group owner ids are distinct from user ids. This is
 * dealt with by a configurable prefix and suffix appended to all group ids.
 *
 * <p>If a user is not a member of any group they own the events.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @version 2.2
 */
public interface AdminGroups {
  /** ===================================================================
   *  The following return the state of the current user.
   *  =================================================================== */

  /** Set the current user group or null if the user is not (acting as) a
   * group member.
   *
   * @param name   String Name of the group.
   * @exception CalFacadeException If there's a problem
   */
  public void setAdminGroup(String name) throws CalFacadeException;

  /** Return a read only object representing the state of the current user
   * group. This will be null if the user is not in any group.
   *
   * @return AdminGroupVORO        object representing user state
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupROVO getAdminGroup() throws CalFacadeException;

  /** ===================================================================
   *  The following should not change the state of the current users
   *  group which is set and retrieved with the above methods.
   * The first group are available even if group maintenance is not.
   *  =================================================================== */

  /** Return an array of objects representing the groups this user is a
   * member of. This will be null if the user is not in any group.
   *
   * @return AdminGroupVO[]         user groups
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO[] getAdminGroups(String userid) throws CalFacadeException;

  /** Show whether user entries can be modified with this
   * class. Some sites may use other mechanisms.
   *
   * @return boolean    true if group maintenance is implemented.
   */
  public boolean getGroupMaintOK();

  /** ===================================================================
   *  The following are available if group maintenance is on.
   *  =================================================================== */

  /** Create an empty group.
   *
   * @param  name          String group name
   * @param  description   String group description
   * @param  groupOwnerid  String id of user who owns the group.
   * @param  ownerid       String group ownerid
   * @exception CalFacadeException If there's a problem
   */
  public void makeGroup(String name,
                        String description,
                        String groupOwnerid,
                        String ownerid) throws CalFacadeException;

  /** Find a group.
   *
   * @param  name          String group name
   * @return AdminGroupVO   grooup object
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO findGroup(String name) throws CalFacadeException;

  /** Find a group given the event owner.
   *
   * @param  id             String event owner id
   * @return AdminGroupVO   group object
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO findGroupByEventOwner(String id)
      throws CalFacadeException;

  /** Add a member to a group
   *
   * @param group    String group name
   * @param id       String member id
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void addMember(String group, String id) throws CalFacadeException;

  /** Remove a member from a group
   *
   * @param group    String group name
   * @param id       String member id
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void removeMember(String group, String id) throws CalFacadeException;

  /** Delete a group
   *
   * @param group    String group name
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void removeGroup(String group) throws CalFacadeException;

  /** Update a group
   *
   * @param group    AdminGroupVO object
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void updateGroup(AdminGroupVO group) throws CalFacadeException;

  /** Return a collection of all groups
   *
   * @return AdminGroupVO[]    groups.
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO[] getAll() throws CalFacadeException;
}
