package edu.washington.cac.calfacade.impl;

import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.EventRef;
import edu.washington.cac.calendar.data.EventsI;
import edu.washington.cac.calendar.data.InstanceRecurrence;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.MasterRecurrence;
import edu.washington.cac.calendar.data.NoRecurrence;
import edu.washington.cac.calendar.data.PersonalEvents;
import edu.washington.cac.calendar.data.Recurrence;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.filter.Calendar;
import edu.washington.cac.calendar.filter.Filter;
import edu.washington.cac.calfacade.shared.CalendarVO;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.EventRefVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.RecurrenceVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserVO;

import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

/** This class provides a number of static methods used to create value
 * objects from internal data objects.
 *
 * @author Mike Douglass    douglm@rpi.edu
 */
public class VOFactory {
  private VOFactory() {}   // No instantiation allowed here

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return a value object representing the given event
   *
   * @param  ev      event
   * @param  es      events cache
   * @param  curUser Stringcurrent user so we canset editable
   * @return EventVO event value object
   */
  public static EventVO getEventVO(Event ev, EventsI es, String curUser) {
    EventVO evvo = new EventVO(ev.getId(),
             ev.getShortdesc(),
             ev.getLongdesc(),
             null, // longdesc1,
             ev.getStartdate(),
             ev.getStarttime(),
             ev.getEnddate(),
             ev.getEndtime(),
             ev.isPublic(),
             ev.getLink(),
             getUserVO(ev.getCreator()),
             ev.getCost(),
             null,
             ev.getLocationid(),
             ev.getSponsorid(),
             ev.getSeq(),
             ev.getLastmod(),
             ev.getCreated());

    evvo.setRecurrence(getRecurrenceVO(ev.getRecurrence(), es, curUser));

    if (ev.getLocationid() > LocationVO.maxReservedId) {
      evvo.setLocation(getLocationVO(ev.getLocation(es)));
    }

    if (ev.getSponsorid() != Sponsor.NO_SPONSOR_ID) {
      evvo.setSponsor(getSponsorVO(ev.getSponsor(es)));
    }

    Iterator it = ev.getKeywords();
    Keywords ks = es.getKeywords();

    while (it.hasNext()) {
      Integer ki = (Integer)it.next();

      try {
        Keyword k = ks.getKeyword(ki.intValue());
        evvo.addKeyword(getKeywordVO(k));
      } catch (Throwable t) {
        t.printStackTrace();
      }
    }

    if (es instanceof PersonalEvents) {
      if (curUser != null) {
        evvo.setEditable(!evvo.isPublic() &&
                         curUser.equals(evvo.getCreator().getName()));
      }

      if (!evvo.getEditable()) {
        if (((PersonalEvents)es).explicitlyContains(ev)) {
          evvo.setKind(EventVO.kindAdded);
        } else {
          evvo.setKind(EventVO.kindUndeletable);
        }
      }
    }

    return evvo;
  }

  /** Return an array of value objects representing the given array
   * of events
   *
   * @param  evs     array of Event
   * @param  es      Event cache
   * @param  curUser Stringcurrent user so we canset editable
   * @return EventVO Event value object array
   */
  public static EventVO[] getEventVOs(Event[] evs, EventsI es,
                                      String curUser) {
    if ((evs == null) || (evs.length == 0)) {
      return null;
    }

    EventVO[] events = new EventVO[evs.length];

    for (int ei = 0; ei < evs.length; ei++) {
      events[ei] = getEventVO(evs[ei], es, curUser);
    }

    return events;
  }

  /** ===================================================================
   *                   Recurrences
   *  =================================================================== */

  /** Return a recurrence value object. If it's a master and it contains the
   * set of instances and the EventsI object is non-null, the resulting
   * object will also contain the set of instances.
   *
   * @param r      Recurrence object
   * @param es     EventsI object - must be non-null to get fully populated result
   * @param  curUser Stringcurrent user so we canset editable
   * @return RecurrenceVO   possibly popluated object.
   */
  public static RecurrenceVO getRecurrenceVO(Recurrence r, EventsI es,
                                             String curUser) {
    if (r instanceof NoRecurrence) {
      return null;
    }

    if (r instanceof InstanceRecurrence) {
      return new RecurrenceVO(false, true, r.getMasterId(), null);
    }

    if (!(r instanceof MasterRecurrence)) {
      return null;
    }

    RecurrenceVO rvo = new RecurrenceVO(true, true, r.getMasterId(), null);

    if (es != null) {
      Iterator it = ((MasterRecurrence)r).getInstances();
      while (it.hasNext()) {
        Event e = (Event)it.next();
        rvo.addInstance(getEventVO(e, es, curUser));
      }
    }

    return rvo;
  }

  /** ===================================================================
   *                   EventRefs
   *  =================================================================== */

  /** Return a value object representing the given event ref
   *
   * @param  er          EventRef object
   * @return EventRefVO  event ref value object
   */
  public static EventRefVO getEventRefVO(EventRef er) {
    EventRefVO ervo = new EventRefVO(er.getId(),
                                     getUserVO(er.getUser()),
                                     er.getPurpose());

    return ervo;
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return a value object representing the given keyword
   *
   * @param  k    Keyword
   * @return LocationVO location value object
   */
  public static KeywordVO getKeywordVO(Keyword k) {
    return new KeywordVO(k.getId(),
                  k.getWord(),
                  null,
                  k.getLongdesc(),
                  k.isPublic(),
                  getUserVO(k.getCreator()));
  }

  /** Return an array of value objects representing the given array
   * of keywords
   *
   * @param  k    array of Keyword
   * @return KeywordVO location value object array
   */
  public static KeywordVO[] getKeywordVOs(Keyword[] k) {
    if ((k == null) || (k.length == 0)) {
      return null;
    }

    KeywordVO[] ks = new KeywordVO[k.length];

    for (int ki = 0; ki < k.length; ki++) {
      ks[ki] = getKeywordVO(k[ki]);
    }

    return ks;
  }

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  /** Return a value object representing the given location
   *
   * @param  l    Location
   * @return LocationVO location value object
   */
  public static LocationVO getLocationVO(Location l) {
    return new LocationVO(l.getId(),
                  l.getAddress(),
                  l.getSubaddress(),
                  l.getLink(),
                  getUserVO(l.getCreator()),
                  l.isPublic());
  }

  /** Return an array of value objects representing the given array
   * of locations
   *
   * @param  l    array of Location
   * @return LocationVO location value object array
   */
  public static LocationVO[] getLocationVOs(Location[] l) {
    if ((l == null) || (l.length == 0)) {
      return null;
    }

    LocationVO[] ls = new LocationVO[l.length];

    for (int li = 0; li < l.length; li++) {
      ls[li] = getLocationVO(l[li]);
    }

    return ls;
  }

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return a value object representing the given sponsor
   *
   * @param  s    Sponsor
   * @return SponsorVO sponsor value object
   */
  public static SponsorVO getSponsorVO(Sponsor s) {
    return new SponsorVO(s.getId(),
                s.getName(),
                s.getPhone(),
                s.getEmail(),
                s.getLink(),
                getUserVO(s.getCreator()),
                s.isPublic());
  }

  /** Return an array of value objects representing the given array
   * of sponsors
   *
   * @param  s    array of Sponsor
   * @return SponsorVO sponsor value object array
   */
  public static SponsorVO[] getSponsorVOs(Sponsor[] s) {
    if ((s == null) || (s.length == 0)) {
      return null;
    }

    SponsorVO[] ss = new SponsorVO[s.length];

    for (int si = 0; si < s.length; si++) {
      ss[si] = getSponsorVO(s[si]);
    }

    return ss;
  }

  /** ===================================================================
   *                   Users
   *  =================================================================== */

  /** Return a user value object representing the given user
   *
   * @param  u       user
   * @return UserVO  user value object
   */
  public static UserVO getUserVO(User u) {
    return new UserVO(u.getNameDB(),
                      u.getName(),
                      u.getId(),
                      u.getRights(),
                      u.getPublicEvents());
  }

  /** ===================================================================
   *                   Calendars
   *  =================================================================== */

  /** Return a value object representing the given calendar
   * We do not process the children.
   *
   * @param  c    calendar
   * @return CalendarVO calendar value object
   */
  public static CalendarVO getCalendarVO(Calendar c) {
    Filter f = c.getFilter();

    return new CalendarVO(c.getId(),
                f.getInternalName(),
                f.getName(),
                null,      // description
                false,      // important
/*                f.showChildren(), */ true);     // creator
  }

  /** Returns a set of CalendarVO objects with all the children based on
   * the parameter set.
   */
  public static SortedSet getCalendarVOs(SortedSet cset) {
    if (cset == null) {
      return null;
    }

    SortedSet csetVO = new TreeSet();

    Iterator it = cset.iterator();

    while (it.hasNext()) {
      Calendar c = (Calendar)it.next();

      CalendarVO cvo = getCalendarVO(c);
      csetVO.add(cvo);

      cvo.setChildren(getCalendarVOs(c.getFilter().getChildren()));
    }

    return csetVO;
  }
}

