package edu.washington.cac.calfacade.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Enumeration;
import java.util.Vector;

import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.Tables;
import edu.washington.cac.calendar.db.WhereClauses;
import edu.washington.cac.calfacade.shared.AuthUserVO;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.UserAuth;
import edu.washington.cac.calfacade.shared.UserAuthRO;

/** Implementation of UserAuth that handles UW DB tables for authorisation.
 *
 * @author Leman Chung, Greg Barnes
 * @version 1.0
 */
public class UserAuthUWDbImpl implements UserAuth {
  /** Ideally this would trigger the debugging log in the underlying jdbc
   * implementation.
   */
  private boolean debug = false;

  /** Current user access
   */
  private int curUsertype = noPrivileges;

  private String curUser;

  private UserAuthRO userAuthRO;

  /** Used for interaction with db
   */
  private transient EventData ed;

  /** Values in the table defining a users capabilities.
   */

  private static final String authValSuperUser = "A";

  private static final String authValContentAdminUser = "C";

  private static final String authValPublicAlertUser = "S";

  private static final String authValPublicEventUser = "P";

  private static final String authValAlertUser = "a";

  private static final String GET_USER =
      "select * from users where username=?";

  private static final String ADD_USER =
      "insert into users (username,created) values (?,?)";

  /** SQL to delete an entry in the authorization table */
  private static final String DELETE_AUTH =
      "delete from " + Tables.AUTH + " where " +
      Fields.AUSERID.getInsertName() + "=?";

  /** SQL to add an entry to the authorization table */
  private static final String ADD_AUTH =
      "insert into " + Tables.AUTH + " (" +
      Fields.AUSERID.getInsertName() + ", " +
      Fields.AUSERTYPE.getInsertName() + ", " +
      Fields.ALASTNAME.getInsertName() + ", " +
      Fields.AFIRSTNAME.getInsertName() + ", " +
      Fields.APHONE.getInsertName() + ", " +
      Fields.AEMAIL.getInsertName() + ", " +
      Fields.ADEPARTMENT.getInsertName() + ") values (?, ?, ?, ?, ?, ?, ?)";

  private static final String GET_AUTH =
      "select " +
        Fields.AUSERID.getSelectName() + ", " +
        "username, " +
        Fields.AUSERTYPE.getSelectName() + ", " +
        Fields.ALASTNAME.getSelectName() + ", " +
        Fields.AFIRSTNAME.getSelectName() + ", " +
        Fields.APHONE.getSelectName() + ", " +
        Fields.AEMAIL.getSelectName() + ", " +
        Fields.ADEPARTMENT.getSelectName() +
      " from " +
        Tables.AUTH.getName() + ", " +
        Tables.USERS.getName() +
      " where " +
        Fields.AUSERID.getStmtName() + "=" +
        Fields.UUSER_ID.getStmtName() + " AND " +
        Fields.UUSERNAME.getStmtName() +
         " =?";

  /** For the time being at least, the authorised user preferences can go
   * in separate tables. Some can go in the authuser table later.
   */

  /** SQL to add an entry to the authorized user preferences table */
  private static final String ADD_AUTH_PREF =
      "insert into " + "authprefs" + " (" +
        "userid, " +
        "autoaddKeywords, " +
        "autoaddLocations, " +
        "autoaddSponsors) " +
      "values (?, ?, ?, ?)";

  /** SQL to update an entry in the authorized user preferences table */
  private static final String UPDATE_AUTH_PREF =
      "update " + "authprefs" + " set " +
        "autoaddKeywords=?, " +
        "autoaddLocations=?, " +
        "autoaddSponsors=? " +
      " where " +
        "userid=?";

  /** SQL to get an entry from the authorized user preferences table */
  private static final String GET_AUTH_PREF =
      "select " +
        "autoaddKeywords, " +
        "autoaddLocations, " +
        "autoaddSponsors " +
      " from authprefs" +
      " where " +
        "userid=?";

  /** SQL to add an entry to the authorized user preferred keyword table */
  private static final String ADD_AUTH_PREF_KEYWORD =
      "insert into " + "authprefKeywords" + " (" +
        "userid, " +
        "keywordid) " +
      "values (?, ?)";

  /** SQL to get the authorized user preferred keywords */
  private static final String GET_AUTH_PREF_KEYWORDS =
      "select " +
        "keywordid " +
      " from authprefKeywords" +
      " where " +
        "userid=?";

  /** SQL to delete a user preferred keyword */
  private static final String DELETE_AUTH_PREF_KEYWORD =
      "delete from authprefKeywords" +
      " where " +
        "userid=? and keywordid=?";

  /** SQL to delete all instances of a preferred keyword */
  private static final String DELETE_AUTH_PREF_KEYWORD_ALL =
      "delete from authprefKeywords" +
      " where " +
        "keywordid=?";

  /** SQL to add an entry to the authorized user preferred location table */
  private static final String ADD_AUTH_PREF_LOCATION =
      "insert into " + "authprefLocations" + " (" +
        "userid, " +
        "locationid) " +
      "values (?, ?)";

  /** SQL to get the authorized user preferred locations */
  private static final String GET_AUTH_PREF_LOCATIONS =
      "select " +
        "locationid " +
      " from authprefLocations" +
      " where " +
        "userid=?";

  /** SQL to delete a user preferred location */
  private static final String DELETE_AUTH_PREF_LOCATION =
      "delete from authprefLocations" +
      " where " +
        "userid=? and locationid=?";

  /** SQL to delete all instances of a preferred location */
  private static final String DELETE_AUTH_PREF_LOCATION_ALL =
      "delete from authprefLocations" +
      " where " +
        "locationid=?";

  /** SQL to add an entry to the authorized user preferred sponsor table */
  private static final String ADD_AUTH_PREF_SPONSOR =
      "insert into " + "authprefSponsors" + " (" +
        "userid, " +
        "sponsorid) " +
      "values (?, ?)";

  /** SQL to get the authorized user preferred sponsors */
  private static final String GET_AUTH_PREF_SPONSORS =
      "select " +
        "sponsorid " +
      " from authprefSponsors" +
      " where " +
        "userid=?";

  /** SQL to delete a user preferred sponsor */
  private static final String DELETE_AUTH_PREF_SPONSOR =
      "delete from authprefSponsors" +
      " where " +
        "userid=? and sponsorid=?";

  /** SQL to delete all instances of a preferred sponsor */
  private static final String DELETE_AUTH_PREF_SPONSOR_ALL =
      "delete from authprefSponsors" +
      " where " +
        "sponsorid=?";

  public UserAuthUWDbImpl() {
    this(false);
  }

  public UserAuthUWDbImpl(boolean debug) {
    super();
    this.debug = debug;
  }

  /** ===================================================================
   *  The following affect the state of the current user.
   *  =================================================================== */

  /** Initialise the implementing object. This method may be called repeatedly
   * with the same or different classes of object.
   *
   * <p>This is not all that well-defined. This area falls somewhere
   * between the back-end and the front-end, depending upon how a site
   * implements its authorisation.
   *
   * <p>To make this interface independent of any particular implementation
   * we make the parameter an Object. We will call this method early on in
   * the life of a web session giving it the incoming request. This allows
   * a role-driven implementation to determine what roles the user has. The
   * object may choose to adjust that users rights at each request or ignore
   * subsequent calls (though note that some containers recheck a users
   * rights periodically)
   *
   * <p>A database driven implementation is free to ignore the call
   * altogether.
   *
   * <p> To summarise: this method will be called with an HttpServletRequest
   * object.
   *
   * @param  val        Object
   * @exception CalFacadeException If there's a problem
   */
  public void initialise(String userid, Object val)
          throws CalFacadeException {
    curUser = userid;
    curUsertype = getUsertype(userid);
    userAuthRO = null;
  }

  /** Initialise the implementing object with an access level.
   *
   * <p>At UW the supplied access is ignored and the current access level set
   * from the database.
   *
   * @param  userid    user whose access we are setting
   * @param  val       int sum  of allowable access - ignored
   * @exception CalFacadeException If there's a problem
   */
  public void initialise(String userid, int val) throws CalFacadeException {
    curUser = userid;
    curUsertype = getUsertype(userid);
    userAuthRO = null;
  }

  /** ===================================================================
   *  The following return the state of the current user.
   *  =================================================================== */

  /** Return the current userid. If non-null this object has been
   * initialised otherwise one of the initialsie methods must be called.
   *
   * @return String          non-null for initialised object.
   * @exception CalFacadeException If there's a problem
   */
   public String getUserid() throws CalFacadeException {
     return curUser;
   }

  /** Return a read only object represeting the state of the current user.
   *
   * @return UserAuthRO        object representing user state
   * @exception CalFacadeException If there's a problem
   */
  public UserAuthRO getUserAuthRO() throws CalFacadeException {
    if (userAuthRO == null) {
      userAuthRO = new UserAuthRO(curUser, curUsertype);
    }

    return userAuthRO;
  }

  /** Return the type of the current user.
   *
   * @return int        user type as defined above,
   * @exception CalFacadeException If there's a problem
   */
  public int getUsertype() throws CalFacadeException {
    return curUsertype;
  }

  /** Check for priv user
   *
   * @return boolean    true for super user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isSuperUser() throws CalFacadeException {
    return (curUsertype & superUser) != 0;
  }

  /** Check for alert user
   *
   * @return boolean    true for alert user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isAlertUser() throws CalFacadeException {
    return (curUsertype & alertUser) != 0;
  }

  /** Check for public events owner user
   *
   * @return boolean    true for public events owner user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isOwnerUser() throws CalFacadeException {
    return (curUsertype & publicEventUser) != 0;
  }

  /** Check for content admin user
   *
   * @return boolean    true for content admin user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isContentAdminUser() throws CalFacadeException {
    return (curUsertype & contentAdminUser) != 0;
  }

  /** ===================================================================
   *  The following should not change the state of the current users
   *  access which is set and retrieved with the above methods.
   *  =================================================================== */

  /** Show whether user entries can be displayed or modified with this
   * class. Some sites may use other mechanisms.
   *
   * @return boolean    true if user maintenance is implemented.
   */
  public boolean getUserMaintOK() {
    return true;
  }

  /** Return the type of the user.
   *
   * @param  userid     String user id
   * @return int        user type as defined above,
   * @exception CalFacadeException If there's a problem
   */
  public int getUsertype(String userid) throws CalFacadeException {
    return getUser(userid).getUsertype();
  }

  /** Set the type of the user.
   *
   * @param  userid     String user id
   * @param  utype      int user type as defined above,
   * @exception CalFacadeException If there's a problem
   */
  public void setUsertype(String userid, int utype)
          throws CalFacadeException {
    throw new CalFacadeException("Unimplemented");
  }

  /** Adjust the user type to take account of site specific requirements.
   *
   * <p>For example, some sites may allow any user who can add public events
   * the authority to add alerts. Others may wish to keep those roles
   * separate.
   *
   * <p>This routine also should ensure all the super user privileges are
   * set.
   *
   * @param userType    int unadjusted user type.
   * @return int   adjusted user type.
   * @exception CalFacadeException   For invalid usertype values.
   */
  public int adjustUsertype(int userType) throws CalFacadeException {
    if ((userType & ~allAuth) != 0) {
      throw new CalFacadeException("Invalid authorised user type: " + userType);
    }

    if ((userType & superUser) != 0) {
      return allAuth;
    }

    /** We'll allow any combination of the remaining flags.
     */
    return userType;
  }

  /** Just see if the user has any special privileges.
   * Same as getUserType(userid) != noPrivileges;
   *
   * @param  userid     String user id
   * @return boolean true if the user has any special privileges
   * @exception CalFacadeException If there's a problem
   */
  public boolean isAuthorised(String userid) throws CalFacadeException {
    return getUser(userid).getUsertype() != noPrivileges;
  }

  /** Check for priv user
   * Same as getUserType(userid) == superUser;
   *
   * @param  userid     String user id
   * @return boolean    true for super user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isSuperUser(String userid) throws CalFacadeException {
    return getUser(userid).getUsertype() == superUser;
  }

  /** Remove any special authorisation for this user.
   *
   * <p>For this implementation, we remove them from the table.
   *
   * @param  val      AuthUserVO users entry
   * @exception CalFacadeException If there's a problem
   */
  public void removeAuth(AuthUserVO val) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(DELETE_AUTH);
      s.setInt(1, val.getUserid());
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close();
    }
  }

  /** Update the user entry
   *
   * @param  val      AuthUserVO users entry
   */
  public void updateUser(AuthUserVO val) throws CalFacadeException {
    AuthUserVO dbEntry = getUserEntry(val.getAccount());

    if (val.getUsertype() == noPrivileges) {
      // Delete it
      if (dbEntry != null) {
        removeAuth(val);
      }

      return;
    }

    if (dbEntry == null) {
      addAuth(val);
      return;
    }

    updateUserEntry(val);
  }

  /** Return the given user. Will always return an entry (except for
   * exceptional conditions.) An unauthorsed user will have a usertype of
   * noPrivileges.
   *
   * @param  userid        String user id
   * @return AuthUserVO    users entry
   * @exception CalFacadeException If there's a problem
   */
  public AuthUserVO getUser(String userid) throws CalFacadeException {
    AuthUserVO au = getUserEntry(userid);

    if (au != null) {
      return au;
    }

    au = new AuthUserVO(0, userid, noPrivileges, null, null, null,
                                     null, null);
//    addAuth(au);
    if (userid != null) {
      getPrefs(au);
    }

    return au;
  }

  /** Return a collection of all authorised users
   *
   * @return AuthUserVO[]    users with any special authorisation.
   * @exception CalFacadeException If there's a problem
   */
  public AuthUserVO[] getAll() throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;
    Vector v = new Vector();

    try {
      s = getEd().getPreparedStatement("select "
                                + Fields.AUSERID.getSelectName() + ", "
                                + "username, "
                                + Fields.AUSERTYPE.getSelectName() + ", "
                                + Fields.ALASTNAME.getSelectName() + ", "
                                + Fields.AFIRSTNAME.getSelectName() + ", "
                                + Fields.APHONE.getSelectName() + ", "
                                + Fields.AEMAIL.getSelectName() + ", "
                                + Fields.ADEPARTMENT.getSelectName()
                                + " from "
                                + Tables.AUTH.getName() + ", "
                                + Tables.USERS.getName() +
                                " where auth.userid=users.userid " +
                                " order by username");

      rs = s.executeQuery();

      while (rs.next()) {
        AuthUserVO au = makeAuthUserVO(rs);

        v.addElement(au);
      }

      int sz = v.size();

      if (sz == 0) {
        return null;
      }

      return (AuthUserVO[])v.toArray(new AuthUserVO[v.size()]);
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      getEd().close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                          Protected methods
   * ==================================================================== */

  /** Return the given user entry or null if no entry found.
   *
   * @param  userid        String user id
   * @return AuthUserVO    users entry or null.
   * @exception CalFacadeException If there's a problem
   */
  protected AuthUserVO getUserEntry(String userid) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getEd().getPreparedStatement(GET_AUTH);

      s.setString(1, userid);

      rs = s.executeQuery();

      if (!rs.next()) {
        return null;
      }

      AuthUserVO user = makeAuthUserVO(rs);
      getPrefs(user);

      return user;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      getEd().close(); // Will free s
      s = null;
    }
  }

  /** Update the given user entry.
   *
   * @param  val    AuthUserVO users entry
   * @exception CalFacadeException If there's a problem
   */
  protected void updateUserEntry(AuthUserVO val) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement("update " +
                                Tables.AUTH.getName() + " set "  +
                                Fields.AUSERTYPE.getInsertName() + "=?, " +
                                Fields.ALASTNAME.getInsertName() + "=?, " +
                                Fields.AFIRSTNAME.getInsertName() + "=?, " +
                                Fields.APHONE.getInsertName() + "=?, " +
                                Fields.AEMAIL.getInsertName() + "=?, " +
                                Fields.ADEPARTMENT.getInsertName() + "=?" +
                                " where " +
                                Fields.AUSERID.getStmtName() + " =?");

      s.setString(1, toDbtype(val.getUsertype()));
      s.setString(2, Fields.ALASTNAME.sizeToFit(val.getLastname()));
      s.setString(3, Fields.AFIRSTNAME.sizeToFit(val.getFirstname()));
      s.setString(4, Fields.APHONE.sizeToFit(val.getUphone()));
      s.setString(5, Fields.AEMAIL.sizeToFit(val.getUemail()));
      s.setString(6, Fields.ADEPARTMENT.sizeToFit(val.getDept()));
      s.setInt(7, val.getUserid());

      s.executeUpdate();

      updatePrefs(val);
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close(); // Will free s
      s = null;
    }
  }

  /** Add an entry to the authorization table
   *
   * @param  val            AuthUserVO object with at least the userid
   * @exception CalFacadeException If there's a problem
   */
  protected void addAuth(AuthUserVO val) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      /* make sure user entry exists */
      s = getEd().getPreparedStatement(GET_USER);
      s.setString(1, val.getAccount());

      rs = s.executeQuery();

      if (!rs.next()) {
        rs.close();
        getEd().close();

        s = getEd().getPreparedStatement(ADD_USER);
        s.setString(1, val.getAccount());
        s.setTimestamp(2, new Timestamp(System.currentTimeMillis()));

        s.executeUpdate();
        getEd().close();

        s = getEd().getPreparedStatement(GET_USER);
        s.setString(1, val.getAccount());

        rs = s.executeQuery();
        if (!rs.next()) {
          throw new CalFacadeException("Create of user " + val.getAccount() +
                                       " failed");
        }
      }

      val.setUserid(rs.getInt("userid"));

      s = getEd().getPreparedStatement(ADD_AUTH);

      s.setInt(1, val.getUserid());
      s.setString(2, toDbtype(val.getUsertype()));
      s.setString(3, Fields.ALASTNAME.sizeToFit(val.getLastname()));
      s.setString(4, Fields.AFIRSTNAME.sizeToFit(val.getFirstname()));
      s.setString(5, Fields.APHONE.sizeToFit(val.getUphone()));
      s.setString(6, Fields.AEMAIL.sizeToFit(val.getUemail()));
      s.setString(7, Fields.ADEPARTMENT.sizeToFit(val.getDept()));
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      getEd().close();
    }
  }

  protected AuthUserVO makeAuthUserVO(ResultSet rs) throws Throwable {
    String dbType = rs.getString(Fields.AUSERTYPE.getName());

    /** No dbtype value means a public events user
     */
    if (dbType == null) {
      dbType = "P";
    }

    return new AuthUserVO(rs.getInt(Fields.AUSERID.getName()),
                          rs.getString("username"),
                          adjustUsertype(toType(dbType)),
                          rs.getString(Fields.ALASTNAME.getName()),
                          rs.getString(Fields.AFIRSTNAME.getName()),
                          rs.getString(Fields.APHONE.getName()),
                          rs.getString(Fields.AEMAIL.getName()),
                          rs.getString(Fields.ADEPARTMENT.getName()));
  }

  /** Update the authorized user preferences. Does not update the
   * preferred keywords, locations or sponsors. They are updated as they
   * are modified.
   */
  protected void updatePrefs(AuthUserVO val) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(UPDATE_AUTH_PREF);

      s.setString(1, dbBool(val.getAutoAddKeywords()));
      s.setString(2, dbBool(val.getAutoAddLocations()));
      s.setString(3, dbBool(val.getAutoAddSponsors()));
      s.setInt(4, val.getUserid());

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close(); // Will free s
      s = null;
    }
  }

  /** Add the authorized user preferences. Does not update the
   * preferred keywords, locations or sponsors. They are updated as they
   * are modified.
   */
  protected void addPrefs(AuthUserVO val) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(ADD_AUTH_PREF);

      s.setInt(1, val.getUserid());
      s.setString(2, dbBool(val.getAutoAddKeywords()));
      s.setString(3, dbBool(val.getAutoAddLocations()));
      s.setString(4, dbBool(val.getAutoAddSponsors()));

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close(); // Will free s
      s = null;
    }
  }

  /** Get all the authorized user preferences.
   */
  protected void getPrefs(AuthUserVO user) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getEd().getPreparedStatement(GET_AUTH_PREF);

      s.setInt(1, user.getUserid());

      rs = s.executeQuery();

      if (!rs.next()) {
        user.setAutoAddKeywords(true);
        user.setAutoAddLocations(true);
        user.setAutoAddSponsors(true);

        addPrefs(user);

        return;
      }

      user.setAutoAddKeywords(boolDb(rs.getString("autoaddKeywords")));
      user.setAutoAddLocations(boolDb(rs.getString("autoaddLocations")));
      user.setAutoAddSponsors(boolDb(rs.getString("autoaddSponsors")));

      user.setPreferredKeywords(getKeys(user.getUserid(),
                                        GET_AUTH_PREF_KEYWORDS,
                                        "keywordid"));

      user.setPreferredLocations(getKeys(user.getUserid(),
                                         GET_AUTH_PREF_LOCATIONS,
                                         "locationid"));

      user.setPreferredSponsors(getKeys(user.getUserid(),
                                        GET_AUTH_PREF_SPONSORS,
                                        "sponsorid"));
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      getEd().close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                       Preferences methods
   * ==================================================================== */

  /** Set the value in the user record. If it's changed update the database.
   */
  public void setAutoAddKeywords(AuthUserVO user, boolean val)
      throws CalFacadeException {
    if (val == user.getAutoAddKeywords()) {
      return;
    }

    user.setAutoAddKeywords(val);
    updatePrefs(user);
  }

  /** Add a keyword key to the preferred keywords. Return true if it was
   * added, false if it was in the list.
   */
  public boolean addKeyword(AuthUserVO user, int key)
      throws CalFacadeException {
    if (!user.addKeyword(key)) {
      return false;
    }

    /** add it to the db table */
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(ADD_AUTH_PREF_KEYWORD);

      s.setInt(1, user.getUserid());
      s.setInt(2, key);

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close();
    }

    return true;
  }

  /** Remove a keyword preference for the given user if supplied or all users
   * if user is null.
   *
   * @param user   AuthUserVO entry for user or null for all users
   * @param key    keyword id in the entry to remove
   */
  public void removeKeyword(AuthUserVO user, int key)
      throws CalFacadeException {
    String sql;
    int id= 0;

    if (user != null) {
      id = user.getUserid();
    }

    if (id == 0) {
      sql = DELETE_AUTH_PREF_KEYWORD_ALL;
    } else {
      sql = DELETE_AUTH_PREF_KEYWORD;
    }

    removePref(id, key, sql, "keywordid");
  }

  /** Set the value in the user record. If it's changed update the database.
   */
  public void setAutoAddLocations(AuthUserVO user, boolean val)
      throws CalFacadeException {
    if (val == user.getAutoAddLocations()) {
      return;
    }

    user.setAutoAddLocations(val);
    updatePrefs(user);
  }

  /** Add a location key to the preferred locations. Return true if it was
   * added, false if it was in the list.
   */
  public boolean addLocation(AuthUserVO user, int key)
      throws CalFacadeException {
    if (!user.addLocation(key)) {
      return false;
    }

    /** add it to the db table */
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(ADD_AUTH_PREF_LOCATION);

      s.setInt(1, user.getUserid());
      s.setInt(2, key);

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close();
    }

    return true;
  }

  /** Remove a location preference for the given user if supplied or all users
   * if user is null.
   *
   * @param user   AuthUserVO entry for user or null for all users
   * @param key    location id in the entry to remove
   */
  public void removeLocation(AuthUserVO user, int key)
      throws CalFacadeException {
    String sql;
    int id= 0;

    if (user != null) {
      id = user.getUserid();
    }

    if (id == 0) {
      sql = DELETE_AUTH_PREF_LOCATION_ALL;
    } else {
      sql = DELETE_AUTH_PREF_LOCATION;
    }

    removePref(id, key, sql, "locationid");
  }

  /** Set the value in the user record. If it's changed update the database.
   */
  public void setAutoAddSponsors(AuthUserVO user, boolean val)
      throws CalFacadeException {
    if (val == user.getAutoAddSponsors()) {
      return;
    }

    user.setAutoAddSponsors(val);
    updatePrefs(user);
  }

  /** Add a sponsor key to the preferred sponsors. Return true if it was
   * added, false if it was in the list.
   */
  public boolean addSponsor(AuthUserVO user, int key)
      throws CalFacadeException {
    if (!user.addSponsor(key)) {
      return false;
    }

    /** add it to the db table */
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(ADD_AUTH_PREF_SPONSOR);

      s.setInt(1, user.getUserid());
      s.setInt(2, key);

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close();
    }

    return true;
  }

  /** Remove a sponsor preference for the given user if supplied or all users
   * if user is null.
   *
   * @param user   AuthUserVO entry for user or null for all users
   * @param key    sponsor id in the entry to remove
   */
  public void removeSponsor(AuthUserVO user, int key)
      throws CalFacadeException {
    String sql;
    int id= 0;

    if (user != null) {
      id = user.getUserid();
    }

    if (id == 0) {
      sql = DELETE_AUTH_PREF_SPONSOR_ALL;
    } else {
      sql = DELETE_AUTH_PREF_SPONSOR;
    }

    removePref(id, key, sql, "sponsorid");
  }

  /* ====================================================================
   *                        Private methods
   * ==================================================================== */

  /** Convert internal db type to external user type.
   */
  private int toType(String dbType) {
    if (dbType == null) {
      return noPrivileges;
    }

    if (dbType.equals("A")) {
      return superUser;
    }

    if (dbType.equals("C")) {
      return contentAdminUser;
    }

    if (dbType.equals("S")) {
      return alertUser + publicEventUser;
    }

    if (dbType.equals("P")) {
      return publicEventUser;
    }

    if (dbType.equals("a")) {
      return alertUser;
    }

    return noPrivileges;
  }

  /** Convert external user type to internal db type.
   *
   * @return String   internal type - null for no privileges.
   */
  private String toDbtype(int usertype) {
    if (usertype == noPrivileges) {
      return null;
    }

    if ((usertype & superUser) != 0) {
      return "A";
    }

    if (usertype == contentAdminUser) {
      return "C";
    }

    if (usertype == publicEventUser) {
      return "P";
    }

    if (usertype == alertUser) {
      return "a";
    }

    if (usertype == (alertUser + publicEventUser)) {
      // semi-priv
      return "S";
    }

    return null;
  }

  private String dbBool(boolean val) {
    if (val) {
      return "T";
    }

    return "F";
  }

  private boolean boolDb(String val) {
    return "T".equals(val);
  }

  /** Get the array of keys using the given statement (field 1 is the user)
   * and the given field name from the result.
   */
  private int[] getKeys(int id, String sql, String fldname)
    throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;
    Vector v = new Vector();

    try {
      s = getEd().getPreparedStatement(sql);
      s.setInt(1, id);
      rs = s.executeQuery();
      while (rs.next()) {
        v.addElement(new Integer(rs.getInt(fldname)));
      }

      int sz = v.size();

      if (sz == 0) {
        return null;
      }

      int[] keys = new int[sz];
      Enumeration enum = v.elements();
      int i = 0;

      while (enum.hasMoreElements()) {
        keys[i] = ((Integer)enum.nextElement()).intValue();
        i++;
      }

      return keys;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      getEd().close();
    }
  }

  /** Remove the entry for the given user id and given key. If id is 0
   * we are removing all entries for the given key.
   *
   * @param id       int id of user or 0
   * @param key      int key in entry
   * @param sql      String sql
   * @param fldname  String name of key field
   */
  private void removePref(int id, int key, String sql, String fldname)
    throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getEd().getPreparedStatement(sql);

      if (id != 0) {
        s.setInt(1, id);
        s.setInt(2, key);
      } else {
        s.setInt(1, key);
      }

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      getEd().close();
    }
  }


  private EventData getEd() {
    if (ed == null) {
      ed = new EventData();
    }
    return ed;
  }
}
