package edu.washington.cac.calfacade.impl;

import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.UserAuth;
import edu.washington.cac.calfacade.shared.UserAuthRO;

/** Implementation of UserAuth that uses roles and assumes servlet based
 * authentication. We use the database table to keep track of information
 * about the authorised user.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @author Leman Chung, Greg Barnes
 * @version 1.0
 */
public class UserAuthRolesImpl extends UserAuthUWDbImpl {
  /** Ideally this would trigger the debugging log in the underlying jdbc
   * implementation.
   */
  private boolean debug = false;

  /** Current user access
   */
  private int curUsertype = noPrivileges;

  private String curUser;

  private UserAuthRO userAuthRO;

  public UserAuthRolesImpl() {
    this(false);
  }

  public UserAuthRolesImpl(boolean debug) {
    super();
    this.debug = debug;
  }

  /** ===================================================================
   *  The following affect the state of the current user.
   *  =================================================================== */

  /** Initialise the implementing object. This method may be called repeatedly
   * with the same or different classes of object.
   *
   * <p>This is not all that well-defined. This area falls somewhere
   * between the back-end and the front-end, depending upon how a site
   * implements its authorisation.
   *
   * <p>To make this interface independent of any particular implementation
   * we make the parameter an Object. We will call this method early on in
   * the life of a web session giving it the incoming request. This allows
   * a role-driven implementation to determine what roles the user has. The
   * object may choose to adjust that users rights at each request or ignore
   * subsequent calls (though note that some containers recheck a users
   * rights periodically)
   *
   * <p>A database driven implementation is free to ignore the call
   * altogether.
   *
   * <p> To summarise: this method will be called with an HttpServletRequest
   * object.
   *
   * @param  val        Object
   * @exception CalFacadeException If there's a problem
   */
  public void initialise(String userid, Object val)
          throws CalFacadeException {
    /* We ignore this call for this object type.
     */
  }

  /** Initialise the implementing object with an access level.
   *
   * @param  userid    user whose access we are setting
   * @param  val       int sum  of allowable access
   * @exception CalFacadeException If there's a problem
   */
  public void initialise(String userid, int val) throws CalFacadeException {
    curUser = userid;
    curUsertype = val;
    userAuthRO = null;
  }

  /** ===================================================================
   *  The following return the state of the current user.
   *  =================================================================== */

  /** Return a read only object represeting the state of the current user.
   *
   * @return UserAuthRO        object representing user state
   * @exception CalFacadeException If there's a problem
   */
  public UserAuthRO getUserAuthRO() throws CalFacadeException {
    if (userAuthRO == null) {
      userAuthRO = new UserAuthRO(curUser, curUsertype);
    }

    return userAuthRO;
  }

  /** Return the type of the current user.
   *
   * @return int        user type as defined above,
   * @exception CalFacadeException If there's a problem
   */
  public int getUsertype() throws CalFacadeException {
    return curUsertype;
  }

  /** Check for priv user
   *
   * @return boolean    true for super user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isSuperUser() throws CalFacadeException {
    return (curUsertype & superUser) != 0;
  }

  /** Check for alert user
   *
   * @return boolean    true for alert user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isAlertUser() throws CalFacadeException {
    return (curUsertype & alertUser) != 0;
  }

  /** Check for public events owner user
   *
   * @return boolean    true for public events owner user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isOwnerUser() throws CalFacadeException {
    return (curUsertype & publicEventUser) != 0;
  }

  /** Check for content admin user
   *
   * @return boolean    true for content admin user
   * @exception CalFacadeException If there's a problem
   */
  public boolean isContentAdminUser() throws CalFacadeException {
    return (curUsertype & contentAdminUser) != 0;
  }

  /** ===================================================================
   *  The remainder are supplied by the superclass.
   *  =================================================================== */
}
