package edu.washington.cac.calfacade.impl;

import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.util.Vector;

import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.Recurrence;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.DBResources;
import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.PubEventsQueries;
import edu.washington.cac.calendar.db.Queries;
import edu.washington.cac.calendar.db.Tables;
import edu.washington.cac.calendar.db.WhereClauses;
import edu.washington.cac.calfacade.impl.CalFactory;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.KeywordAttrsVO;
import edu.washington.cac.calfacade.shared.UserVO;

/** PubEvents:
 *  Object to handle sql statments to add/update/delete events in db tables.
 *
 * <p>I have gathered togetehr the queries found in the pubevents modules
 * and intend combining them.
 *
 * <p>My intent is to simplify the process. The query rate is unlikely to
 * be high enough to stress the system so not getting all the columns
 * is probably an unneccessary complication.
 *
 * <p>Also, attempting to retrieve everything with a single query probably
 * causes portability problems.
 *
 * @author Leman Chung, Greg Barnes
 * @author Mike Douglass douglm@rpi.edu
 * @version 2.2
 */
public class PubEvents extends EventData {
  /** Ideally this would trigger the debugging log in the underlying jdbc
   * implementation.
   */
  private boolean debug = false;

  private Calintf cal;

  public PubEvents(Calintf cal) {
    this(cal, false);
  }

  public PubEvents(Calintf cal, boolean debug) {
    super();
    this.cal = cal;
    this.debug = debug;
  }

  /* ====================================================================
   *                   Touch - update lastmods table
   * ==================================================================== */

  /** Update lastmod timestamp.
   *
   * @exception CalFacadeException If there's a database access problem
   */
  public void touch() throws CalFacadeException {
    PreparedStatement s = null;

    try {
      String sql = "update lastmods set lastmod = ?" +
                      "where name =?;";

      s = getPreparedStatement(sql);
      s.setTimestamp(1,
                     new java.sql.Timestamp(new java.util.Date().getTime()));
      s.setString(2, "pubevents");

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /* ====================================================================
   *                            Sponsors
   * ==================================================================== */

  /** SQL to check that an event exists with a particular sponsor */
  private static final String SELECT_SPONSOR_FROM_EVENTS =
      "select " + Fields.ESPONSOR_ID.getSelectName() + " from "
      + Tables.EVENTS.getName() + " where "
      + Fields.ESPONSOR_ID.getStmtName() + "=?";

  /** SQL to check that a filter exists with a particular sponsor */
  private static final String SELECT_SPONSOR_FROM_CALENDARS =
      "select REF_NUM from "
      + Tables.CALENDARS.getName() + " where "
      + "TYPE='S' AND REF_NUM=?";

  /** See if the given sponsor id is referenced
   *
   *  @param  sponsorid   int sponsor id we are looking for
   *  @return boolean     true if found
   *  @exception CalFacadeException If there's a database access problem
   */
  public boolean sponsorReffed(int sponsorid) throws CalFacadeException {
      PreparedStatement s = null;
      ResultSet rs = null;

    try {
      s = getPreparedStatement(SELECT_SPONSOR_FROM_EVENTS);
      s.setInt(1, sponsorid);
      rs = s.executeQuery();

      if (rs.next()) {
        return true;
      }

      rs.close();
      close();

      s = getPreparedStatement(SELECT_SPONSOR_FROM_CALENDARS);
      s.setInt(1, sponsorid);
      rs = s.executeQuery();

      return rs.next();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                            Sponsors
   * ==================================================================== */

  /** SQL to check whether a location is still used by any event */
  private static final String SELECT_LOCATION_FROM_EVENTS =
      "select " + Fields.ELOCATION_ID.getSelectName() + " from "
      + Tables.EVENTS.getName() + " where "
      + Fields.ELOCATION_ID.getStmtName() + "=?";

  /** SQL to check that a filter exists with a particular sponsor */
  private static final String SELECT_LOCATION_FROM_CALENDARS =
      "select REF_NUM from "
      + Tables.CALENDARS.getName() + " where "
      + "TYPE='L' AND REF_NUM=?";

  /** See if the given location id exists in events
   *
   *  @param  locationid   int location id we are looking for
   *  @return boolean     true if found
   *  @exception CalFacadeException If there's a database access problem
   */
  public boolean locationReffed(int locationid) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(SELECT_LOCATION_FROM_EVENTS);
      s.setInt(1, locationid);
      rs = s.executeQuery();

      if (rs.next()) {
        return true;
      }

      rs.close();
      close();

      s = getPreparedStatement(SELECT_LOCATION_FROM_CALENDARS);
      s.setInt(1, locationid);
      rs = s.executeQuery();

      return rs.next();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                            Events
   * ==================================================================== */

  /** This should come from PubEventsQueries I guess. This needs to have
   * all the fields in an EventVO class.
   */
  private static final String eventQuerySelectFields =
                          Fields.EEVENT_ID.getSelectName() + ", "
                        + Fields.ESTARTDATE.getSelectName() + ", "
                        + Fields.ESTARTTIME.getSelectName() + ", "
                        + Fields.EENDDATE.getSelectName() + ", "
                        + Fields.EENDTIME.getSelectName() + ", "
                        + Fields.ESHORTDESC.getSelectName() + ", "
                        + Fields.ELONGDESC.getSelectName() + ", "
                            + Fields.ELONGDESC1.getSelectName() + ", "
                         + Fields.EPUBLIC.getSelectName() + ", "
                         + Fields.ELINK.getSelectName() + ", "
                         + Fields.ECOST.getSelectName() + ", "
                         + Fields.ESPONSOR_ID.getSelectName() + ", "
                         + Fields.ELOCATION_ID.getSelectName() + ", "
                         + Fields.ESEQ.getSelectName() + ", "
                         + Fields.ECREATOR.getSelectName() + ", "
                         + Fields.ERECURRING_STATUS.getSelectName() + ", "
                            + Fields.ELASTMOD.getSelectName() + ", "
                            + Fields.ECREATED.getSelectName();

  private static final String populateEventQueryPrefix =
                            "select " + eventQuerySelectFields
                            + " from "
                            + Tables.EVENTS.getName() + " where " +
                            Fields.EPUBLIC.getStmtName() + "='T' and " +
                            Fields.ECREATOR.getStmtName() + "=? ";

  private static final String populateEventQueryCheckRangeStartDate =
                            " and ( " +
                            Fields.ESTARTDATE.getStmtName() + ">=? or " +
                            Fields.EENDDATE.getStmtName() + ">=? ) ";

  private static final String populateEventQueryCheckRangeEndDate =
                            " and ( " +
                            Fields.ESTARTDATE.getStmtName() + "<=? or " +
                            Fields.EENDDATE.getStmtName() + "<=? ) ";

  private static final String populateEventQueryCheckRangeStartEndDate =
                            " and ( " +
                            "( " +
                            Fields.ESTARTDATE.getStmtName() + ">=? AND " +
                            Fields.ESTARTDATE.getStmtName() + "<=? ) or ( " +
                            Fields.EENDDATE.getStmtName() + ">=? ) and " +
                            Fields.EENDDATE.getStmtName() + "<=? ) ) ";

  private static final String populateEventStartdateOrder =
                            " order by " +
                            Fields.ESTARTDATE.getStmtName();

  private static final String getEventKeywordIdsSql =
                 "select " +
                     Fields.EKKEYWORD_ID.getSelectName() +
                  " from " +
                     Tables.EVENTKEYWORDS.getName() +
                  " where " +
                     Fields.EKEVENT_ID.getStmtName() + "=?";

  /** Return an array of public events for the given creator, optionally
   *  within the given date range.
   *
   * <p>Either or both of the date range may be null.
   *
   * @param creator       if non-null must match
   * @param rangeStart    Date - optional start date
   * @param rangeEnd      Date - optional end date
   * @return Event[]      event objects
   * @exception CalFacadeException If there's a database access problem
   */
  public Event[] findPublicEvents(String creator,
                                  Date rangeStart,
                                  Date rangeEnd) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      StringBuffer sql = new StringBuffer(populateEventQueryPrefix);

      if ((rangeStart != null) && (rangeEnd == null)) {
        sql.append(populateEventQueryCheckRangeStartDate);
      } else if ((rangeStart == null) && (rangeEnd != null)) {
        sql.append(populateEventQueryCheckRangeEndDate);
      } else if ((rangeStart != null) && (rangeEnd != null)) {
        sql.append(populateEventQueryCheckRangeStartEndDate);
      }

      sql.append(populateEventStartdateOrder);

      if (debug) {
        System.out.println("About to execute query:");
        System.out.println(sql.toString());
      }

      s = getPreparedStatement(sql.toString());

      s.setString(1, creator);

      if ((rangeStart != null) && (rangeEnd == null)) {
        s.setDate(2, rangeStart);
        s.setDate(3, rangeStart);
      } else if ((rangeStart == null) && (rangeEnd != null)) {
        s.setDate(2, rangeEnd);
        s.setDate(3, rangeEnd);
      } else if ((rangeStart != null) && (rangeEnd != null)) {
        s.setDate(2, rangeStart);
        s.setDate(3, rangeStart);
        s.setDate(4, rangeEnd);
        s.setDate(5, rangeEnd);
      }

      rs = s.executeQuery();
      Vector v = new Vector();

      while (rs.next()) {
        Event ev = new Event(rs.getInt(Fields.EEVENT_ID.getName()),
            rs.getString(Fields.ESHORTDESC.getName()),
            rs.getString(Fields.ELONGDESC.getName()),
            // rs.getString(Fields.ELONGDESC1.getName()),
            rs.getDate(Fields.ESTARTDATE.getName()),
            rs.getTime(Fields.ESTARTTIME.getName()),
            rs.getDate(Fields.EENDDATE.getName()),
            rs.getTime(Fields.EENDTIME.getName()),
            rs.getString(Fields.EPUBLIC.getName()).equals(WhereClauses.TRUE),
            rs.getString(Fields.ELINK.getName()),
            new User(rs.getString(Fields.ECREATOR.getName())),
            rs.getString(Fields.ECOST.getName()),
            Recurrence.create(rs.getString(Fields.ERECURRING_STATUS.getName()),
                              rs.getInt(Fields.EEVENT_ID.getName())),
            rs.getInt(Fields.ELOCATION_ID.getName()),
            rs.getInt(Fields.ESPONSOR_ID.getName()),
            rs.getInt(Fields.ESEQ.getName()),
            rs.getTimestamp(Fields.ELASTMOD.getName()),
            rs.getTimestamp(Fields.ECREATED.getName()));

        v.addElement(ev);
      }

      close();

      int sz = v.size();

      if (sz == 0) {
        return null;
      }

      Event[] evs = (Event[])v.toArray(new Event[v.size()]);

      /* Now populate with keyword ids
       */
      for (int i = 0; i < evs.length; i++) {
        Event ev = evs[i];

        getEventKeywordIds(ev);
      }

      return evs;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close();
    }
  }

  /** Add keyword ids to event
   *
   * @param   ev        the event
   */
  private void getEventKeywordIds(Event ev)
    throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(getEventKeywordIdsSql);
      s.setInt(1, ev.getId());
      rs = s.executeQuery();
      while (rs.next()) {
        ev.addKeyword(rs.getInt(Fields.EKKEYWORD_ID.getName()));
      }
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close();
    }
  }

  /** SQL to delete an event */
  private static final String DELETE_EVENTS =
      "delete from " + Tables.EVENTS + " where " +
      Fields.EEVENT_ID.getInsertName() + "=?";

  /** Delete an event
   *
   * @param eventid   int id of event to delete
   * @exception CalFacadeException If there's a database access problem
   */
  public void deleteEvent(int eventid) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(DELETE_EVENTS);
      s.setInt(1, eventid);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /* ====================================================================
   *                            Marked Events
   * This apparently is on its way out.
   * ==================================================================== */

  /** Query to see if event is marked.
  */
  private static final String checkMarked =
      "select " +
         Fields.MEVENT_ID.getSelectName() +
      " from " +
         Tables.MARKEDEVENTS.getName() +
      " where " +
         Fields.MEVENT_ID.getSelectName() + "=?";

  /** Determine if an event exists in the marked table
   *
   *  @param   eventid    ID of the event
   *  @return  boolean    True if it's there
   *  @exception CalFacadeException If there's a database access problem
   */
  public boolean isMarkedEvent(int eventid) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(checkMarked);
      s.setInt(1, eventid);
      rs = s.executeQuery();

      return rs.next();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                       Keyword attributes
   * ==================================================================== */

  /** Get the keyword attributes entry for a keyword id
   *
   * @param keywordid       int id of the keyword entry
   * @return KeywordAttrsVO keyword attributes
   */
  public KeywordAttrsVO getKeywordAttrs(int keywordid)
      throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement("select " +
                               "kakeywordid, " +
                               "kaname, " +
                               "kaval " +
                               "from keyword_attrs where " +
                               "kakeywordid =?");

      s.setInt(1, keywordid);
      rs = s.executeQuery();

      KeywordAttrsVO ka = new KeywordAttrsVO(0, keywordid, null);

      while (rs.next()) {
        ka.addAttrVal(rs.getString("kaname"),
                      rs.getString("kaval"));
      }

      return ka;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close();
    }
  }

  /** Delete all keyword attrs entries for a keyword
   *
   * @param keywordid         int Id of the keyword entry
   */
  public void deleteKeywordAttrs(int keywordid) throws CalFacadeException {
    String sql = "delete from " +
                 "keyword_attrs " +
                 " where " +
                 "kakeywordid" + "=?";

    try {
      PreparedStatement s = getPreparedStatement(sql);
      s.setInt(1, keywordid);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Add a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void addKeywordAttr(int keywordid,
                             String name,
                             String val) throws CalFacadeException {
    String findsql = "select kaid from " +
                        "keyword_attrs " +
                        " where " +
                        "kakeywordid" + "=?" + " and " +
                        "kaname" + "=?" + " and " +
                        "kaval" + "=?";

    String insertsql = "insert into keyword_attrs " +
                       "(kakeywordid, kaname, kaval) " +
                        " values " +
                        "(?, ?, ?)";

    ResultSet rs = null;

    try {
      PreparedStatement s = getPreparedStatement(findsql);
      s.setInt(1, keywordid);
      s.setString(2, name);
      s.setString(3, val);

      rs = s.executeQuery();
      if (rs.next()) {
        // Already exists
        return;
      }

      close();

      s = getPreparedStatement(insertsql);
      s.setInt(1, keywordid);
      s.setString(2, name);
      s.setString(3, val);

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close();
    }
  }

  /** Remove a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void removeKeywordAttr(int keywordid,
                                String name,
                                String val) throws CalFacadeException {
    String deletesql = "delete from " +
                        "keyword_attrs " +
                        " where " +
                        "kakeywordid" + "=?" + " and " +
                        "kaname" + "=?" + " and " +
                        "kaval" + "=?";
    try {
      PreparedStatement s = getPreparedStatement(deletesql);
      s.setInt(1, keywordid);
      s.setString(2, name);
      s.setString(3, val);

      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /* ====================================================================
   *                            Keywords
   * ==================================================================== */

  /** SQL to check whether a keyword is still used by any event */
  private static final String SELECT_KEYWORD_FROM_EVENTS =
      "select " + Fields.EKKEYWORD_ID.getSelectName() + " from "
      + Tables.EVENTKEYWORDS.getName() + " where "
      + Fields.EKKEYWORD_ID.getStmtName() + "=?";

  /** SQL to check that a filter exists with a particular keyword */
  private static final String SELECT_KEYWORD_FROM_CALENDARS =
      "select REF_NUM from "
      + Tables.CALENDARS.getName() + " where "
      + "TYPE='K' AND REF_NUM=?";

  /** See if the given keyword id exists in events
   *
   *  @param  id          int keyword id we are looking for
   *  @return boolean     true if found
   *  @exception CalFacadeException If there's a database access problem
   */
  public boolean keywordReffed(int id) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(SELECT_KEYWORD_FROM_EVENTS);
      s.setInt(1, id);
      rs = s.executeQuery();

      if (rs.next()) {
        return true;
      }

      rs.close();
      close();

      s = getPreparedStatement(SELECT_KEYWORD_FROM_CALENDARS);
      s.setInt(1, id);
      rs = s.executeQuery();

      return rs.next();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /* ====================================================================
   *                       Event keywords
   * ==================================================================== */

  /** SQL to add an entry into the event_keywords table */
  private static final String ADD_EVENT_KEYWORDS =
      "insert into " + Tables.EVENTKEYWORDS + " (" +
      Fields.EKEVENT_ID.getInsertName() + ", " +
      Fields.EKKEYWORD_ID.getInsertName() + ") values (?, ?);";

  /** Given a eventid and keywordid add an entry to the event keyword table
   *
   * @param eventid   Id of event
   * @param keywordid Id of keyword
   * @exception CalFacadeException If there's a database access problem
   */
  public void addEventKeyword(int eventid, int keywordid)
      throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(ADD_EVENT_KEYWORDS);
      s.setInt(1, eventid);
      s.setInt(2, keywordid);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** SQL to delete an entry in the event_keywords table */
  private static final String DELETE_EK =
      "delete from " + Tables.EVENTKEYWORDS + " where " +
      Fields.EKEVENT_ID.getInsertName() + "=?";

  /** Given a eventid delete an entry from the event keyword table
   *
   * @param eventid   Id of event
   * @exception CalFacadeException If there's a database access problem
   */
  public void deleteEventKeyword(int eventid) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(DELETE_EK);
      s.setInt(1, eventid);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /* ====================================================================
   *                       Misc
   * ==================================================================== */

  /** Print the queries used in the class to standard output
   *
   * @param args    String[] ignored
   */
  public static void main(String[] args) {
    System.out.println(SELECT_SPONSOR_FROM_EVENTS);
    System.out.println(SELECT_LOCATION_FROM_EVENTS);
    System.out.println(DELETE_EVENTS);
    System.out.println(ADD_EVENT_KEYWORDS);
    System.out.println(DELETE_EK);
  }
}
