package edu.washington.cac.calfacade.impl;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.DuplicateItemException;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.EventRef;
import edu.washington.cac.calendar.data.Events;
import edu.washington.cac.calendar.data.EventsI;
import edu.washington.cac.calendar.data.ItemException;
import edu.washington.cac.calendar.data.ItemAccessException;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.Locations;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PersonalEvents;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.Sponsors;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.filter.Calendar;
import edu.washington.cac.calendar.filter.Filter;
import edu.washington.cac.calendar.filter.FilteredEvents;
import edu.washington.cac.calendar.filter.LastmodGEFilter;
import edu.washington.cac.calendar.filter.PublicCalendars;
import edu.washington.cac.calendar.filter.SearchFilter;
import edu.washington.cac.calendar.filter.SelectAllFilter;
import edu.washington.cac.calenv.CalEnv;
import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.CalendarVO;
import edu.washington.cac.calfacade.shared.CalFacadeAccessException;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.Calintf;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.EventRefVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.KeywordAttrsVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserAuth;
import edu.washington.cac.calfacade.shared.UserVO;

import java.sql.Date;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;
import org.apache.log4j.Logger;

/** All sql exceptions should be trapped by the real calendar api and
 * converted to calendar exceptions.
 */
import java.sql.SQLException;
import java.sql.Timestamp;

/** This class acts as an interface to the calendar.
 *
 * @author Mike Douglass   douglm@rpi.edu
 */
public class CalintfImpl implements Calintf {
  private boolean debug;

  /** User for whom we maintain this facade
   */
  private User user;

  /** And the value object
   */
  private UserVO userVO;

  /** The user authorisation object
   */
  private UserAuth userAuth;

  /** The administrative groups object.
   */
  private AdminGroups adminGroups;

  /** True if we are administrators
   */
  private boolean publicAdmin;

  private EventsI eventsCache;

  /** Current filter for public calendars
   */
  private Filter curFilter;

  /** The default filter for a guest */
  private static Filter defaultFilter = new SelectAllFilter();

  private SortedSet calendars;

  /** Current calendar for public calendars - null if no calendar (filter) set
   */
  private CalendarVO curCalendar;

  /** Current search string - null if no search (filter) set
   */
  private String curSearch;

  /** We cache keyword attrs here for the moment
   */
  private HashMap keyAttrs = new HashMap();

  private transient Logger log;

  /** ===================================================================
   *                   initialisation
   *  =================================================================== */

  /** Must be called to initialise the new object.
   *
   * @param user        String user of the application
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param isGuest     boolean true if user is guest
   * @param publicAdmin boolean true if this is a public events admin app
   * @param debug       boolean true to turn on debugging trace
   */
  public void init(String user,
                   int rights,
                   boolean isGuest,
                   boolean publicAdmin,
                   boolean debug) throws CalFacadeException {
    this.debug = debug;

    if (debug) {
      getLogger().debug("CalIntfImpl.init - user=" + user +
                        " publicAdmin=" + publicAdmin);
    }

    if (isGuest) {
      setUser(null, rights, publicAdmin);
    } else {
      setUser(user, rights, publicAdmin);
    }

    this.publicAdmin = publicAdmin;
  }

  /** ===================================================================
   *                   Misc methods
   *  =================================================================== */

  /** Set this up for the given user. Must be called before any other action
   *
   * @param val         String user id - null for guest
   * @param rights      int access rights value as defined by
   *                    edu.washington.cac.calfacade.shared.UserAuth
   * @param publicAdmin boolean true if this is a public events admin app
   */
  public void setUser(String val, int rights,
                      boolean publicAdmin) throws CalFacadeException {
    refreshEvents();

    if (val == null) {
      user = new User();
    } else {
      user = new User(val);
      user.setRights(rights);
      user.setPublicEvents(publicAdmin);

      try {
        user.logon();
      } catch (Exception e) {
        throw new CalFacadeException("Unable to set user " + val + ": " +
                            e.getMessage());
      }
    }

    if (debug) {
      log.debug("User " + user + " set in calintf");
    }

    userVO = VOFactory.getUserVO(user);
  }

  /** Returns a value object representing the current user.
   *
   * @return UserVO       representing the current user
   */
  public UserVO getUserVO() throws CalFacadeException {
    return userVO;
  }

  /** Get a UserAuth object which allows the application to determine what
   * special rights the user has.
   *
   * <p>Note that the returned object may require a one time initialisation
   * call to set it up correctly.
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, int)
   * @see edu.washington.cac.calfacade.shared.UserAuth#initialise(String, Object)
   *
   * @return UserAuth    implementation.
   */
  public UserAuth getUserAuth() throws CalFacadeException {
    if (userAuth != null) {
      return userAuth;
    }

    try {
      String userAuthClassName = CalEnv.getEnv().getUserAuthClassName();

      if (userAuthClassName == null) {
        throw new CalFacadeException("No property defined in calenv for " +
                            "getUserAuthClassName()");
      }

      Object o = Class.forName(userAuthClassName).newInstance();

      if (o == null) {
        throw new CalFacadeException("Class " + userAuthClassName +
                            " not found");
      }

      if (!(o instanceof UserAuth)) {
        throw new CalFacadeException("Class defined in calenv for " +
                            "getUserAuthClassName() is not a subclass of " +
                            UserAuth.class.getName());
      }

      userAuth = (UserAuth)o;
    } catch (CalFacadeException ce) {
      throw ce;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    }

    return userAuth;
  }

  /** Get an AdminGroups object which allows the application to handle
   * administrative groups.
   *
   * @return AdminGroups    implementation.
   */
  public AdminGroups getAdminGroups() throws CalFacadeException {
    if (adminGroups != null) {
      return adminGroups;
    }

    try {
      String adminGroupsClassName = CalEnv.getEnv().getAdminGroupsClassName();

      if (adminGroupsClassName == null) {
        throw new CalFacadeException("No property defined in calenv for " +
                            "getAdminGroupsClassName()");
      }

      Object o = Class.forName(adminGroupsClassName).newInstance();

      if (o == null) {
        throw new CalFacadeException("Class " + adminGroupsClassName +
                            " not found");
      }

      if (!(o instanceof AdminGroups)) {
        throw new CalFacadeException("Class defined in calenv for " +
                        "getAdminGroupsClassName() is not a subclass of " +
                        AdminGroups.class.getName());
      }

      adminGroups = (AdminGroups)o;
    } catch (CalFacadeException ce) {
      throw ce;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    }

    return adminGroups;
  }

  /** Refresh the users events cache
   */
  public void refreshEvents() {
    eventsCache = null;
    keyAttrs = new HashMap();
  }

  /** Update lastmod timestamp.
   *
   * @exception CalFacadeException If there's a database access problem
   */
  public void touch() throws CalFacadeException {
    EventsI es = getEventsI();

    if ((es != null) && (es instanceof PublicEvents)) {
      ((PublicEvents)es).forceFlush();
    }
    getPubEvents().touch();
    refreshEvents();
  }

  /** Get the value of the current public events timestamp.
   *
   * @return long       current timestamp value
   */
  public long getPublicLastmod() {
    return PublicEvents.getPublicEvents().getLastmod();
  }

  /** ===================================================================
   *                   Calendars and search
   *  =================================================================== */

  /** Return a list of calendar we can present to the client as a list of
   * available calendars.
   *
   * @return SortedSet   of CalendarVO
   */
  public SortedSet getCalendars() {
    if (calendars == null) {
      SortedSet cals = PublicCalendars.mainCalendars();
      calendars = VOFactory.getCalendarVOs(cals);
    }

    return calendars;
  }

  /** Set a search filter using the suppplied search string
   *
   * @param val    String search parameters
   */
  public void setSearch(String val) {
    if (val == null) {
      curCalendar = null;
      curSearch = null;
      curFilter = defaultFilter;
    } else {
      curCalendar = null;
      curSearch = val;
      curFilter = new SearchFilter(val);
    }

    /* We can either do a setFilter if es is a FilteredEvents object
       or just drop the events object forcing a refresh
     */
    eventsCache = null; // force refresh
  }

  /** Return the current search string
   *
   * @return  String     search parameters
   */
  public String getSearch() {
    return curSearch;
  }

  /** Set the calendar we are interested in. This is represented by the name
   * of a calendar.
   *
   * @param  val     int id of calendar - 0 for none
   * @return boolean true for OK, false for unknown calendar
   */
  public boolean setCalendar(int val) throws CalFacadeException {
    if (val == 0) {
      curCalendar = null;
      curSearch = null;
      curFilter = defaultFilter;

      return true;
    }

    /** Try to locate the calendar
     */
    try {
      Calendar c = PublicCalendars.getPublicCalendars().getCalendar(val);
      curCalendar = VOFactory.getCalendarVO(c);
      curSearch = null;
      curFilter = c.getFilter();
    } catch (NoSuchItemException nsie) {
      curFilter = null;
    }

    if (curFilter == null) {
      curCalendar = null;
      curFilter = defaultFilter;

      return false;
    }

    /* We can either do a setFilter if es is a FilteredEvents object
       or just drop the events object forcing a refresh
     */
    eventsCache = null; // force refresh
    return true;
  }

  /** Get the name of the current calendar we are displaying
   *
   * @return String    name of calendar or null for all events
   */
  public String getCalendarTitle() {
    if (curCalendar == null) {
      return null;
    }

    return curCalendar.getTitle();
  }

  /** This method is called to add a calendar. No consistency checks are
   * carried out at the moment.
   *
   * @param val  String values for db
   */
  public void addCalendarDef(String val) throws CalFacadeException {
    try {
      new Caldata().addCalendar(val);
    } catch (SQLException se) {
      handleSQLException(se);
    }
  }

  /** Set a filter which returns events starting after a given Timestamp.
   *
   * @param val   Timestamp object.
   */
  public void setLastmodGECalendar(Timestamp val) throws CalFacadeException {
    curFilter = new LastmodGEFilter(val);
    curSearch = curFilter.getName();

    /* We can either do a setFilter if es is a FilteredEvents object
       or just drop the events object forcing a refresh
     */
    eventsCache = null; // force refresh
  }

  /** Set a filter which returns events starting after a given date of the
   * form yyyymmdd.
   *
   * @param date   String of form yyyymmdd.
   */
  public void setLastmodGEDateCalendar(String date) throws CalFacadeException {
    if ((date == null) || (date.length() != 8)) {
      throw new CalFacadeException("Invalid lastmod date " + date);
    }

    String tsdate = date.substring(0, 4) + "-" +
                    date.substring(4, 6) + "-" +
                    date.substring(6) +
                    " 00:00:00.000000000";
    setLastmodGECalendar(Timestamp.valueOf(tsdate));
  }

  /** Set a filter which returns events starting after a given date and time
   * of the form yyyymmddhhmm.
   *
   * @param dateTime   String of form yyyymmddhhmm.
   */
  public void setLastmodGEDateTimeCalendar(String dateTime)
      throws CalFacadeException {
    if ((dateTime == null) || (dateTime.length() != 12)) {
      throw new CalFacadeException("Invalid lastmod date " + dateTime);
    }

    String tsdate = dateTime.substring(0, 4) + "-" +
                    dateTime.substring(4, 6) + "-" +
                    dateTime.substring(6, 8) + " " +
                    dateTime.substring(8, 10) + ":" +
                    dateTime.substring(10) +
                    ":00.000000000";
    setLastmodGECalendar(Timestamp.valueOf(tsdate));
  }

  /** ===================================================================
   *                   Subscriptions
   *  =================================================================== */

  /** Ensure that the users subscriptions match a set of calendars
   *
   * @param crs      TreeSet of Integer calendar ids
   * @return int[]   Number of calendars subscribed to, and number of
   *                 calendars unsubscribed to
   */
  public int[] fixSubscriptions(TreeSet crs) throws CalFacadeException {
    EventsI es = getEventsI();

    if (!(es instanceof PersonalEvents)) {
      throw new CalFacadeException("Not a personal events calendar");
    }

    try {
      return ((PersonalEvents)es).fixSubscriptions(crs);
    } catch (ItemAccessException iae) {
      throw new CalFacadeException(iae);
    } catch (SQLException se) {
      handleSQLException(se);
      return null;
    }
  }

  /** Determine if user is subscribed to a calendar
   *
   * @param id       int calendar id
   * @return boolean true if subscribed to calendars
   */
  public boolean getSubscribed(int id) throws CalFacadeException {
    EventsI es = getEventsI();

    if (!(es instanceof PersonalEvents)) {
      throw new CalFacadeException("Not a personal events calendar");
    }

    try {
      return ((PersonalEvents)es).containsSubscription(id);
    } catch (NoSuchItemException nsie) {
      return false;
    }
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return all keywords for this user. For a public admin client this is
   * the same as getPublicKeywords().
   *
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO[] getKeywords() throws CalFacadeException {
    EventsI es = getEventsI();

    Keywords ks = es.getKeywords();
    Iterator it = ks.sortedElements();

    Vector v = new Vector();

    while (it.hasNext()) {
      v.addElement(it.next());
    }

    return VOFactory.getKeywordVOs(
       (Keyword[])v.toArray(new Keyword[v.size()]));
  }

  /** Add a Keyword to the database. The id will be set in the parameter
   * object.
   *
   * @param val   KeywordVO object to be added
   * @return int  new Keyword id
   */
  public int addKeyword(KeywordVO val) throws CalFacadeException {
//    EventsI es = getEventsI();
    Keyword k = null;

    requireSuper();

    if (val.getCreator() == null) {
      val.setCreator(userVO);
    }

    try {
      k = CalFactory.getKeyword(val);
//      k.setCaldata(new Caldata());
//      es.getKeywords().add(k);
      k.add(k.getCreator());

    } catch (SQLException se) {
      handleSQLException(se);
    }

    int id = k.getId();
    val.setId(id);

    return id;
  }

  /** Find a Keyword with the given id
   *
   * @param id     int id of the Keyword
   * @return KeywordVO object representing the location in question
   *                     null if it doesn't exist.
   */
  public KeywordVO getKeyword(int id) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Keyword k = es.getKeywords().getKeyword(id);
      return VOFactory.getKeywordVO(k);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Check to see if a Keyword is referenced.
   *
   * @param id       int id of the Keyword
   * @return boolean true if the Keyword is referenced somewhere
   */
  public boolean checkKeywordRefs(int id) throws CalFacadeException {
    return getPubEvents().keywordReffed(id);
  }

  /** ===================================================================
   *                   Public Keywords
   *  =================================================================== */

  /** Return all public keywords
   *
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO[] getPublicKeywords() throws CalFacadeException {
    //Keywords ks = PublicEvents.getPublicKeywords(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Keywords ks = PublicEvents.getPublicKeywords(/*user*/);
    Iterator it = ks.sortedElements();

    Vector v = new Vector();

    while (it.hasNext()) {
      v.addElement(it.next());
    }

    return VOFactory.getKeywordVOs(
       (Keyword[])v.toArray(new Keyword[v.size()]));
  }

  /** Return given public keyword
   *
   * @param id     int id of the keyword
   * @return KeywordVO[]  array of keywords value objects
   */
  public KeywordVO getPublicKeyword(int id) throws CalFacadeException {
    //Keywords ks = PublicEvents.getPublicKeywords(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Keywords ks = PublicEvents.getPublicKeywords(/*user*/);

    try {
      Keyword k = ks.getKeyword(id);
      return VOFactory.getKeywordVO(k);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Replace a keyword in the database.
   *
   * @param val   KeywordVO object to be replaced
   */
  public void replaceKeyword(KeywordVO val) throws CalFacadeException {
//    EventsI es = getEventsI();
    requireSuper();

    try {
      Keyword k = CalFactory.getKeyword(val);

      k.replace(k.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
    }
  }

  /** Delete a keyword with the given id
   *
   * @param id       int id of the keyword
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteKeyword(int id) throws CalFacadeException {
    requireSuper();

    EventsI es = getEventsI();

    try {
      Keyword k = es.getKeywords().getKeyword(id);
      k.delete(k.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
      return false;  // never reached
    } catch (NoSuchItemException nsie) {
      return false;
    }

    return true;
  }

  /** ===================================================================
   *                   Keyword attributes
   *  =================================================================== */

  /** Get the keyword attributes entry for a keyword id
   * For the moment we'll cache them here
   *
   * @param keywordid       int id of the keyword entry
   * @return KeywordAttrsVO keyword attributes
   */
  public KeywordAttrsVO getKeywordAttrs(int keywordid)
      throws CalFacadeException {
    Integer ikey = new Integer(keywordid);

    KeywordAttrsVO ka = (KeywordAttrsVO)keyAttrs.get(ikey);

    if (ka != null) {
      return ka;
    }

    ka = getPubEvents().getKeywordAttrs(keywordid);
    keyAttrs.put(ikey, ka);

    return ka;
  }

  /** Delete all keyword attrs entries for a keyword
   *
   * @param keywordid         int Id of the keyword entry
   */
  public void deleteKeywordAttrs(int keywordid) throws CalFacadeException {
    requireSuper();

    getPubEvents().deleteKeywordAttrs(keywordid);
  }

  /** Add a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void addKeywordAttr(int keywordid,
                             String name,
                             String val) throws CalFacadeException {
    requireSuper();
    getPubEvents().addKeywordAttr(keywordid, name, val);
  }

  /** Remove a keyword attribute name/value pair for a keyword
   *
   * @param keywordid       int id of the keyword entry
   * @param name            String attribute name
   * @param val             String attribute value
   */
  public void removeKeywordAttr(int keywordid,
                                String name,
                                String val) throws CalFacadeException {
    requireSuper();
    getPubEvents().removeKeywordAttr(keywordid, name, val);
  }

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  /** Return all locations for this user
   *
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getLocations() throws CalFacadeException {
    EventsI es = getEventsI();

    Locations ls = es.getLocations();
    Iterator it = ls.sortedElements();

    /* It appears that the public locations get dragged in as well -
        the inclusion of filtered events seems to include all public events.
        Remove them here
     */
    Vector v = new Vector();

    while (it.hasNext()) {
      Location l = (Location)it.next();

      if (!l.isPublic() || (l.getId() < Location.DELETED_LOCATION_ID)) {
        v.addElement(l);
      }
    }

    return VOFactory.getLocationVOs(
       (Location[])v.toArray(new Location[v.size()]));
  }

  /** Add a location to the database. The id will be set in the parameter
   * object.
   *
   * @param val   LocationVO object to be added
   * @return int  new location id
   */
  public int addLocation(LocationVO val) throws CalFacadeException {
    EventsI es = getEventsI();
    Location l = null;

    if (val.getCreator() == null) {
      val.setCreator(userVO);
    }

    try {
      l = CalFactory.getLocation(val);
//      l.setCaldata(new Caldata());
      es.getLocations().add(l);
    } catch (ItemAccessException iae) {
      throw new CalFacadeException(iae);
    } catch (SQLException se) {
      handleSQLException(se);
    }

    int id = l.getId();
    val.setId(id);

    return id;
  }

  /** Replace a location in the database.
   *
   * @param val   LocationVO object to be replaced
   */
  public void replaceLocation(LocationVO val) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Location l = CalFactory.getLocation(val);

      l.replace(l.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
    }
  }

  /** Find a location with the given id
   *
   * @param id     int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getLocation(int id) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Location l = es.getLocations().getLocation(id);
      return VOFactory.getLocationVO(l);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Delete a location with the given id
   *
   * @param id       int id of the location
   * @return boolean false if it didn't exist, or cannot be deleted,
   *                 true if it was deleted.
   */
  public boolean deleteLocation(int id) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Locations ls = es.getLocations();
      Location l = ls.getLocation(id);
      if (l.isPublic()) {
        return false;
      }
      ls.delete(l);
    } catch (SQLException se) {
      handleSQLException(se);
    } catch (ItemException ie) {
      return false;
    }

    return true;
  }

  /** Check to see if a location is referenced.
   *
   * @param id       int id of the location
   * @return boolean true if the location is referenced somewhere
   */
  public boolean checkLocationRefs(int id) throws CalFacadeException {
    return getPubEvents().locationReffed(id);
  }

  /** ===================================================================
   *                   Public Locations
   *  =================================================================== */

  /** Return public locations - excluding reserved entries.
   *
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getPublicLocations() throws CalFacadeException {
    return getPublicLocations(false);
  }

  /** Return public locations - optionally including reserved entries.
   *
   * @param includeReserved boolean true include reserved, false exclude
   * @return LocationVO[]  array of location value objects
   */
  public LocationVO[] getPublicLocations(boolean includeReserved)
      throws CalFacadeException {
    //Locations ls = PublicEvents.getPublicLocations(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Locations ls = PublicEvents.getPublicLocations(/*user*/);
    Iterator it = ls.sortedElements();

    Vector v = new Vector();

    while (it.hasNext()) {
      Location l = (Location)it.next();

      if (includeReserved || (l.getId() > LocationVO.maxReservedId)) {
        v.addElement(l);
      }
    }

    return VOFactory.getLocationVOs(
       (Location[])v.toArray(new Location[v.size()]));
  }

  /** Return given public location
   *
   * @param id          int id of the location
   * @return LocationVo object representing the location in question
   *                     null if it doesn't exist.
   */
  public LocationVO getPublicLocation(int id) throws CalFacadeException {
    //Locations ls = PublicEvents.getPublicLocations(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Locations ls = PublicEvents.getPublicLocations(/*user*/);

    try {
      Location l = ls.getLocation(id);
      return VOFactory.getLocationVO(l);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Delete given public location
   *
   * @param id          int id of the location
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deletePublicLocation(int id) throws CalFacadeException {
    //Locations ls = PublicEvents.getPublicLocations(user);
    // Until access control works in back end
    checkPublicAccess(writeAccess);
    Locations ls = PublicEvents.getPublicLocations(/*user*/);

    try {
      Location l = ls.getLocation(id);
      l.delete(l.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
    } catch (NoSuchItemException nsie) {
      return false;
    }

    return true;
  }

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return all sponsors - excluding reserved entries.
   *
   * @return SponsorVO[]  array of sponsor value objects
   */
  public SponsorVO[] getSponsors() throws CalFacadeException {
    return getSponsors(false);
  }

  /** Return all sponsors - optionally including reserved entries.
   *
   * @param includeReserved boolean true include reserved, false exclude
   * @return SponsorVO[]  array of sponsor value objects
   */
  public SponsorVO[] getSponsors(boolean includeReserved)
      throws CalFacadeException {
    //Sponsors ss = PublicEvents.getPublicSponsors(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Sponsors ss = PublicEvents.getPublicSponsors(/*user*/);
    Iterator it = ss.sortedElements();

    Vector v = new Vector();

    while (it.hasNext()) {
      Sponsor sp = (Sponsor)it.next();

      if (includeReserved || (sp.getId() > SponsorVO.maxReservedId)) {
        v.addElement(sp);
      }
    }

    return VOFactory.getSponsorVOs(
       (Sponsor[])v.toArray(new Sponsor[v.size()]));
  }

  /** Find a sponsor with the given id
   *
   * @param id     int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getSponsor(int id) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Sponsor s = es.getSponsors().getSponsor(id);
      return VOFactory.getSponsorVO(s);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Add a sponsor to the database. The id will be set in the parameter
   * object. Only for public events.
   *
   * @param val   SponsorVO object to be added
   * @return int  new sponsor id
   */
  public int addSponsor(SponsorVO val) throws CalFacadeException {
    checkPublicAccess(writeAccess);
    EventsI es = getEventsI();
    Sponsor s = null;

    if (val.getCreator() == null) {
      val.setCreator(userVO);
    }

    try {
      s = CalFactory.getSponsor(val);
//      s.setCaldata(new Caldata());
      es.getSponsors().add(s);
    } catch (ItemAccessException iae) {
      throw new CalFacadeException(iae);
    } catch (SQLException se) {
      handleSQLException(se);
    }

    int id = s.getId();
    val.setId(id);

    return id;
  }

  /** Replace a sponsor in the database.
   *
   * @param val   SponsorVO object to be replaced
   */
  public void replaceSponsor(SponsorVO val) throws CalFacadeException {
    checkPublicAccess(writeAccess);
    EventsI es = getEventsI();

    try {
      Sponsor s = CalFactory.getSponsor(val);

      s.replace(s.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
    }
  }

  /** Delete a sponsor with the given id
   *
   * @param id       int id of the sponsor
   * @return boolean false if it didn't exist, true if it was deleted.
   */
  public boolean deleteSponsor(int id) throws CalFacadeException {
    checkPublicAccess(writeAccess);
    EventsI es = getEventsI();

    try {
      Sponsor s = es.getSponsors().getSponsor(id);
      s.delete(s.getCreator());
    } catch (SQLException se) {
      handleSQLException(se);
    } catch (NoSuchItemException nsie) {
      return false;
    }

    return true;
  }

  /** Check to see if a sponsor is referenced.
   *
   * @param id       int id of the sponsor
   * @return boolean true if the sponsor is referenced somewhere
   */
  public boolean checkSponsorRefs(int id) throws CalFacadeException {
    return getPubEvents().sponsorReffed(id);
  }

  /** Return given public sponsor
   *
   * @param id         int id of the sponsor
   * @return SponsorVo object representing the sponsor in question
   *                     null if it doesn't exist.
   */
  public SponsorVO getPublicSponsor(int id) throws CalFacadeException {
    //Sponsors ss = PublicEvents.getPublicSponsors(user);
    // Until access control works in back end
    checkPublicAccess(readAccess);
    Sponsors ss = PublicEvents.getPublicSponsors(/*user*/);

    try {
      Sponsor s = ss.getSponsor(id);
      return VOFactory.getSponsorVO(s);
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return a single event for the current user
   *
   * @param   eventId   int id of the event
   * @return  EventVO   value object representing event.
   */
  public EventVO getEvent(int eventId) throws CalFacadeException {
    if (eventId < 0) {
      throw new CalFacadeException("Bad eventid " + eventId);
    }

    EventsI es = getEventsI();

    try {
      Event e = es.getEvent(eventId, true);

      checkAccess(readAccess, e);

      return VOFactory.getEventVO(e, es, userVO.getName());
    } catch (NoSuchItemException nsie) {
      return null;
    }
  }

  /** Return the events for the given day as an array of value objects
   *
   * @param   date    MyCalendar object defining day
   * @return  EventVO[]  one days events or null for no events.
   */
  public EventVO[] getDaysEvents(MyCalendarVO date) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Event[] evs = es.oneDaysEvents(CalFactory.getMyCalendar(date));
      EventVO[] evos = VOFactory.getEventVOs(evs, es, userVO.getName());

      return evos;
    } catch (ItemException iae) {
      handleUnserializableException(iae);
    } catch (SQLException se) {
      handleSQLException(se);
    }

    return null;
  }

  /** Add an event to the database. The id will be set in the parameter
   * object.
   *
   * @param val   EventVO object to be added
   * @return int  new event id
     @exception CalFacadeException If there's a db problem or problem with
         the event
   */
  public int addEvent(EventVO val) throws CalFacadeException {
    try {
      EventsI es = getEventsI();

      requireNotGuest();

      if (val.getCreator() == null) {
        val.setCreator(userVO);
      }

      if ((val.getLocationid() == 0) ||
          (val.getLocationid() == EventVO.DefaultLocationId)) {
        if (val.getLocation() != null) {
          val.setLocationid(val.getLocation().getId());
        } else {
          val.setLocationid(EventVO.DefaultLocationId);
        }
      }

      if ((val.getSponsorid() == 0) ||
          (val.getSponsorid() == EventVO.DefaultSponsorId)) {
        if (val.getSponsor() != null) {
          val.setSponsorid(val.getSponsor().getId());
        } else {
          val.setSponsorid(EventVO.DefaultSponsorId);
        }
      }

      Event e = CalFactory.getEvent(val);

      /* We can't use PublicEvents yet to create events - the user gets set
         to null by the calls to getPublicKeywords etc. However, we can't give
         them a user either as that stops personal calendars working.
       */
      e.add(e.getCreator());
//      es.add(e);

      int id = e.getId();
      val.setId(id);

      return id;
    } catch (SQLException se) {
      handleSQLException(se);
      return 0;	// never reached
//    } catch (ItemException ie) {
//      handleUnserializableException(ie);
//      return 0;	// never reached
    }
  }

  /** Replace an event with the same id in the database.
   *
   * @param val   EventVO object to be replaced
     @exception CalFacadeException If there's a db problem or problem with
         the event
   */
  public void replaceEvent(EventVO val) throws CalFacadeException {
    EventsI es = getEventsI();

    try {
      Event e = CalFactory.getEvent(val);

      checkAccess(writeAccess, e);

      es.replace(e);
      refreshEvents();
    } catch (ItemException ie) {
      handleUnserializableException(ie);
    } catch (SQLException se) {
      handleSQLException(se);
    }
  }

  /** Delete an event
   *
   * @param eventid   int id of event to delete
   * @return boolean false if it didn't exist, true if it was deleted.
   * @exception CalFacadeException If there's a database access problem
   */
  public boolean deleteEvent(int eventid) throws CalFacadeException {
    try {
      EventsI es = getEventsI();
      Event e = es.getEvent(eventid, true);

      if (e == null) {
        return false;
      }

      checkAccess(writeAccess, e);
      es.delete(e);
    } catch (ItemAccessException e) {
      handleUnserializableException(e);
    } catch (ItemException e) {
      return false;
    } catch (SQLException se) {
      handleSQLException(se);
    }

    return true;
  }

  /** ===================================================================
   *                   Public Events
   *  =================================================================== */

  /** Return the public events for the current user. This is an administrative
   * function for the public events admin client.
   *
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents() throws CalFacadeException {
    //EventsI es = PublicEvents.getPublicEvents(user);
    // Until access control works in back end
    if ((!publicAdmin) || (user.getName() == null)) {
      throw new CalFacadeAccessException();
    }

    EventsI es = PublicEvents.getPublicEvents();

    Event[] evs = getPubEvents().findPublicEvents(user.getName(), null, null);
    EventVO[] evos = VOFactory.getEventVOs(evs, es, userVO.getName());

    return evos;
  }

  /** Return the public events for the current user within the given date
   * range. This is an administrative function for the public events admin
   * client.
   *
   * @param   startDate  MyCalendarVO optional start date
   * @param   endDate    MyCalendarVO optional end date
   * @return  EventVO[]  events or null for no events.
   */
  public EventVO[] findPublicEvents(MyCalendarVO startDate,
                                    MyCalendarVO endDate)
          throws CalFacadeException {
    //EventsI es = PublicEvents.getPublicEvents(user);
    // Until access control works in back end
    if ((!publicAdmin) || (user.getName() == null)) {
      throw new CalFacadeAccessException();
    }

    EventsI es = PublicEvents.getPublicEvents();

    Date sdt = null;
    Date edt = null;

    if (startDate != null) {
      sdt = new Date(startDate.getTime().getTime());
    }

    if (endDate != null) {
      edt = new Date(endDate.getTime().getTime());
    }

    Event[] evs = getPubEvents().findPublicEvents(user.getName(), sdt, edt);
    EventVO[] evos = VOFactory.getEventVOs(evs, es, userVO.getName());

    return evos;
  }

  /* * Return the public events for the given day as an array of value objects
   *
   * @param   date    MyCalendar object defining day
   * @return  EventVO[]  one days events or null for no events.
   * /
  public EventVO[] getDaysPublicEvents(MyCalendarVO date)
      throws CalFacadeException {
    //EventsI es = PublicEvents.getPublicEvents(user);
    // Until access control works in back end
    EventsI es = PublicEvents.getPublicEvents();

    try {
      Event[] evs = es.oneDaysEvents(CalFactory.getMyCalendar(date));
      EventVO[] evos = VOFactory.getEventVOs(evs, es);

      return evos;
    } catch (ItemException iae) {
      handleUnserializableException(iae);
    } catch (SQLException se) {
      handleSQLException(se);
    }

    return null;
  }*/

  /** ===================================================================
   *                   Event keywords
   *  =================================================================== */

  /** Given a eventid and keywordid add an entry to the event keyword table
   *
   * @param eventid   Id of event
   * @param keywordid Id of keyword
   * @exception CalFacadeException If there's a database access problem
   */
  public void addEventKeyword(int eventid, int keywordid)
      throws CalFacadeException {
    getPubEvents().addEventKeyword(eventid, keywordid);
  }

  /** Given a eventid delete an entry from the event keyword table
   *
   * @param eventid   Id of event
   * @exception CalFacadeException If there's a database access problem
   */
  public void deleteEventKeyword(int eventid) throws CalFacadeException {
    getPubEvents().deleteEventKeyword(eventid);
  }

  /** ===================================================================
   *                   EventRefs
   *  =================================================================== */

  /** Create an event ref for the current user.
   *
   * @param   eventId   int id of the event to add
   * @return  boolean   false for no such item.
   */
  public boolean addEventRef(int eventId) throws CalFacadeException {
    EventsI es = getEventsI();
    if (!(es instanceof PersonalEvents)) {
      // Watch for attempts to fool public client
      return false;
    }

    EventRefVO ervo = new EventRefVO(eventId, userVO);

    EventRef er = CalFactory.getEventRef(ervo);

    try {
      PublicEvents pe = PublicEvents.getPublicEvents(user);

      Event ev = pe.getEvent(eventId, true);
      ((PersonalEvents)es).addRef(er, new MyCalendar(ev.getStartdate()));
    } catch (ItemAccessException iae) {
      throw new CalFacadeException(iae);
    } catch (NoSuchItemException e) {
      return false;
    } catch (CaldataException cde) {
      handleUnserializableException(cde);
    } catch (SQLException se) {
      if (isConstraintException(se)) {
        // seems likely this was caused by duplicating an item
        throw new CalFacadeException(se.getMessage());
      }
      handleSQLException(se);
    }

    return true;
  }

  /** Delete an event ref for the current user.
   *
   * @param   eventId   int id of the event to delete
   * @return  boolean   false for no such item.
   */
  public boolean deleteEventRef(int eventId) throws CalFacadeException {
    EventsI es = getEventsI();

    EventRefVO ervo = new EventRefVO(eventId, userVO);

    EventRef er = CalFactory.getEventRef(ervo);

    try {
      ((PersonalEvents)es).deleteRef(er);
    } catch (SQLException se) {
      handleSQLException(se);
      return false;
    } catch (NoSuchItemException e) {
      return false;
    } catch (Throwable t) {
      throw new CalFacadeException(t.getMessage());
    }

    return true;
  }

  /** ===================================================================
                       Private methods
      =================================================================== */

  /** Start of an SQL state that indicates an integrity contraint was
     violated.

     This should really be in the calendar api - we should not see SQL
     exceptions at the presentation level.
   */
  /* Note:  I cannot find the X/Open SQL spec on which these codes
     are based; I base this seemingly arbitrary number on an IBM
     list of SQLstate error codes, which you can find at:
     http://www.redbooks.ibm.com/tstudio/db2_400/CLI/sqlmsg/allhtml.htm

     Note that other lists of codes I could find (i.e., Informix's
     cursory list) at least agree that if it starts
     with '23', it's an 'integrity constraint violation'.
   */
  private static final String INTEGRITY_PREFIX = "23";

  /**
    @return Is the given exception one that indicates an integrity
    constraint was violated?
    @param e The exception in question
   */
  private boolean isConstraintException(SQLException e) {
    return e.getSQLState() != null &&
           e.getSQLState().startsWith(INTEGRITY_PREFIX);
  }

  private PubEvents pubEvents;

  private PubEvents getPubEvents() {
    if (pubEvents == null) {
      pubEvents = new PubEvents(this, debug);
    }

    return pubEvents;
  }

  private EventsI getEventsI() {
    try {
      if (eventsCache == null) {
        if (publicAdmin) {
          //eventsCache = PublicEvents.getPublicEvents(user);
          // Until access control works in back end
          eventsCache = PublicEvents.getPublicEvents();

          if (debug) {
            getLogger().debug("Got PublicEvents object for " + user);
          }
        } else if (user.isGuest()) {
          if (curFilter == null) {
            curFilter = defaultFilter;
          }
          eventsCache = new FilteredEvents(curFilter);

          if (debug) {
            getLogger().debug("Got FilteredEvents object");
          }
        } else {
          eventsCache = new PersonalEvents(user);

          if (debug) {
            getLogger().debug("Got PersonalEvents object for " + user);
          }
        }
      }

      return eventsCache;
    } catch (Throwable t) {
      getLogger().error(t);
      return null;
    }
  }

  private static final int readAccess = 0;
  private static final int writeAccess = 1;

  /** This does limited access checking to bar access to public objects
   * other than events.
   * If the current user is not a public admin user we throw an exception.
   *
   * @param access      int desired access (read/write)
   */
  private void checkPublicAccess(int access) throws CalFacadeException {
    if (access == readAccess) {
      return;
    }

    if (publicAdmin) {
      return;
    }

    throw new CalFacadeAccessException();
  }

  /** This does access checking for events.
   *
   * @param access      int desired access (read/write)
   * @param e           Event object
   */
  private void checkAccess(int access, Event e) throws CalFacadeException {
    String creator = null;
    if (e.getCreator() != null) {
     creator = e.getCreator().getName();
    }
    checkAccess(access, e.isPublic(), creator);
  }

  /** This does the access check for all objects.
   *
   * @param access      int desired access (read/write)
   * @param publicObj   true for a public object
   * @param creator     String creator of object
   */
  private void checkAccess(int access, boolean publicObj, String creator)
          throws CalFacadeException {
    boolean reading = access == readAccess;

    if (user.isGuest()) {
      if (reading && publicObj) {
        return;
      }

      throw new CalFacadeAccessException();
    }

    if (publicAdmin) {
      if (!publicObj) {
        //We probably want to allow read access by superuser
        throw new CalFacadeAccessException();
      }

      if (reading) {
        // read anything public in admin
        return;
      }
    } else {
      if (publicObj) {
        if (reading) {
          // Why not - it's public
          return;
        }

        throw new CalFacadeAccessException();
      }
    }

    // Have to be owner
    if (userVO.getName().equals(creator))  {
      return;
    }

    throw new CalFacadeAccessException();
  }

  /** Ensure the current user has super user access.
   */
  private void requireSuper() throws CalFacadeException {
    if (publicAdmin && (userVO != null) && userVO.isSuperUser()) {
      return;
    }

    throw new CalFacadeAccessException();
  }

  /** Ensure the current user is not a guest.
   */
  private void requireNotGuest() throws CalFacadeException {
    if (!user.isGuest())  {
      return;
    }

    throw new CalFacadeAccessException();
  }

  private void handleSQLException(SQLException se) throws CalFacadeException {
    // I believe SQLExceptions are not ALWAYS serializable
    // XXX --- SQLExceptions *are* Serializable -- gsb
    throw new CalFacadeException(se.getMessage());
  }

  /**
    Handle an unserializable exception
    @param e The unserializable exception
   */
  private void handleUnserializableException(Exception e)
      throws CalFacadeException {
    getLogger().error(e);
    throw new CalFacadeException(e.getMessage());
  }

  /** Get a logger for messages
   */
  private Logger getLogger() {
    if (log == null) {
      log = Logger.getLogger(this.getClass());
    }

    return log;
  }
}

