package edu.washington.cac.calfacade.impl;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.EventRef;
import edu.washington.cac.calendar.data.InstanceRecurrence;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.MasterRecurrence;
import edu.washington.cac.calendar.data.NoRecurrence;
import edu.washington.cac.calendar.data.Recurrence;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.EventVO;
import edu.washington.cac.calfacade.shared.EventRefVO;
import edu.washington.cac.calfacade.shared.KeywordVO;
import edu.washington.cac.calfacade.shared.LocationVO;
import edu.washington.cac.calfacade.shared.MyCalendarVO;
import edu.washington.cac.calfacade.shared.RecurrenceVO;
import edu.washington.cac.calfacade.shared.SponsorVO;
import edu.washington.cac.calfacade.shared.UserVO;

import java.util.Iterator;

/** This class provides a number of static methods used to create internal
 * calendar data objects from value objects.
 *
 * @author Mike Douglass    douglm@rpi.edu
 */
public class CalFactory {
  private CalFactory() {}   // No instantiation allowed here

  /** ===================================================================
   *                   Events
   *  =================================================================== */

  /** Return an event built from a value object
   *
   * @param  evvo    event value object
   * @return Event   event
   */
  public static Event getEvent(EventVO evvo) throws CalFacadeException {
    try {
      Event ev = new Event(evvo.getId(),
               evvo.getShortdesc(),
               evvo.getLongdesc(),
               evvo.getStartdate(),
               evvo.getStarttime(),
               evvo.getEnddate(),
               evvo.getEndtime(),
               evvo.isPublic(),
               evvo.getLink(),
               getUser(evvo.getCreator()),
               evvo.getCost(),
               null,
               evvo.getLocationid(),
               evvo.getSponsorid(),
               evvo.getSeq(),
               evvo.getLastmod(),
               evvo.getCreated());

      ev.setRecurrence(getRecurrence(evvo.getRecurrence()));
      return ev;
    } catch (CaldataException cde) {
      handleCaldataException(cde);
      return null;
    }
  }

  /** ===================================================================
   *                   Recurrences
   *  =================================================================== */

  /** Return a recurrence built from a value object
   *
   * @param  rvo     recurrence value object
   * @return r       recurrence
   */
  public static Recurrence getRecurrence(RecurrenceVO rvo)
      throws CalFacadeException {
    if (rvo == null) {
      return NoRecurrence.NO_RECUR;
    }

    if (!rvo.isMaster()) {
      return new InstanceRecurrence(rvo.getMasterId());
    }

    MasterRecurrence mr = new MasterRecurrence(rvo.getMasterId());

    Iterator it = rvo.getInstances().iterator();

    while (it.hasNext()) {
      mr.addInstance(getEvent((EventVO)it.next()));
    }

    return mr;
  }

  /** ===================================================================
   *                   EventRefs
   *  =================================================================== */

  /** Return an event ref built from a value object
   *
   * @param  ervo     EventRefVO  value object
   * @return EventRef event ref
   */
  public static EventRef getEventRef(EventRefVO ervo)
      throws CalFacadeException {
    EventRef er = new EventRef(ervo.getEventid(),
                               getUser(ervo.getUser()),
                               ervo.getPurpose());

    return er;
  }

  /** ===================================================================
   *                   Keywords
   *  =================================================================== */

  /** Return a keyword represented by the value object
   *
   * @param  kvo     KeywordVO value object
   * @return Keyword keyword object
   */
  public static Keyword getKeyword(KeywordVO kvo) throws CalFacadeException {
    try {
      return new Keyword(kvo.getId(),
                    kvo.getWord(),
                    kvo.getDescription(),
                    getUser(kvo.getCreator()),
                    kvo.isPublic());
    } catch (CaldataException cde) {
      handleCaldataException(cde);
      return null;
    }
  }

  /** ===================================================================
   *                   Locations
   *  =================================================================== */

  /** Return a location represented by the value object
   *
   * @param  lvo     Location value object
   * @return Location location
   */
  public static Location getLocation(LocationVO lvo)
     throws CalFacadeException {
  try {
    return new Location(lvo.getId(),
                  lvo.getAddress(),
                  lvo.getSubaddress(),
                  lvo.getLink(),
                  getUser(lvo.getCreator()),
                  lvo.isPublic());
    } catch (CaldataException cde) {
      handleCaldataException(cde);
      return null;
    }
  }

  /** ===================================================================
   *                   Sponsors
   *  =================================================================== */

  /** Return a sponsor represented by the value object
   *
   * @param  svo     Sponsor value object
   * @return Sponsor sponsor
   */
  public static Sponsor getSponsor(SponsorVO svo) throws CalFacadeException {
    try {
      return new Sponsor(svo.getId(),
                  svo.getName(),
                  svo.getPhone(),
                  svo.getEmail(),
                  svo.getLink(),
                  getUser(svo.getCreator()),
                  svo.isPublic());
    } catch (CaldataException cde) {
      handleCaldataException(cde);
      return null;
    }
  }

  /** ===================================================================
   *                   Users
   *  =================================================================== */

  /** Return a user represented by the value object
   *
   * @param  uvo     user value object
   * @return User    user
   */
  public static User getUser(UserVO uvo) throws CalFacadeException {
    User u =  new User(uvo.getName(),
                       uvo.getNameApp());
    u.setId(uvo.getId());
    u.setRights(uvo.getRights());
    u.setPublicEvents(uvo.getPublicEvents());

    return u;
  }

  /** ===================================================================
   *                   MyCalendar
   *  =================================================================== */

  /** Return a MyCalendar represented by the value object
   *
   * @param  mvo        MyCalendarVO value object
   * @return MyCalendar calendar object
   */
  public static MyCalendar getMyCalendar(MyCalendarVO mvo)
      throws CalFacadeException {
    return new MyCalendar(mvo.getCalendar());
  }

  private static void handleCaldataException(CaldataException cde) throws CalFacadeException {
    // I believe CaldataException are not serializable
    throw new CalFacadeException(cde.getMessage());
  }
}

