package edu.washington.cac.calfacade.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.Vector;

import edu.washington.cac.calfacade.shared.CalFacadeException;
import edu.washington.cac.calfacade.shared.AdminGroups;
import edu.washington.cac.calfacade.shared.AdminGroupROVO;
import edu.washington.cac.calfacade.shared.AdminGroupVO;

/** An implementation of AdminGroups which stores the groups in the calendar
 * database.
 *
 * @author Mike Douglass douglm@rpi.edu
 * @version 1.0
 */
public class AdminGroupsDbImpl extends EventData implements AdminGroups {
  private AdminGroupROVO currentGroup;

  private static final String groupTable = "adminGroups";
  private static final String groupMemberTable = "adminGroupMembers";

  private static final String fldGroupName = "groupname";
  private static final String fldGroupDesc = "description";
  private static final String fldGroupGroupOwner = "groupOwner";
  private static final String fldGroupOwnerid = "ownerid";
  private static final String fldUserid = "userid";

  /** SQL to delete a group */
  private static final String deleteGroupSQL =
      "delete from " + groupTable + " where " +
        fldGroupName + "=?";

  /** SQL to add an entry to the group table */
  private static final String addGroupSQL =
      "insert into " + groupTable + " (" +
          fldGroupName + ", " +
          fldGroupDesc + ", " +
          fldGroupGroupOwner + ", " +
          fldGroupOwnerid + ") values (?, ?, ?, ?)";

  /** SQL to find an entry in the group table */
  private static final String findGroupSQL =
      "select " +
          fldGroupName + ", " +
          fldGroupDesc + ", " +
          fldGroupGroupOwner + ", " +
          fldGroupOwnerid + " from " +
        groupTable + " where " +
          fldGroupName + "=? ";

  /** SQL to find an entry in the group table with the given event owner. */
  private static final String findGroupByEventOwnerSQL =
      "select " +
          fldGroupName + ", " +
          fldGroupDesc + ", " +
          fldGroupGroupOwner + ", " +
          fldGroupOwnerid + " from " +
        groupTable + " where " +
          fldGroupOwnerid + "=? ";

  /** SQL to return all entries in the group table */
  private static final String allGroupsSQL =
      "select " +
          fldGroupName + ", " +
          fldGroupDesc + ", " +
          fldGroupGroupOwner + ", " +
          fldGroupOwnerid + " from " +
        groupTable + " order by " +
          fldGroupName;

  /** SQL to update an entry in the group table */
  private static final String updateGroupSQL =
      "update " + groupTable + " set " +
          fldGroupDesc + "=?, " +
          fldGroupGroupOwner + "=?, " +
          fldGroupOwnerid + "=? where " +
          fldGroupName + "=? ";

  /** SQL to delete a group member */
  private static final String deleteGroupMemberSQL =
      "delete from " + groupMemberTable + " where " +
        fldGroupName + "=? and " +
        fldUserid + "=?";

  /** SQL to delete all group members */
  private static final String deleteAllGroupMembersSQL =
      "delete from " + groupMemberTable + " where " +
        fldGroupName + "=?";

  /** SQL to add an entry to the group member table */
  private static final String addGroupMemberSQL =
      "insert into " + groupMemberTable + " (" +
          fldGroupName + ", " +
          fldUserid +") values (?, ?)";

  /** SQL to return members of a group */
  private static final String findGroupMembersSQL =
      "select " +
         fldUserid + " from " +
          groupMemberTable + " where " +
            fldGroupName + "=?";

  /** SQL to return names of groups for a user */
  private static final String findMemberGroupsSQL =
      "select " +
         fldGroupName + " from " +
          groupMemberTable + " where " +
            fldUserid + "=?";

  /** ===================================================================
   *  The following return the state of the current user.
   *  =================================================================== */

  /** Set the current user group or null if the user is not (acting as) a
   * group member.
   *
   * @param name            String name of the group.
   * @exception CalFacadeException If there's a problem
   */
  public void setAdminGroup(String name) throws CalFacadeException {
    if (name == null) {
      currentGroup = null;
    } else {
      AdminGroupVO ug = findGroup(name);

      if (ug == null) {
        throw new CalFacadeException("Group " + name + " does not exist.");
      }

      currentGroup = ug.getAdminGroupROVO();
    }
  }

  /** Return a read only object representing the state of the current user
   * group. This will be null if the user is not in any group.
   *
   * @return AdminGroupVORO        object representing user state
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupROVO getAdminGroup() throws CalFacadeException {
    return currentGroup;
  }

  /** ===================================================================
   *  The following should not change the state of the current users
   *  group which is set and retrieved with the above methods.
   * The first group are available even if group maintenance is not.
   *  =================================================================== */

  /** Return an array of objects representing the groups this user is a
   * member of. This will be null if the user is not in any group.
   *
   * @return AdminGroupVO[]         user groups
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO[] getAdminGroups(String userid)
      throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;
    Vector v = new Vector();

    try {
      if (userid != null) {
        s = getPreparedStatement(findMemberGroupsSQL);

        s.setString(1, userid);
      } else {
        s = getPreparedStatement(allGroupsSQL);
      }

      rs = s.executeQuery();

      while (rs.next()) {
        v.addElement(rs.getString(fldGroupName));
      }

      int sz = v.size();

      if (sz == 0) {
        return null;
      }

      AdminGroupVO[] ugs = new AdminGroupVO[sz];

      int i = 0;
      Enumeration enum = v.elements();

      while (enum.hasMoreElements()) {
        ugs[i] = findGroup((String)enum.nextElement());

        i++;
      }

      /** Add the members
       */
      for (int ugi = 0; ugi < ugs.length; ugi++) {
        AdminGroupVO ug = ugs[ugi];

        ug.setMembers(getMembers(ug.getName()));
      }

      return ugs;
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /** Find a group.
   *
   * @param  name          String group name
   * @return AdminGroupVO   group object
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO findGroup(String name) throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(findGroupSQL);

      s.setString(1, name);
      rs = s.executeQuery();

      return makeGroupVO(rs);
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /** Find a group given the event owner.
   *
   * @param  id             String event owner id
   * @return AdminGroupVO   group object
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO findGroupByEventOwner(String id)
      throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;

    try {
      s = getPreparedStatement(findGroupByEventOwnerSQL);

      s.setString(1, id);
      rs = s.executeQuery();

      return makeGroupVO(rs);
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }

  /** Show whether user entries can be modified with this
   * class. Some sites may use other mechanisms.
   *
   * @return boolean    true if group maintenance is implemented.
   */
  public boolean getGroupMaintOK() {
    return true;
  }

  /** ===================================================================
   *  The following are available if group maintenance is on.
   *  =================================================================== */

  /** Create an empty group.
   *
   * @param  name          String group name
   * @param  description   String group description
   * @param  groupOwnerid  String id of user who owns the group.
   * @param  ownerid       String group ownerid
   * @exception CalFacadeException If there's a problem
   */
  public void makeGroup(String name,
                        String description,
                        String groupOwnerid,
                        String ownerid) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(addGroupSQL);

      s.setString(1, name);
      s.setString(2, description);
      s.setString(3, groupOwnerid);
      s.setString(4, ownerid);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Add a member to a group
   *
   * @param group    String group name
   * @param id       String member id
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void addMember(String group, String id) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(addGroupMemberSQL);

      s.setString(1, group);
      s.setString(2, id);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Remove a member from a group
   *
   * @param group    String group name
   * @param id       String member id
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void removeMember(String group, String id) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(deleteGroupMemberSQL);

      s.setString(1, group);
      s.setString(2, id);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Delete a group
   *
   * @param group    String group name
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void removeGroup(String group) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(deleteGroupSQL);

      s.setString(1, group);
      s.executeUpdate();

      s = getPreparedStatement(deleteAllGroupMembersSQL);

      s.setString(1, group);
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Update a group
   *
   * @param group    AdminGroupVO object
   * @exception CalFacadeException   For invalid usertype values.
   */
  public void updateGroup(AdminGroupVO group) throws CalFacadeException {
    PreparedStatement s = null;

    try {
      s = getPreparedStatement(updateGroupSQL);

      s.setString(1, group.getDescription());
      s.setString(2, group.getGroupOwnerid());
      s.setString(3, group.getOwnerid());
      s.setString(4, group.getName());
      s.executeUpdate();
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      close();
    }
  }

  /** Return a collection of all groups
   *
   * @return AdminGroupVO[]    groups.
   * @exception CalFacadeException If there's a problem
   */
  public AdminGroupVO[] getAll() throws CalFacadeException {
    return getAdminGroups(null);
  }


  /** Return an array of group members
   *
   * @param  name             String group name
   * @return String[]         group members
   * @exception CalFacadeException If there's a problem
   */
  private String[] getMembers(String name)
      throws CalFacadeException {
    PreparedStatement s = null;
    ResultSet rs = null;
    Vector v = new Vector();

    try {
      s = getPreparedStatement(findGroupMembersSQL);

      s.setString(1, name);
      rs = s.executeQuery();

      while (rs.next()) {
        v.addElement(rs.getString(fldUserid));
      }

      int sz = v.size();

      if (sz == 0) {
        return null;
      }

      return (String[])v.toArray(new String[v.size()]);
    } catch (Throwable t) {
      throw new CalFacadeException(t);
    } finally {
      if (rs != null) {
        try {
          rs.close();
        } catch (Throwable t) {}
        rs = null;
      }

      close(); // Will free s
      s = null;
    }
  }


  /** Make a populated group object from a result set.
   *
   * @param  rs             ResultSet from search
   * @return AdminGroupVO   grooup object
   */
  public AdminGroupVO makeGroupVO(ResultSet rs) throws Throwable {
    if (!rs.next()) {
      return null;
    }

    AdminGroupVO ug = new AdminGroupVO(
                rs.getString(fldGroupName),
                rs.getString(fldGroupDesc),
                rs.getString(fldGroupGroupOwner),
                rs.getString(fldGroupOwnerid),
                null);

    ug.setMembers(getMembers(ug.getName()));

    return ug;
  }
}
