package edu.washington.cac.calenv;

import java.sql.Connection;
import java.util.Properties;
import org.apache.log4j.Logger;

/** An interface to the outside world - the methods here provide access to
 * properties and resources supplied by the environment. The concrete class
 * is site specific and its name is defined in the build time properties.
 *
 * <p>The name of the supplied concrete class will be built into the
 * resource "/properties/calendar/env.properties".
 *
 * <p>This is required as a minimum to get the system going. Other properties
 * may be stored elsewhere and can be initialised through a call to one of
 * the abstract init methods, either from the web conetxt or from the ejb
 * context for example.
 *
 * <p>Alternatively, put all the properties in that file.
 *
 * <p>This approach needs revisiting to split out core parameters, those
 * that affect the back end only and those that are client only.
 */
public abstract class CalEnv {
  /** Location of the properties file */
  private static final String propertiesFile =
      "/properties/calendar/env.properties";

  private static final String envClassProperty = "org.cal.envclass";

  /** This can be static. There's only one for the application.
   *
   * <p>However, it's probably not valid for the j2ee world.
   */
  private static CalEnv env;

  protected static Properties pr;

  public static CalEnv getEnv() throws CalEnvException {
    return getEnv(false);
  }

  public static CalEnv getEnv(boolean debug) throws CalEnvException {
    if (env != null) {
      return env;
    }

    pr = new Properties();

    try {
      /** Load properties file */

      pr.load(CalEnv.class.getResourceAsStream(propertiesFile));

      String envClassName = (String)pr.get(envClassProperty);

      if (envClassName == null) {
        throw new CalEnvException("No property \"" + envClassProperty +
                                  "\" defined in file " + propertiesFile);
      }

      Object o = Class.forName(envClassName).newInstance();

      if (!(o instanceof CalEnv)) {
        throw new CalEnvException("Class \"" + envClassName +
                                  "\" defined in file " + propertiesFile +
                                  "\" is not a subclass of " +
                                  CalEnv.class.getName());
      }

      env = (CalEnv)o;

      env.init(debug);
    } catch (CalEnvException cee) {
      throw cee;
    } catch (Throwable t) {
      Logger.getLogger(CalEnv.class).error("getEnv error", t);
      throw new CalEnvException(t.getMessage());
    }

    return env;
  }

  /** Return all properties from the global environment.
   *
   * @return Properties    global properties object
   */
  public Properties getProperties() throws CalEnvException {
    return pr;
  }

  /* ====================================================================
   *                 Methods returning env properties.
   * These methods by default return values from the env.properties file.
   * They can be overrriden by the implementing class.
   * ==================================================================== */

  /** Can this application change public entities (events, locations, etc.?)
    @return Can this application change public entities (events, locations, 
    	etc.?)
   */
  public boolean appCanChangePublicEntities() {
    return new Boolean(pr.getProperty("org.cal.appCanChangePublicEntities")).
               booleanValue();
  }

  /** Get the name of the class which implements the UserAuth interface
   *  defined in the facade.
   *
   * @return String    name of UserAuth class
   */
  public String getUserAuthClassName() throws CalEnvException {
    /* We don't have to call getEnv() to ensure load of properties as this
       is not a static method so the caller must have called getEnv();
       */
    return pr.getProperty("org.cal.userauthclass");
  }

  /** Get the name of the class which implements the AdminGroups interface
   *  defined in the facade.
   *
   * @return String    name of AdminGroups class
   */
  public String getAdminGroupsClassName() throws CalEnvException {
    /* We don't have to call getEnv() to ensure load of properties as this
       is not a static method so the caller must have called getEnv();
       */
    return pr.getProperty("org.cal.admingroupsclass");
  }

  /** Get the name of the class which implements the interface to the
   *  calendar back end module.
   *
   * @return String    name of class
   */
  public String getCalIntfClassName() throws CalEnvException {
    /* We don't have to call getEnv() to ensure load of properties as this
       is not a static method so the caller must have called getEnv();
       */
    return pr.getProperty("org.cal.calintfclass");
  }

  /** Return the update check interval in millisecs.
   * This controls how often the public events object checks for changes to
   * the database. When a change is made it will flush the data.
   *
   * If set to < 0 no check for updates will take place.
   *             0 will cause a cehck on every call to public events methods
   *         5000 for example means a check at most every 5 seconds.
   *
   * @return long    update check interval in millisecs
   */
  public long getUpdateCheckInterval() throws CalEnvException {
    /* We don't have to call getEnv() to ensure load of properties as this
       is not a static method so the caller must have called getEnv();
       */
    String s = pr.getProperty("org.cal.update.check.interval");

    if (s == null) {
      return -1;
    }

    try {
      return Long.parseLong(s);
    } catch (Throwable t) {
      Logger.getLogger(this.getClass()).error("Bad update interval", t);
      throw new CalEnvException(t.getMessage());
    }
  }

  /* ====================================================================
   *                      The abstract methods
   * ==================================================================== */

  /** Called after we obtain the object to set the properties.
   * The defautl is to read the properties from a file resource.
   * Subclasses may override to do whatever they need here.
   *
   * @param debug     boolean true for debugging
   */
  public abstract void init(boolean debug) throws CalEnvException;

  /** Method which provides an application or site specific way to set the
   * properties. The parameter object may be used by the implementing class
   * to obtain run-time properties.
   *
   * <p>It will be called early in a web-session with the incoming request.
   * This allows the CalEnv class to retrieve run-time values from the web
   * environment if required.
   *
   * <p>Alternatively, in a j2ee environment, it may ignore the call(s) to
   * initProperties and set up all its values from the deployment descriptor.
   *
   * @param val         implementation specific but will be called at least
   *                    once with an HttpServletRequest
   */
  public abstract void initProperties(Object val) throws CalEnvException;

  /** Returns a Connection object for database interactions.
   *
   * @return Connection  to the database.
   */
  public abstract Connection getConnection() throws CalEnvException;

  /* ====================================================================
   *                      Predefined creators
   * ==================================================================== */

  /** Get the id used to create special entries in the database,
   *  such as default Locations and Sponsors that should always exist.
   *
   * @return User    special user name
   */
  public String specialCreator() {
    return pr.getProperty("org.cal.specialCreator");
  }

  /** The user used to create keywords that can be used with official
   * public events.
   *
   * @return User  special keyword user name
   */
  public String officialKeywordCreator() {
    return pr.getProperty("org.cal.officialKeywordCreator");
  }

  /** The user used to create keywords that can be used with unofficial
   * public events.
   *
   * @return String    unofficial events keyword user name
   */
  public String unofficialKeywordCreator() {
    return pr.getProperty("org.cal.unofficialKeywordCreator");
  }

  /** The user used to create public entities that should not be loaded by
   * default
   *
   * @return User  not to be loaded creator name
   */
  public String notToBeLoadedByDefaultCreator() {
    return pr.getProperty("org.cal.notToBeLoadedByDefaultCreator");
  }
}

