// java

package edu.washington.cac.calendar.filter;

//import edu.washington.cac.calendar.CalParameter;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Sponsor;

/**
  A filter that selects events that have a certain sponsor

  @author Greg Barnes
  @version 1.0
 */
public class SponsorFilter extends Filter
{
  public static final String INTERNAL_NAME = "sponsor";

  /** The Sponsor to match */
  private Sponsor s;

  /**
    Create a new instance
    @param s Sponsor to match
   */
  public SponsorFilter(Sponsor s)
  {
    this.s = s;
  }

  /**
    Create a new instance
    @param id Id of the Sponsor to match
    @exception NoSuchItemException if there is no such sponsor
   */
  public SponsorFilter(int id) throws NoSuchItemException
  {
    this.s = PublicEvents.getPublicSponsors().getSponsor(id);
  }

  /**
    @return If the filter selects the event
    @param e Event to test
   */
  public boolean match(Event e)
  {
    /* If a public sponsor is entered in a slightly different way (e.g.,
       with a different phone number), a new sponsor is created.  Matching
       on ID, therefore, will probably miss many valid matches, so we
       match on name
     */
    return s.getName().equals(
           e.getSponsor(PublicEvents.getPublicEvents()).getName());
  }

  /**
    Get a human-readable name for the calendar represented by the filter
    @return a human-readable name for the calendar represented by the filter
   */
  public String getName()
  {
    return this.s.getName();
  }

  /**
    Get an internal name for the calendar represented by the filter
    @return an internal name for the calendar represented by the filter
   */
  public String getInternalName()
  {
//%%% this could fail miserably if the filter was constructed by passing
//%%% in a bogus sponsor with the correct name
    return INTERNAL_NAME + this.s.getId();
  }
}
