// java

package edu.washington.cac.calendar.filter;

import java.sql.SQLException;
import java.util.Iterator;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.Locations;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.Sponsors;

/**
  A filter that selects events that contain a string in certain fields

  @author Greg Barnes
  @version 1.0
 */
public class SearchFilter extends Filter
{
  public static final String INTERNAL_NAME = "searchString";

  /** The String to match */
  private String s;

  /**
    Create a new instance
    @param s String to match
   */
  public SearchFilter(String s)
  {
    this.s = s;
  }

  /**
    Does string #1 contain string #2 as a substring?
    @param s1 String #1
    @param s2 String #2
    @return Does string #1 contain string #2 as a substring?
   */
  private boolean contains(String s1, String s2)
  {
    return s1 != null && s1.toLowerCase().indexOf(s2.toLowerCase()) != -1;
  }

  /**
    Does the filter selects the event?
    @param e Event to test
    @return If the filter selects the event
   */
  public boolean match(Event e)
  {
    Keywords ks = PublicEvents.getPublicKeywords();
    Locations ls = PublicEvents.getPublicLocations();
    Sponsors ss = PublicEvents.getPublicSponsors();

    if (contains(e.getShortdesc(), s) ||
        contains(e.getLongdesc(), s) ||
        contains(e.getSponsor(ss).getName(), s) ||
        contains(e.getLocation(ls).getAddress(), s) ||
        contains(e.getLocation(ls).getSubaddress(), s) ||
        contains(e.getSponsor(ss).getEmail(), s) ||
        contains(e.getSponsor(ss).getPhone(), s) ||
        contains(e.getCost(), s) )
    {
      return true;
    }

    Iterator keywordIds = e.getKeywords();

    while (keywordIds.hasNext()) {
      Integer i = (Integer) keywordIds.next();
      Keyword k;

      try {
        k = ks.getKeyword(i.intValue());
      } catch (NoSuchItemException ex) {
        // not a big deal, but print a message to the log
        Logger.getLogger(this.getClass()).warn(
            "SearchFilter.match: public keyword not found? " + i);
        continue;
      }

      if (contains(k.getWord(), s) || contains(k.getLongdesc(), s)) {
        return true;
      }
    }

    return false;
  }

  /**
    Get a human-readable name for the calendar represented by the filter
    @return a human-readable name for the calendar represented by the filter
   */
  public String getName()
  {
    return this.s;
  }

  /**
    Get an internal name for the calendar represented by the filter
    @return an internal name for the calendar represented by the filter
   */
  public String getInternalName()
  {
    return INTERNAL_NAME + getName();
  }
}
