// java

package edu.washington.cac.calendar.filter;

import java.sql.SQLException;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import edu.washington.cac.calendar.data.Cache;
import edu.washington.cac.calendar.data.CacheI;
import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.ItemException;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.Caldata;

/**
  The set of all public calendars.  

  @author Greg Barnes
  @version 1.1
 */
/*
  Below, we assume there are 2 'top-level' sets of calendars, 'Main', 
  and the singleton 'Default'.  We should allow for any number, probably by
  saying any calendar whose id is < (some value in a properties file) 
  constitutes a top-level calendar
 */
public class PublicCalendars extends Cache implements CacheI
{
  /** Initialize the set
    */
  public PublicCalendars() 
  {
    super(new User());
  }

  /** Initialize the set, loading it from the database
     @exception SQLException if we can't load the set from the database
     @exception ItemException if the set contains bad data
   */
  public void doLoadAll() throws SQLException, ItemException
  {
    try {
      getCaldata().loadCalendars(this);
    } catch (NoSuchItemException e) {
      e.printStackTrace();
      throw new CaldataException("Couldn't find item:" + e.getMessage());
    }
  }

  /**
    Get the set of public calendars
    @return the set of public calendars
   */
  public static PublicCalendars getPublicCalendars()
  {
    return PublicEvents.getPublicCalendars();
  }

  /**
    Get a calendar from the set
    @param calendarId unique id of the calendar to get
    @return The calendar
    @exception NoSuchItemException If the calendar isn't in the cache
   */
  public Calendar getCalendar(int calendarId) throws NoSuchItemException
  {
    return (Calendar) get(calendarId);
  }

  /**
    Get a calendar from the set
    @param name The name of the calendar to get.  Note that this might
      be ambiguous
    @return The first calendar with the same name
    @exception NoSuchItemException If the calendar isn't in the cache
   */
  public Calendar getCalendar(String name) throws NoSuchItemException
  {
    for (Iterator i = elements(); i.hasNext(); ) {
      Calendar c = (Calendar) i.next();

      if (c.getFilter().getName().equals(name)) {
        return c;
      }
    }

    throw new NoSuchItemException();
  }

  /** The id of the default calendar */
  public static final int DEFAULT_ID = 1;

  /** The id of the parent of the main calendars */
  public static final int MAIN_ID = 2;

  /**
    Get the default calendar
    @return the default calendar
   */
  public static Calendar defaultCalendar()
  {
    try {
      return getPublicCalendars().getCalendar(DEFAULT_ID);
    } catch (NoSuchItemException e) {
      throw new IllegalStateException("No default calendar!");
    }
  }

  /**
    Get the set of main calendars
    @return the set of main calendars
   */
  public static SortedSet mainCalendars()
  {
    try {
      return ((OrFilter) getPublicCalendars().getCalendar(MAIN_ID).getFilter()).
          getChildren();
    } catch (NoSuchItemException e) {
      return new TreeSet();
    }
  }

  /**
    Get the set of printable subcalendars of a calendar
    @return the set of printable subcalendars of a calendar
   */
  public static Vector printableSubcalendars(Calendar c)
  {
    Filter filter = c.getFilter();

    if (filter.showChildren()) {
      Iterator i = filter.getChildren().iterator();
      Vector subcalendars = new Vector();

      while (i.hasNext()) {
        Calendar child = (Calendar) i.next();
        
        if (!child.isAlias()) {
          subcalendars.add(child);
          subcalendars.addAll(printableSubcalendars(child));
        }
      }

      return subcalendars;
    } else {
      return new Vector();
    }
  }

  /**
    Get the set of printable calendars
    @return the set of printable calendars
   */
  public static Vector printableCalendars()
  {
    Vector pc = new Vector();
    Iterator i = getPublicCalendars().elements();

    while (i.hasNext()) {
      Calendar c = (Calendar) i.next();

      if (c.isTopLevel()) {
        if (c.getId() != MAIN_ID) {
          pc.add(c);
        }

        pc.addAll(printableSubcalendars(c));
      }
    }

    return pc;
  }

  /**
    Not implemented
    @exception IllegalStateException always thrown
   */
  protected void loadOne(int id) throws IllegalStateException
  {
    throw new IllegalStateException("Can't just load one calendar.");
  }

  /**
    Test getCalendar(String), and printableCalendars()
    @param args First argument is name of calendar to get
    @exception Exception If there's a problem
   */
  public static void main(String args[]) throws Exception
  {
    if (args.length != 1) {
      System.out.println("Usage: PublicCalendars <name>");
      return;
    }

    Calendar c = PublicCalendars.getPublicCalendars().getCalendar(args[0]);
    System.out.println(c.getId() + " " + c.getFilter().getName());

    Iterator i = printableCalendars().iterator();

    while (i.hasNext()) {
      c = (Calendar) i.next();
      System.out.println(c.getFilter().getName() + " " + c.getId());
    }
  }
}
