// java

package edu.washington.cac.calendar.filter;

import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;

/**
  A filter that selects events that have a particular location

  @author Greg Barnes
  @version 1.0
 */
public class LocationFilter extends Filter
{
  public static final String INTERNAL_NAME = "location";

  /** The Location to match */
  private Location l;

  /**
    Create a new instance
    @param l Location to match
   */
  public LocationFilter(Location l)
  {
    this.l = l;
  }

  /**
    Create a new instance
    @param id Id of the Location to match
    @exception NoSuchItemException if there is no such location
   */
  public LocationFilter(int id) throws NoSuchItemException
  {
    this.l = PublicEvents.getPublicLocations().getLocation(id);
  }

  /**
    @return If the filter selects the event
    @param e Event to test
   */
  public boolean match(Event e)
  {
    /* If a public location is entered in a slightly different way (e.g.,
       with a different address), a new location is created.  Matching
       on ID, therefore, will probably miss many valid matches, so we
       match on name
     */
    return l.getAddress().equals(
           e.getLocation(PublicEvents.getPublicEvents()).getAddress());
  }

  /**
    Get a human-readable name for the calendar represented by the filter
    @return a human-readable name for the calendar represented by the filter
   */
  public String getName()
  {
    return this.l.getAddress();
  }

  /**
    Get an internal name for the calendar represented by the filter
    @return an internal name for the calendar represented by the filter
   */
  public String getInternalName()
  {
//%%% this could fail miserably if the filter was constructed by passing
//%%% in a bogus location with the correct name
//    return CalParameter.LOCATION + this.l.getId();
    return INTERNAL_NAME + this.l.getId();
  }
}
