// java

package edu.washington.cac.calendar.filter;

import java.sql.SQLException;
import java.sql.Timestamp;

import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.db.Caldata;

/** A filter that selects all events with a lastmod greater than or equal
 * to a given time
 *
 * @author Greg Barnes
 * @version 1.0
 */
public class LastmodGEFilter extends Filter {
  /** Timestamp we are matching */
  private Timestamp lastmod;

  /** Create a new instance
   *
   * @param lastmod      Timestamp defining those events we want to see
   */
  public LastmodGEFilter(Timestamp lastmod) {
    this.lastmod = lastmod;
  }

  /** Create a new instance by defining the Timestamp value as a string of
   * the form yyyy-mm-dd hh:mm:ss.fffffffff,
   *          where ffffffffff indicates nanoseconds.
   *
   * @param dateTime       defining those events we want to see
   */
  public LastmodGEFilter(String dateTime) {
    this.lastmod = Timestamp.valueOf(dateTime);
  }

  /**
   * @param e          Event to test
   * @return boolean   true if the event is after our Timestamp or
   *                   lastmod is null.
   */
  public boolean match(Event e) {
    Timestamp thatLastmod = e.getLastmod();

    if (thatLastmod == null) {
      return true;
    }

    return (thatLastmod.compareTo(lastmod) >= 0);
  }

  /** Get a human-readable name for the calendar represented by the filter
   *
   * @return String   human-readable name for the calendar represented by the
   *                  filter
   */
  public String getName() {
    return "Events after " + lastmod;
  }

  /** Get an internal name for the calendar represented by the filter
   *
   * @return String    internal name for the calendar represented by the
   *                   filter
   */
  public String getInternalName() {
    return getName();
  }

  /** Get the lastmod we're filtering on
   *
   * @return Timestamp   value we're filtering on
   */
  Timestamp getLastmod() {
    return lastmod;
  }
}
