// java

package edu.washington.cac.calendar.filter;

import java.util.Iterator;
import java.util.NoSuchElementException;

import edu.washington.cac.calendar.data.Event;

/**
  An iterator that filters another iterator, yielding only
  the events the filter selects

  @author Greg Barnes
  @version 1.0
 */
public class FilteredIterator implements Iterator
{
  /** The Filter to apply */
  private Filter filter;

  /** The unfiltered iterator */
  private Iterator iterator;

  /** next matching element */
  private Event nextEvent;

  /**
    Create a new instance
    @param iterator Iterator.  Each element of the iterator should 
      be an Event, or ClassCastExceptions will result
    @param filter Filter to apply to the iterator
   */
  public FilteredIterator(Iterator iterator, Filter filter)
  {
    this.iterator = iterator;
    this.filter = filter;
    this.nextEvent = nextMatch();  // set up for next call
  }

  /**
    @return The next event in the iterator that matches the filter,
      or null if there are no more
   */
  private Event nextMatch()
  {
    while (iterator.hasNext()) {
      Event e = (Event) iterator.next();

      if (this.filter.match(e)) {
        return e;
      }
    } 

    return null;
  }

  /**
    Are there more elements?
    @return Are there more elements?
   */
  public boolean hasNext()
  {
    return this.nextEvent != null;
  }

  /** 
    Get The next element in the iterator
    @return The next element in the iterator
    @exception NoSuchElementException If there are no more elements left
   */
  public Object next() throws NoSuchElementException
  {
    Object o = this.nextEvent;

    if (hasNext()) {
      this.nextEvent = nextMatch();  // set up for next call
      return o;
    } else {
      throw new NoSuchElementException();
    }
  }

  /**
    unsupported
    @exception UnsupportedOperationException always thrown
   */
  public void remove() throws UnsupportedOperationException
  {
    throw new UnsupportedOperationException();
  }
}
