// java

package edu.washington.cac.calendar.filter;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.CalendarObject;
import edu.washington.cac.calendar.data.EntityRef;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.EventsI;
import edu.washington.cac.calendar.data.ItemException;
import edu.washington.cac.calendar.data.ItemAccessException;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Locations;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Sponsors;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.Caldata;

/**
  A set of events with an attached filter.  Methods which return
  the events in the set will return only those events which the filter
  selects.  (The implementation, however, may use an underlying events object
  that may contain events that the filter does not select).

  For now, we will assume this is only used by the guest user.

  @author Greg Barnes
  @version 1.0
 */
public class FilteredEvents implements EventsI
{
  /** The underlying events object */
  private EventsI events;

  /** The filter to use */
  private Filter filter;

  /** Initialize the set

    @param events Cache of events to filter
    @param filter Filter to apply
    */
  public FilteredEvents(EventsI events, Filter filter)
  {
    this.events = events;
    this.filter = filter;
  }

  /** Initialize the set for the guest user
    @param filter Filter to apply
    */
  public FilteredEvents(Filter filter)
  {
    this(PublicEvents.getPublicEvents(), filter);
  }

  /**
    Get the current filter
    @return the current filter
   */
  public Filter getFilter()
  {
    return this.filter;
  }

  /**
    Set the filter
    @param filter new filter
   */
  public void setFilter(Filter filter)
  {
    this.filter = filter;
  }

  // overridden methods
  /**
    @return If the key represents an object in the set
    @param o Key
   */
  public boolean containsKey(Object o)
  {
    try {
      return this.filter.match(this.events.getEvent(((Integer) o).intValue()));
    } catch (NoSuchItemException e) {
      return false;
    }
  }

  /**
    Get the event in the set with the given id

    @return the event in the set with the given id
    @param eventID Unique ID of the event
    @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID) throws NoSuchItemException
  {
    if (this.filter.match(this.events.getEvent(eventID))) {
      return this.events.getEvent(eventID);
    } else {
      throw new NoSuchItemException("id not in filtered events: " + eventID);
    }
  }

  /** Get the event in the set with the given id. Optionally load it into
   * the cache if it's not already there.
   *
   * @param eventID    int Unique ID of the event
   * @param loadIt     load the event if not in cache
   * @return Event     from the set with the given id
   * @exception NoSuchItemException if no such event exists
   */
  public Event getEvent(int eventID,
                        boolean loadIt) throws NoSuchItemException {
    if (this.filter.match(this.events.getEvent(eventID, loadIt))) {
      return this.events.getEvent(eventID);
    } else {
      throw new NoSuchItemException("id not in filtered events: " + eventID);
    }
  }

  /**
    get an item from the Cache

    @param id ID of the item
    @return The CalendarObject with the same id
    @exception NoSuchItemException if no item exists with that id
   */
  public CalendarObject get(int id) throws NoSuchItemException
  {
    return getEvent(id);
  }

  /**
    Get all the items in the cache
    @return All the items in the cache
   */
  public Iterator elements()
  {
    return new FilteredIterator(this.events.elements(), this.filter);
  }

  /**
    Get all the items in the cache as a collection.
    @return all the items in the cache as a collection.
   */
  public Collection values()
  {
    Iterator e = elements();
    Vector v = new Vector();

    while (e.hasNext()) {
      v.add(e.next());
    }

    return v;
  }

  /**
    Get the number of items in the Cache
    @return The number of items in the Cache
   */
  public int size()
  {
    return values().size();
  }

  /**
    @return all the elements in the cache, in sorted order
   */
  public Iterator sortedElements()
  {
    return new FilteredIterator(this.events.sortedElements(), this.filter);
  }

  /**
    Given an array of events, return an array of only the events selected by
    the filter
    @param ea Array of events
    @return an array of only the events selected by the filter
   */
  private Event[] filteredArray(Event[] ea)
  {
    Vector v = new Vector();

    for (int i = 0; i < ea.length; i++) {
      if (this.filter.match(ea[i])) {
        v.add(ea[i]);
      }
    }

    Event[] newArray = new Event[v.size()];

    for (int i = 0; i < v.size(); i++) {
      newArray[i] = (Event) v.elementAt(i);
    }

    return newArray;
  }

  /**
    Given a List of events, return a List of only the events selected by
    the filter
    @param el List of events
    @return A List of only the events selected by the filter
   */
  private List filteredList(List el)
  {
    Vector v = new Vector();

    for (Iterator i = el.iterator(); i.hasNext(); ) {
      Event e = (Event) i.next();
      if (this.filter.match(e)) {
        v.add(e);
      }
    }

    return v;
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadKeywords should the keywords be loaded (if the events are?)

    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc, boolean loadKeywords)
      throws SQLException, ItemException
  {
    return filteredArray(this.events.oneDaysEvents(mc, loadKeywords));
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way

    @param mc Date, represented as a <code>MyCalendar</code>
    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc)
      throws SQLException, ItemException
  {
    return filteredArray(this.events.oneDaysEvents(mc));
  }

  /**
    Get the events for a user over a series of days, sorted in an appropriate
    way

    @param startDate First day in the series, represented as a
        <code>MyCalendar</code>
    @param endDate Last day in the series, represented as a
        <code>MyCalendar</code>
    @return the events for a user over the series of days, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public List manyDaysEvents(MyCalendar startDate, MyCalendar endDate)
      throws SQLException, ItemException
  {
    return filteredList(this.events.manyDaysEvents(startDate, endDate));
  }

  // methods that can just be passed on to the underlying events object
  public void add(CalendarObject c) throws SQLException, ItemAccessException
  {
    this.events.add(c);
  }

  /**
    Add an item to the cache
    @param c Item to add
    @param saveToDB Save to database as well?
    @exception SQLException If there's a database problem
    @exception ItemAccessException If the item can't be added to the cache
   */
  public void add(CalendarObject c, boolean saveToDB)
      throws SQLException, ItemAccessException
  {
    this.events.add(c, saveToDB);
  }

  /**
    delete an item from the Cache

    @param c the item to delete
    @exception SQLException if there is a problem with the database
    @exception ItemException if there's a problem with c
   */
  public void delete(CalendarObject c) throws SQLException, ItemException
  {
    this.events.delete(c);
  }

  /**
    replace an item in the Cache

    @param c the item to replace (the current item with the same id
      as c will be replaced with c)
    @exception ItemException if there's a problem with the item specified
    @exception SQLException if there is a problem with the database
   */
  public void replace(CalendarObject c)
      throws SQLException, ItemException
  {
    this.events.replace(c);
  }

  /** Set the keywords cache
    @param ks new Keywords cache
   */
  public void setKeywords(Keywords ks)
  {
    this.events.setKeywords(ks);
  }

  /** Set the Locations cache
    @param ls new Locations cache
   */
  public void setLocations(Locations ls)
  {
    this.events.setLocations(ls);
  }

  /** Set the sponsors cache
    @param ss new Sponsors cache
   */
  public void setSponsors(Sponsors ss)
  {
    this.events.setSponsors(ss);
  }

  /**
    Get the set of associated locations for a user
    @return the set of associated locations for a user
    */
  public Locations getLocations()
  {
    return this.events.getLocations();
  }

  /**
    Get the set of associated sponsors for a user
    @return the set of associated sponsors for a user
   */
  public Sponsors getSponsors()
  {
    return this.events.getSponsors();
  }

  /**
     Get the set of associated keywords for a user
     @return the set of associated keywords for a user
   */
  public Keywords getKeywords() {
    return this.events.getKeywords();
  }

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  public void prepareManyDays(Calendar start,
                              Calendar end)
      throws SQLException, ItemException
  {
    this.events.prepareManyDays(start, end);
  }

//  /** load a result set containing events for one day
//      @param mc The day in question
//      @param rs The <code>java.sql.ResultSet</code> containing the events
//      @exception SQLException if there is database trouble
//      @exception CaldataException if the data we get back is bad
//    */
//  public void load(MyCalendar mc, ResultSet rs)
//      throws SQLException, CaldataException
//  {
//    this.events.load(mc, rs);
//  }

  /**
    Are all the events loaded?
    @return Are all the events loaded?
   */
  public boolean isLoaded()
  {
    return this.events.isLoaded();
  }
}
