// java

package edu.washington.cac.calendar.filter;

import java.util.Enumeration;
import java.util.NoSuchElementException;

import edu.washington.cac.calendar.data.Event;

/**
  An enumeration that filters another enumeration, yielding only
  the events the filter selects

  @author Greg Barnes
  @version 1.0
 */
public class FilteredEnumeration implements Enumeration
{
  /** The Filter to apply */
  private Filter filter;

  /** The unfiltered enumeration */
  private Enumeration enumeration;

  /** next matching element */
  private Event nextEvent;

  /**
    Create a new instance
    @param enumeration Enumeration.  Each element of the enumeration should 
      be an Event, or ClassCastExceptions will result
    @param filter Filter to apply to the enumeration
   */
  public FilteredEnumeration(Enumeration enumeration, Filter filter)
  {
    this.enumeration = enumeration;
    this.filter = filter;
    this.nextEvent = nextMatch();  // set up for next call
  }

  /**
    @return The next event in the enumeration that matches the filter,
      or null if there are no more
   */
  private Event nextMatch()
  {
    while (enumeration.hasMoreElements()) {
      Event e = (Event) enumeration.nextElement();

      if (this.filter.match(e)) {
        return e;
      }
    } 

    return null;
  }

  /**
    @return Are there more elements?
   */
  public boolean hasMoreElements()
  {
    return this.nextEvent != null;
  }

  /** 
    @return The next element in the enumeration
    @exception NoSuchElementException If there are no more elements left
   */
  public Object nextElement() throws NoSuchElementException
  {
    Object o = this.nextEvent;

    if (hasMoreElements()) {
      this.nextEvent = nextMatch();  // set up for next call
      return o;
    } else {
      throw new NoSuchElementException();
    }
  }
}
