// java

package edu.washington.cac.calendar.filter;

import java.util.SortedSet;
import java.util.TreeSet;

import edu.washington.cac.calendar.data.Event;

/**
  A filter selects events (and possibly other
  entities) that fulfill certain criteria.  For example, "All events
  that have the keyword 'Lecture'".

  A filter can have any number of children, although the effect of
  the children may (as in an OrFilter) or may not (as in a KeywordFilter)
  have something to do with which events it selects.

  @author Greg Barnes
  @version 1.1
 */
public abstract class Filter
{
  /** The children of the filter */
  private SortedSet children = new TreeSet();

  /**
    @return If the filter selects the event
    @param e Event to test
   */
  public abstract boolean match(Event e);

  /**
    Get a human-readable name for the calendar represented by the filter
    @return a human-readable name for the calendar represented by the filter
   */
  public abstract String getName();

  /**
    Get an internal name for the calendar represented by the filter.
    This name can be passed along as a form parameter, for example, and
    used to reconstruct the filter.
    @return an internal name for the calendar represented by the filter
   */
  public abstract String getInternalName();

  /**
    Is this filter the same as another object?
    @param o Object to check against
    @return Is this filter the same as the object?
   */
  public boolean equals(Object o)
  {
    if (o instanceof Filter) {
      return getInternalName().equals(((Filter) o).getInternalName());
    } else {
      return false;
    }
  }

  /**
    Add a calendar to the set of children that make up this filter
    @param c calendar to add
   */
  public void addChild(Calendar c)
  {
    children.add(c);
  }

  /**
    Get the set of children
    @return the calendars that make up this filter
   */
  public SortedSet getChildren()
  {
    return this.children;
  }

  /**
    Should the children of this calendar be displayed?
    @return Should the children of this calendar be displayed?
      Default is true; can be overridden in subclasses
   */
  public boolean showChildren()
  {
    return true;
  }
}
