// java

package edu.washington.cac.calendar.filter;

import java.sql.ResultSet;
import java.sql.SQLException;

import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.CalendarObject;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.db.Fields;

/** A filter that is stored in the database
   @author Greg Barnes
   @version 1.0
  */
public class Calendar extends CalendarObject 
{
  /** Unique id of the calendar */
  private int calendarId;

  /** Unique id of the parent of this calendar (0 if there is no parent) */
  private int parentId = 0;

  /** Is this calendar an alias for another calendar? */
  private boolean isAlias = false;

  /** The underlying filter that implements the calendar */
  private Filter filter;

  /** create a public calendar 

    @param calendarId Unique id of the calendar
    @param filter The underlying filter that implements the calendar
   */
  public Calendar(int calendarId, Filter filter)
  {
    super(new User(), true);    // always public, at least for now
    this.calendarId = calendarId;
    this.filter = filter;
  }

  /**
    Create the appropriate filter from the fields in the database
    @param name Name of the filter
    @param type Type of the filter
    @param ref Reference for the filter, if appropriate (to a keyword, 
        location, etc.)
    @exception CaldataException If the type is bad
    @exception NoSuchItemException If the reference is to a non-existent item
    @exception SQLException If we have to go to the database, and there is
      an error
   */
  private static Filter createFilter(String name, String type, int ref)
      throws CaldataException, NoSuchItemException, SQLException
  {
//System.err.println(name + " " + type + " " + ref);
    if (type.equals("O")) {
      return new OrFilter(name);
    } else if (type.equals("A")) {
      return new AndFilter(name);
    } else if (type.equals("N")) {
      return new NotFilter(name, ref);
    } else if (type.equals("C")) {
      return new CreatorFilter(ref);
    } else if (type.equals("K")) {
      return new KeywordFilter(ref);
    } else if (type.equals("L")) {
      return new LocationFilter(ref);
    } else if (type.equals("S")) {
      return new SponsorFilter(ref);
    } else if (type.equals("a")) {
      return PublicCalendars.getPublicCalendars().getCalendar(ref).getFilter();
    } else {
      throw new CaldataException("Bad filter type " + type);
    }
  }

  /**
     Set up a parent-child relationship for a calendar
     @param parentId unique id of the parent calendar, or 0 if there is 
        no parent
     @exception NoSuchItemException If the parent does not exist
   */
  private void setParent(int parentId) throws NoSuchItemException
  {
    this.parentId = parentId;

    if (parentId != 0) {
      PublicEvents.getPublicCalendars().getCalendar(parentId).
          addChild(this);
    }
  }

  /** create a calendar from the next row in a
       <code>java.sql.ResultSet</code>

    @param rs The <code>ResultSet</code> with the calendar
    @exception SQLException if there is a problem with the ResultSet
    @exception CaldataException if there is a problem with the data in the 
      ResultSet
    @exception NoSuchItemException if resultset refers to a non-existent
      object
   */
  public Calendar(ResultSet rs) 
      throws SQLException, CaldataException, NoSuchItemException
  {
    this(rs.getInt(Fields.CCALENDAR_ID.getName()),
         createFilter(rs.getString(Fields.CNAME.getName()), 
                      rs.getString(Fields.CTYPE.getName()),
                      rs.getInt(Fields.CREF_NUM.getName())));
    setParent(rs.getInt(Fields.CPARENT.getName()));
    this.isAlias = rs.getString(Fields.CTYPE.getName()).equals("a");
  }

  /** 
    Get the unique id of the calendar
    @return the unique id of the calendar
   */
  public int getId() 
  {
    return this.calendarId;
  }

  /**
    Get the underlying filter
    @return the underlying filter
   */
  public Filter getFilter()
  {
    return this.filter;
  }

  /**
    Is this calendar an alias for another?
    @return Is this calendar an alias for another?
   */
  public boolean isAlias()
  {
    return this.isAlias;
  }

  /**
    Is the calendar a top-level calendar (i.e., it has no parent?)
    @return Is the calendar a top-level calendar
   */
  public boolean isTopLevel()
  {
    return this.parentId == 0;
  }

  /**
    Is the calendar's parent a given calendar?
    @param c calendar to test for parenthood
    @return Is the calendar's parent the given calendar?
   */
  public boolean hasParent(Calendar c)
  {
    return c.getId() == this.parentId;
  }

  /** 
    Test whether this calendar is less than another calendar

    @param c The other calendar to compare.  If c is a  
      <code>CalendarObject</code> that is not a calendar, the method returns
      <code>false</code>
    @return whether this calendar is less than another calendar
   */
  public boolean lessThan(CalendarObject c) 
  {
    if (!(c instanceof Calendar)) {
      return false;
    } 

    return getId() < ((Calendar) c).getId();
  }

  /**
    Add a child to this calendar
    @param c The child to add
   */
  void addChild(Calendar c) 
  {
    this.filter.addChild(c);
  }

  /**
    Add a calendar to the database for a user
    @exception UnsupportedOperationException Always thrown, as this isn't
       yet implemented
   */
  public void add(User user)
  {
    throw new 
        UnsupportedOperationException("Can't add a calendar for a user yet.");
  }

  /**
    Delete a calendar from the database for a user
    @exception UnsupportedOperationException Always thrown, as this isn't
       yet implemented
   */
  public void delete(User user)
  {
    throw new UnsupportedOperationException(
        "Can't delete a calendar for a user yet.");
  }

  /**
    Replace a calendar in the database for a user
    @exception UnsupportedOperationException Always thrown, as this isn't
       yet implemented
   */
  public void replace(User user)
  {
    throw new UnsupportedOperationException(
        "Can't replace a calendar for a user yet.");
  }
}
