// java

package edu.washington.cac.calendar.db;

import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.User;

/** utility classes for where clauses
    @author Greg Barnes
    @version 2.0
 */
public class WhereClauses
{
  static final char QUOTE = '\'';

  static String quote(String x)
  {
    return QUOTE + x + QUOTE;
  }

  static String op(String x, String op, String y)
  {
    return x + op + y;
  }

  static String op(String x, String op, Field y)
  {
    return op(x, op, y.getStmtName());
  }

  static String op(Field x, String op, String y)
  {
    return op(x.getStmtName(), op, y);
  }

  static String op(Field x, String op, Field y)
  {
    return op(x.getStmtName(), op, y);
  }

  static String equals(String x, String y)
  {
    return op(x, "=", y);
  }

  static String equals(String x, Field y)
  {
    return op(x, "=", y);
  }

  static String equals(Field x, String y)
  {
    return op(x, "=", y);
  }

  static String equals(Field x, Field y)
  {
    return op(x, "=", y);
  }

  static String notEquals(Field x, String y)
  {
    return op(x, "!=", y);
  }

  static String matches(Field x, String y)
  {
    return op(x, " like ", quote("%" + y + "%"));
  }

  static String in(Field x, String y)
  {
    return op(x, " in ", y);
  }

  static String join(Field field1, Field field2)
  {
    return equals(field1, field2);
  }

  private static String creatorIsUser(String tableName, User user)
  {
    return equals(Tables.qualify(tableName, Fields.CREATOR),
                  quote(user.getNameDB()));
  }

  static String eventsCreatorClause(User user)
  {
    return creatorClause(Tables.EVENTS_TABLE, user);
  }

  static String creatorClause(String tableName, User user) {
      return creatorIsUser(tableName, user);
  }

  /** convert a string in YYYYMMDD format to a more db friendly
     YYYY-MM-DD format
    @param date The date to convert
    @return The String in YYYY-MM-DD format
   */
  public static String convertDate(String date) {
    if (date.length() != 8) {
      Logger.getLogger("edu.washington.cac.calendar.db.WhereClauses").warn(
          "convertDate: date not proper length " + date);
      return date;
    }

    return date.substring(0, 4) + "-" + date.substring(4, 6) + "-" +
           date.substring(6);
  }

  static String dateClause(Field dateField, String op, String date) {
    return op(dateField, op, quote(convertDate(date)));
  }

  /**
    Get the raw where clause that chooses rows that occur
    in a certain range of dates
    @param startField Name of the startdate field
    @param endField Name of the enddate field
    @param startDate First date in the range
    @param endDate Last date in the range
    @return a string that chooses rows that occur in a certain range
    of dates
   */
  static String[] datesWhereClauses(Field startField,
                                    Field endField,
                                    String startDate,
                                    String endDate)
  {
    String[] clauses = {
      dateClause(startField, "<=", endDate),
      dateClause(endField, ">=", startDate)
    };

    return clauses;
  }

  /**
    Get the raw where clause that chooses events that occur in a certain range
    of dates
    @param startDate First date in the range
    @param endDate Last date in the range
    @param useEventsTable Use fields in the events table?  If false, use
           similar fields in the master/details table
    @return a string that chooses events that occur in a certain range
    of dates
   */
  static String[] datesWhereClauses(String startDate,
                                    String endDate,
                                    boolean useEventsTable)
  {
    if (useEventsTable) {
      return datesWhereClauses(Fields.ESTARTDATE, Fields.EENDDATE, startDate,
                               endDate);
    } else {
      return datesWhereClauses(Fields.MDSTARTDATE, Fields.MDENDDATE, startDate,
                               endDate);
    }
  }

  /**
    Get a where clause that chooses events that occur in a certain range
    of dates
    @param startDate First date in the range
    @param endDate Last date in the range
    @param useEventsTable Use fields in the events table?  If false, use
           similar fields in the master/details table
    @return a where clause that chooses events that occur in a certain range
    of dates
   */
  private static WhereClause datesWhere(String startDate,
                                        String endDate,
                                        boolean useEventsTable)
  {
    return new WhereClause(datesWhereClauses(startDate, endDate,
                                             useEventsTable));
  }

  /** A 'true' value */
  public static final String TRUE = "T";

  /** A quoted 'true' value */
  private static final String QTRUE = quote(TRUE);

  /** A 'false' value */
  static final String FALSE = "F";

  /** A quoted 'false' value */
  private static final String QFALSE = quote(FALSE);

  /** A clause that is true only if an event is not public */
  static final String NOT_PUBLIC_EVENT = equals(Fields.EPUBLIC, QFALSE);

  /** Clauses to join the events table with the sponsors and locations tables */
  static final String[] EVENT_JOIN_CLAUSES =
  {
    join(Fields.ELOCATION_ID, Fields.LLOCATION_ID),
    join(Fields.ESPONSOR_ID, Fields.SSPONSOR_ID)
  };

  /** Clauses to join the events table with eventrefs and users tables */
  static final String[] EVENTREF_JOIN_CLAUSES =
  {
    join(Fields.EEVENT_ID, Fields.EREVENT_ID),
    join(Fields.ERUSER_ID, Fields.UUSER_ID)
  };

  /**
    Get a subclause that selects a user's private events
    @param user The user
    @return a subclause that selects a user's private events
   */
  static WhereClause usersPrivateEvents(User user)
  {
    return new WhereClause().
               addClause(eventsCreatorClause(user)).
               addClause(NOT_PUBLIC_EVENT);
  }

  /**
    Get a subclause that selects all public events chosen by a user
    @param user The user
    @return a subclause that selects all public events chosen by a user
   */
  private static WhereClause userPubeventsWhere(User user)
  {
    return new WhereClause().
        addClause(equals(Fields.UUSERNAME, quote(user.getNameDB()))).
        addClauses(EVENTREF_JOIN_CLAUSES);
  }

  /**
    Get the where clause for the query to get a user's selected public
      events
    @param user The user
    @param startDate Earliest date of events we're interested in.  If null,
      select all of user's select public events
    @param endDate Latest date of events we're interested in
    @return the where clause for the query to get a user's selected public
      events
   */
  static WhereClause userPubeventsWhere(User user,
                                        String startDate,
                                        String endDate)
  {
    return (startDate == null ? new WhereClause() :
                                datesWhere(startDate, endDate, true)).
        addClause(userPubeventsWhere(user));
  }

  /** Join clause that joins the events table and the master/details table */
  static final String MD_EVENT_JOIN_CLAUSE =
      join(Fields.EEVENT_ID, Fields.MDMASTER_EVENT_ID);

  /** Clauses to join the master/details, eventrefs and users tables */
  static final String[] MD_EVENTREF_JOIN_CLAUSES =
  {
    join(Fields.MDMASTER_EVENT_ID, Fields.EREVENT_ID),
    join(Fields.ERUSER_ID, Fields.UUSER_ID)
  };

  /**
    Get the where clause for the query to get a user's selected
      recurring public events
    @param user The user
    @param startDate Earliest date of events we're interested in.  If null,
      select all of a user's selected recurring public events
    @param endDate Latest date of events we're interested in
    @return the where clause for the query to get a user's selected
      recurring public events
   */
  static WhereClause userPubRecurInstancesWhere(User user,
                                                String startDate,
                                                String endDate)
  {
    return (startDate == null ? new WhereClause() :
                               datesWhere(startDate, endDate, false)).
        addClause(equals(Fields.UUSERNAME, quote(user.getNameDB()))).
        addClauses(MD_EVENTREF_JOIN_CLAUSES);
  }

  /**
    Get The base where clause for the standard query to get a
      user's events or keywords
    @param user Name of the user
    @param startDate Earliest date of events we're interested in
    @param endDate Latest date of events we're interested in
    @return The base where clause for the standard query to get a
      user's events or keywords
   */
  static WhereClause userEventsBase(User user)
  {
    return usersPrivateEvents(user);
  }

  /** a clause that is true only if the field does not match the
      notToBeLoadedByDefaultCreator
      @param field Field to match
      @return a clause that is true only if the field does not match the
      notToBeLoadedByDefaultCreator
   */
  static WhereClause toBeLoadedMatch(Field field)
  {
    return new WhereClause().addClause(notEquals(field,
        quote(DBResources.notToBeLoadedByDefaultCreator().toString())));
  }

  /** A clause that is true only if a keyword is to be loaded
      by default
   */
  static final WhereClause KEYWORD_SHOULD_BE_LOADED =
      toBeLoadedMatch(Fields.KCREATOR);

  /** A clause that is true only if an event is to be loaded
      by default
   */
  static final WhereClause EVENT_SHOULD_BE_LOADED =
      toBeLoadedMatch(Fields.ECREATOR);

  /** A clause that is true only if an event is public */
  static final WhereClause IS_PUBLIC_EVENT =
      new WhereClause().addClause(equals(Fields.EPUBLIC, QTRUE));

  /**
    The base where clause for the standard query to get a
      guest's events or keywords
    @return The base where clause for the standard query to get a
      guest's events or keywords
   */
  static WhereClause guestEventsBase()
  {
    return new WhereClause().addClause(EVENT_SHOULD_BE_LOADED).
                             addClause(IS_PUBLIC_EVENT);
  }

  /**
    Get the base of the most queries to get a user or guest's events
    @param user Name of the user
    @return the base of the most queries to get a user or guest's events
   */
  private static WhereClause eventsBase(User user)
  {
//    return user.isGuest() ? guestEventsBase() :
//                            userEventsBase(user);
    if (user.isGuest() || user.getPublicEvents()) {
      return guestEventsBase();
    }

    return userEventsBase(user);
  }

  /**
    Get the base of the most queries to get a user or guest's events
      for a series of dates
    @param user Name of the user
    @param startDate Earliest date of events we're interested in.  If null,
      select all of a user's events
    @param endDate Latest date of events we're interested in
    @param useEventsDates Test date fields in the events table?  If false,
      use similar fields in the master/details table
    @return the base of the most queries to get a user or guest's events
      for a series of dates
   */
  static WhereClause eventsBase(User user,
                                String startDate,
                                String endDate,
                                boolean useEventsDates)
  {
    if (startDate == null) {
      return eventsBase(user);
    } else {
      return datesWhere(startDate, endDate, useEventsDates).
          addClause(eventsBase(user));
    }
  }

  /**
    Get The where clause for the standard query to get a user's events
    @param user Name of the user
    @param startDate Earliest date of events we're interested in.  If null,
      select all of a user's events
    @param endDate Latest date of events we're interested in
    @param loadAuxData Will location and sponsor data be returned as well?
    @return The where clause for the standard query to get a user's events
   */
  static WhereClause eventsWhere(User user,
                                 String startDate,
                                 String endDate,
                                 boolean loadAuxData)
  {
    WhereClause wc = eventsBase(user, startDate, endDate, true);

    if (loadAuxData) {
      wc.addClauses(EVENT_JOIN_CLAUSES);
    }

    return wc;
  }

  /**
    Get The where clause for the query to get recurrence instances for a user
    @param user Name of the user
    @param startDate Earliest date of events we're interested in.  If null,
       select all of a user's recurrence instances
    @param endDate Latest date of events we're interested in
    @return The where clause for the query to get recurrence instances
       for a user
   */
  static WhereClause recurInstancesWhere(User user,
                                         String startDate,
                                         String endDate)
  {
    return eventsBase(user, startDate, endDate, false).
           addClause(MD_EVENT_JOIN_CLAUSE);
  }

  /**
    Get The where clause for a query to get all instances of a single
    recurrence
    @param eventid ID of the master event for the recurrence
    @return The where clause for a query to get all instances of a single
    recurrence
   */
  static WhereClause recurInstancesWhere(int eventid)
  {
    return eventId(eventid).addClause(IS_PUBLIC_EVENT).
           addClause(MD_EVENT_JOIN_CLAUSE);
  }

  /**
    Get The where clause for the query to get a single recurrence instance
    @param eventid ID of the event corresponding to the instance
    @return The where clause for the query to get a single recurrence instance
   */
  static WhereClause recurInstanceWhere(int eventid)
  {
    return instanceId(eventid).addClause(IS_PUBLIC_EVENT).
           addClause(MD_EVENT_JOIN_CLAUSE);
  }

  /**
    Get the where clause for a query to get a single event
    @param eventid ID of the event
    @return the where clause for a query to get a single event
   */
  static WhereClause singleEventWhere(int eventid)
  {
    return eventId(eventid).addClauses(EVENT_JOIN_CLAUSES);
  }

  /**
    Get the where clause for a query to get a single location
    @param locationid ID of the location
    @return the where clause for a query to get a single location
   */
  static WhereClause singleLocationWhere(int locationid)
  {
    return locationId(locationid);
  }

  /**
    Get the where clause for a query to get a single sponsor
    @param sponsorid ID of the sponsor
    @return the where clause for a query to get a single sponsor
   */
  static WhereClause singleSponsorWhere(int sponsorid)
  {
    return sponsorId(sponsorid);
  }

  /**
    Get the where clause for a query to get a single keyword
    @param keywordid ID of the keyword
    @return the where clause for a query to get a single keyword
   */
  static WhereClause singleKeywordWhere(int keywordid)
  {
    return keywordId(keywordid);
  }

  /**
    Get the whereclause that selects all of a user's locations
    @param user The user
    @return the whereclause that selects all of a user's locations
   */
  static WhereClause locationsWhere(User user)
  {
    if (user.isGuest() || user.getPublicEvents()) {
      return new WhereClause().
             addClause(equals(Fields.LPUBLIC, QTRUE));
    }
    return new WhereClause().
           addClause(creatorClause(Tables.LOCATIONS_TABLE, user) + " or " +
                     Fields.LLOCATION_ID.getStmtName() + "<" +
                     Location.DELETED_LOCATION_ID);
  }

  /**
    Get the whereclause that selects all of a user's sponsors
    @param user The user
    @return the whereclause that selects all of a user's sponsors
    @exception IllegalArgumentException If the user is not a guest
   */
  static WhereClause sponsorsWhere(User user) throws IllegalArgumentException
  {
    if (user.isGuest() || user.getPublicEvents()) {
      return new WhereClause().addClause(equals(Fields.SPUBLIC, QTRUE));
    }

    return new WhereClause().
           addClause(creatorClause(Tables.SPONSORS_TABLE, user) + " or " +
                     Fields.SSPONSOR_ID.getStmtName() + "<" +
                     Sponsor.DELETED_SPONSOR_ID);
  }

  /**
    Get the whereclause that selects all of a user's keywords
    @param user The user
    @return the whereclause that selects all of a user's keywords
    @exception IllegalArgumentException If the user is not a guest
   */
  static WhereClause keywordsWhere(User user) throws IllegalArgumentException
  {
    if (user.isGuest() || user.getPublicEvents()) {
      return new WhereClause().addClause(KEYWORD_SHOULD_BE_LOADED).
                               addClause(equals(Fields.KPUBLIC, QTRUE));
    }

    return new WhereClause().
        addClause(creatorClause(Tables.KEYWORDS_TABLE, user));
  }

  static final String[] KEYWORD_JOIN_CLAUSES = {
      join(Fields.EEVENT_ID, Fields.EKEVENT_ID),
      join(Fields.EKKEYWORD_ID, Fields.KKEYWORD_ID)
  };

  /**
    Get The where clause for the standard query to get a user's keywords
    @param user Name of the user
    @param startDate Earliest date of events we're interested in.  If null,
      we're interested in all events
    @param endDate Latest date of events we're interested in
    @return The where clause for the standard query to get a user's keywords
   */
  static WhereClause keywordsWhere(User user,
                                   String startDate,
                                   String endDate)
  {
    return eventsBase(user, startDate, endDate, true).
           addClauses(KEYWORD_JOIN_CLAUSES);
  }

  /** @return The where clause for a query to get keywords
       corresponding to a single event
    @param eventid ID of the event
   */
  static WhereClause singleEventKeywordsWhere(int eventid)
  {
    return eventId(eventid).addClause(IS_PUBLIC_EVENT).
                            addClauses(KEYWORD_JOIN_CLAUSES);
  }

  /**
    Get the whereclause that tests that an eventid matches a given id
    @param id Id to match
    @return the whereclause that tests that an eventid matches a given id
   */
  static WhereClause eventId(int id)
  {
    return new WhereClause().addClause(equals(Fields.EEVENT_ID, id + ""));
  }

  /**
    Get the whereclause that tests that an locationid matches a given id
    @param id Id to match
    @return the whereclause that tests that an locationid matches a given id
   */
  static WhereClause locationId(int id)
  {
    return new WhereClause().addClause(equals(Fields.LLOCATION_ID, id + ""));
  }

  /**
    Get the whereclause that tests that an sponsorid matches a given id
    @param id Id to match
    @return the whereclause that tests that an sponsorid matches a given id
   */
  static WhereClause sponsorId(int id)
  {
    return new WhereClause().addClause(equals(Fields.SSPONSOR_ID, id + ""));
  }

  /**
    Get the whereclause that tests that an keywordid matches a given id
    @param id Id to match
    @return the whereclause that tests that an keywordid matches a given id
   */
  static WhereClause keywordId(int id)
  {
    return new WhereClause().addClause(equals(Fields.KKEYWORD_ID, id + ""));
  }

  /**
    Get the whereclause that tests that an instanceid matches a given id
    @param eventid Id to match (note: an id in the events cache, which has
      to be converted to an instanceid)
    @return the whereclause that tests that an instanceid matches a given id
   */
  static WhereClause instanceId(int eventid)
  {
    return new WhereClause().addClause(equals(Fields.MDDETAIL_ID,
        Fields.getDBInstanceId(eventid) + ""));
  }

  /**
    Turn a series of strings into a list suitable for an SQL 'in' clause
    @param strings The strings in question
    @return a list suitable for an SQL 'in' clause
   */
  private static String listify(Iterator strings)
  {
    String list = "(";
    boolean first = true;

    while (strings.hasNext()) {
      if (!first) {
        list += ", ";
      } else {
        first = false;
      }

      list += quote((String) strings.next());
    }

    return list + ")";
  }

  /**
    Get the whereclause for a query that selects a 'schedule' of events that
      contain one of a set of keywords
    @param ids Names of the keywords in the set
    @return the whereclause for a query that selects a 'schedule' of events
      that contain one of a set of keywords
   */
  static WhereClause scheduleWhere(Iterator ids)
  {
    return new WhereClause().addClause(IS_PUBLIC_EVENT).
                             addClauses(KEYWORD_JOIN_CLAUSES).
                             addClause(in(Fields.KWORD, listify(ids)));
  }

  /**
    Get the whereclause for a query that selects a 'schedule' of events for
      a user whose keywords match a particular integer
    @param scheduleType type of the schedule
    @param keywordMatch integer that all keywords should match
    @param user User in question
    @return the whereclause for a query that selects a 'schedule' of events
      for a user whose keywords match a particular integer
   */
  static WhereClause scheduleWhere(char scheduleType,
                                   int keywordMatch,
                                   User user)
  {
    return new WhereClause().addClause(userPubeventsWhere(user)).
                             addClauses(KEYWORD_JOIN_CLAUSES).
                             addClause(equals(Fields.ERPURPOSE,
                                              quote(scheduleType + ""))).
                             addClause(matches(Fields.KWORD,
                                               keywordMatch + "_"));
  }

  /**
    Get the whereclause to choose a user by their id #
    @param userid the id #
    @return the whereclause to choose a user by their id #
   */
  static WhereClause userWhere(int userid)
  {
    return new WhereClause().addClause(equals(Fields.UUSER_ID, userid + ""));
  }

  /**
    Get the whereclause to choose a user by their name
    @param username the user's name
    @return the whereclause to choose a user by their name
   */
  static WhereClause userWhere(String username)
  {
    return new WhereClause().addClause(equals(Fields.UUSERNAME,
                                              quote(username)));
  }

  /**
    Get a subclause that selects a user's subscriptions
    @param user The user
    @return a subclause that selects a user's subscriptions
   */
  static WhereClause subscriptionsWhere(User user)
  {
    return new WhereClause().
        addClause(equals(Fields.UUSERNAME, quote(user.getNameDB()))).
        addClause(join(Fields.CCALENDAR_ID, Fields.SUCALENDAR_ID)).
        addClause(join(Fields.SUUSER_ID, Fields.UUSER_ID));
  }

  /**
    Print where clauses that involve dates
    @param startDate startDate
    @param endDate endDate
   */
  private static void printDatesWhere(String startdate, String enddate)
  {
    System.out.println(
        recurInstancesWhere(new User("gsbarnes"), startdate, enddate).
        toString());
    System.out.println(recurInstancesWhere(new User(), startdate, enddate).
        toString());
    System.out.println(
        userPubRecurInstancesWhere(new User("gsbarnes"), startdate, enddate).
        toString());
    System.out.println(
        userPubeventsWhere(new User("gsbarnes"), startdate, enddate).
        toString());
    System.out.println(
        eventsWhere(new User("gsbarnes"), startdate, enddate, true).
        toString());
    System.out.println(
        eventsWhere(new User(), startdate, enddate, true).
        toString());
    System.out.println(
        eventsWhere(new User("gsbarnes"), startdate, enddate, false).
        toString());
    System.out.println(
        eventsWhere(new User(), startdate, enddate, false).
        toString());
    System.out.println(
        keywordsWhere(new User("gsbarnes"), startdate, enddate).
        toString());
    System.out.println(keywordsWhere(new User(), startdate, enddate).
        toString());
  }

  /** Generate and print all the WhereClauses
    @param args Ignored
   */
  public static void main(String[] args)
  {
    String startdate = "20001225";
    String enddate = "20010101";
    printDatesWhere(startdate, enddate);
    printDatesWhere(null, null);
    System.out.println(locationsWhere(new User("gsbarnes")).toString());
    System.out.println(sponsorsWhere(new User()).toString());
    System.out.println(sponsorsWhere(new User("gsbarnes")).toString());
    System.out.println(keywordsWhere(new User()).toString());
    System.out.println(keywordsWhere(new User("gsbarnes")).toString());
    System.out.println(singleEventWhere(500).toString());
    System.out.println(singleLocationWhere(500).toString());
    System.out.println(singleSponsorWhere(500).toString());
    System.out.println(singleKeywordWhere(500).toString());
    System.out.println(recurInstancesWhere(50).toString());
    System.out.println(recurInstanceWhere(-50).toString());
    System.out.println(singleEventKeywordsWhere(500).toString());
    Vector v = new Vector();
    v.add("dummy1");
    v.add("dummy2");
    System.out.println(scheduleWhere(v.iterator()));
    System.out.println(scheduleWhere('S', 20031, new User("gsbarnes")).toString());
    System.out.println(userWhere("gsbarnes"));
    System.out.println(userWhere(7));
    System.out.println(subscriptionsWhere(new User("gsbarnes")));
  }
}
