//java

package edu.washington.cac.calendar.db;

import java.util.Iterator;
import java.util.Vector;

import edu.washington.cac.calendar.data.User;

/** Common uwcal queries
  @author Greg Barnes
  @version 1.0
 */
public class Queries
{
  /** Field names to appear in the select clause of a user's public events
      query
   */
  private static final SimpleField[] EVENTID_FIELD = { Fields.EEVENT_ID };

  /** Field names to appear in the select clause of a user's
      recurring public events query
   */
  private static final Field[] RECUR_EVENTID_FIELD = {
      Fields.MDDETAIL_ID
  };

  /** Tables to appear in the select clause of a user's public events
      query
   */
  private static final SimpleTable[] USER_PUBEVENTS_TABLES = {
    Tables.EVENTS, Tables.EVENTREFS, Tables.USERS
  };

  /** Tables to appear in the select clause of a user's recurring public events
      query
   */
  private static final SimpleTable[] USER_PUB_RECUR_INSTANCES_TABLES = {
    Tables.EVENTREFS, Tables.USERS, Tables.MASTERDETAILS
  };

  /** A table representing the tables in a user's public events query */
  private static Table USER_PUBEVENTS_TABLE =
      new Table(JoinedQuery.mergedName(USER_PUBEVENTS_TABLES), EVENTID_FIELD);

  /** A table representing the tables in a user's recurring public events
      query */
  private static Table USER_PUB_RECUR_INSTANCES_TABLE =
      new Table(JoinedQuery.mergedName(USER_PUB_RECUR_INSTANCES_TABLES),
                RECUR_EVENTID_FIELD);

  /**
    @return A query to get the eventids of public events a user
      chose to display
    @param user The user
    @param startDate Earliest date we're interested in.  If null, get all
      eventids for the user
    @param endDate Latest date we're interested in
   */
  static Query userPubeventsQuery(User user, String startDate, String endDate)
  {
    return new Query(USER_PUBEVENTS_TABLE,
                     WhereClauses.userPubeventsWhere(user, startDate, endDate));
  }

  /**
    @return A query to get the eventids of recurring public events a user
      chose to display
    @param user The user
    @param startDate Earliest date we're interested in.  If null, get all
      recurring public events
    @param endDate Latest date we're interested in
   */
  static Query userPubRecurInstancesQuery(User user,
                                          String startDate,
                                          String endDate)
  {
    return new Query(USER_PUB_RECUR_INSTANCES_TABLE,
        WhereClauses.userPubRecurInstancesWhere(user, startDate, endDate));
  }

  /** Tables whose fields should appear in the select clause of an
    events query
   */
  private static final SimpleTable[] EVENT_TABLES = {
    Tables.EVENTS, Tables.LOCATIONS, Tables.SPONSORS
  };

  /** Tables which should appear in the from clause of a user's public
    events query, but whose fields should not appear in the select clause
   */
  private static final SimpleTable[] USER_FROM_TABLES = {
    Tables.EVENTREFS, Tables.USERS
  };

  /**
    Get A query to get the events for a user for a string of dates
    @param user The user
    @param startDate Earliest date we're interested in.  If null, get all
      events.
    @param endDate Latest date we're interested in
    @param should data about the event's location and sponsor be retrieved
       as well?
    @return A query to get the events for a user for a string of dates
   */
  static Query eventsQuery(User user,
                           String startDate,
                           String endDate,
                           boolean loadAuxData)
  {
    return loadAuxData ?
        new JoinedQuery(EVENT_TABLES,
            WhereClauses.eventsWhere(user, startDate, endDate, true),
            OrderBys.EVENT_ORDERBY) :
        new Query(Tables.EVENTS,
            WhereClauses.eventsWhere(user, startDate, endDate, false),
            OrderBys.EVENT_ORDERBY);
  }

  /** Tables whose fields should appear in the select clause of a
    recurrence instances query
   */
  private static final SimpleTable[] RECUR_INSTANCES_TABLES = {
    Tables.EVENTS, Tables.MASTERDETAILS
  };

  /**
    Get a query to get recurrence instances for a user for a string of dates

    @param user The user
    @param startDate Earliest date we're interested in.  If null, get all
      recurrence info for the user
    @param endDate Latest date we're interested in
    @return A query to get recurrence instances for a user for a string of dates
   */
  static Query recurInstances(User user, String startDate, String endDate)
  {
    return new JoinedQuery(RECUR_INSTANCES_TABLES,
        WhereClauses.recurInstancesWhere(user, startDate, endDate),
        new OrderBy());
  }

  /**
    Get a query to get all instances of a recurrence

    @param eventid ID of the master event for the recurrence
    @return a query to get all instances of a recurrence
   */
  static Query recurInstances(int eventid)
  {
    return new JoinedQuery(RECUR_INSTANCES_TABLES,
        WhereClauses.recurInstancesWhere(eventid),
        new OrderBy());
  }

  /**
    Get a query to get a single recurrence instance corresponding to
    an eventid

    @param eventid ID in the events cache of the instance
    @return a query to get a single recurrence instance corresponding to
    an eventid
   */
  static Query recurInstance(int eventid)
  {
    return new JoinedQuery(RECUR_INSTANCES_TABLES,
        WhereClauses.recurInstanceWhere(eventid),
        new OrderBy());
  }

  /**
    Get a query that returns a single event
    @param eventid Id of the event
    @return A query to get a single event
   */
  static Query singleEventQuery(int eventid)
  {
    return new JoinedQuery(EVENT_TABLES,
                           WhereClauses.singleEventWhere(eventid),
                           new OrderBy());
  }

  /**
    Get a query that returns a single location
    @return A query to get a single location
    @param locationid Id of the location
   */
  static Query singleLocationQuery(int locationid)
  {
    return new Query(Tables.LOCATIONS,
                     WhereClauses.singleLocationWhere(locationid),
                     new OrderBy());
  }

  /**
    Get a query that returns a single sponsor
    @param sponsorid Id of the sponsor
    @return A query to get a single sponsor
   */
  static Query singleSponsorQuery(int sponsorid)
  {
    return new Query(Tables.SPONSORS,
                     WhereClauses.singleSponsorWhere(sponsorid),
                     new OrderBy());
  }

  /**
    Get a query that returns a single keyword
    @param keywordid Id of the keyword
    @return A query to get a single keyword
   */
  static Query singleKeywordQuery(int keywordid)
  {
    return new Query(Tables.KEYWORDS,
                     WhereClauses.singleKeywordWhere(keywordid),
                     new OrderBy());
  }

  /** Tables whose fields should appear in the select clause of a
    keywords query
   */
  static final SimpleTable[] KEYWORD_TABLES = {
    Tables.EVENTS, Tables.KEYWORDS
  };

  /** Tables to appear in the from clause of a user's keywords query, but
    whose fields should not be in the select clause
   */
  static final SimpleTable[] USER_KEYWORD_FROM_TABLES = {
    Tables.EVENTKEYWORDS
  };

  /** Tables to appear in the from clause of a guest's keywords query, but
    whose fields should not be in the select clause
   */
  static final SimpleTable[] GUEST_KEYWORD_FROM_TABLES = {
    Tables.EVENTKEYWORDS
  };

  /**
    @return A query to get the keywords associated with a user's events
        for a string of dates
    @param user The user
    @param startDate Earliest date we're interested in.  If null, get all
      keyword info
    @param endDate Latest date we're interested in
   */
  static Query keywordsQuery(User user,
                             String startDate,
                             String endDate)
  {
    return new JoinedQuery(KEYWORD_TABLES,
        user.isGuest() ? GUEST_KEYWORD_FROM_TABLES :
                         USER_KEYWORD_FROM_TABLES,
        WhereClauses.keywordsWhere(user, startDate, endDate),
        OrderBys.KEYWORD_ORDERBY);
  }

  /**
    @return A query to get the keywords associated with a single event
    @param eventid ID of the event
   */
  static Query keywordsQuery(int eventid)
  {
    return new JoinedQuery(KEYWORD_TABLES, USER_KEYWORD_FROM_TABLES,
        WhereClauses.singleEventKeywordsWhere(eventid),
        OrderBys.KEYWORD_ORDERBY);
  }

  private static Field[] lastIdField(String fieldName) {
    Field[] field = { new Field("max(" + fieldName + ")") };
    return field;
  }

  private static Query lastId(Field field, String tableName, User user)
  {
    return new Query(
        new Table(tableName, lastIdField(field.getStmtName())),
        new WhereClause().
          addClause(WhereClauses.creatorClause(tableName, user)),
        new OrderBy());
  }

  public static Query lastEventId(User user) {
      return lastId(Fields.EEVENT_ID, Tables.EVENTS_TABLE, user);
  }

  static Query lastLocationId(User user) {
      return lastId(Fields.LLOCATION_ID, Tables.LOCATIONS_TABLE, user);
  }

  static Query lastSponsorId(User user) {
      return lastId(Fields.SSPONSOR_ID, Tables.SPONSORS_TABLE, user);
  }

  static Query lastKeywordId(User user) {
      return lastId(Fields.KKEYWORD_ID, Tables.KEYWORDS_TABLE, user);
  }

  private static Field[] EVENT_ID_FIELD = { Fields.EEVENT_ID };

  static Query simpleEventsQuery(User user)
  {
      return new Query(
        new Table(Tables.EVENTS_TABLE, EVENT_ID_FIELD),
        new WhereClause().
          addClause(WhereClauses.creatorClause(Tables.EVENTS_TABLE, user)));
  }

  /**
    Get a query that selects all of a user's locations
    @param user The user
    @return a query that selects all of a user's locations
   */
  static Query locationsQuery(User user)
  {
    return new Query(Tables.LOCATIONS, WhereClauses.locationsWhere(user),
                     OrderBys.LOCATION_ORDERBY);
  }

  /**
    Get a query that selects all of a user's sponsors
    @param user The user
    @return a query that selects all of a user's sponsors
   */
  static Query sponsorsQuery(User user)
  {
    return new Query(Tables.SPONSORS, WhereClauses.sponsorsWhere(user),
                     OrderBys.SPONSOR_ORDERBY);
  }

  /**
    Get a query that selects all of a user's keywords
    @param user The user
    @return a query that selects all of a user's keywords
   */
  static Query keywordsQuery(User user)
  {
    return new Query(Tables.KEYWORDS, WhereClauses.keywordsWhere(user),
                     OrderBys.KEYWORD_ORDERBY);
  }

  /** Tables to appear in the select clause of a query to get eventids to
      add a schedule
   */
  private static final SimpleTable[] SCHEDULE_TABLES1 = {
    Tables.EVENTS
  };

  /** Tables to appear in the from clause of a query to get eventids to
      add a schedule
   */
  private static final SimpleTable[] SCHEDULE_TABLES1_FROM = {
    Tables.KEYWORDS, Tables.EVENTKEYWORDS
  };

  /** Tables to appear in the from clause of a query to get eventids to remove
      a schedule
   */
  private static final SimpleTable[] SCHEDULE_TABLES2_NAMES = {
    Tables.EVENTS, Tables.EVENTREFS, Tables.USERS, Tables.KEYWORDS,
    Tables.EVENTKEYWORDS
  };

  /** A table representing the tables in a query to get eventids to remove
      a schedule */
  private static Table SCHEDULE_TABLES2 =
      new Table(JoinedQuery.mergedName(SCHEDULE_TABLES2_NAMES), EVENTID_FIELD);

  /**
    Get a query that selects a 'schedule' of events that contain one of
      a set of keywords
    @param ids Names of the keywords in the set
    @return a query that selects a 'schedule' of events that contain one of
      a set of keywords
   */
  static Query scheduleQuery(Iterator ids)
  {
    return new JoinedQuery(SCHEDULE_TABLES1, SCHEDULE_TABLES1_FROM,
                           WhereClauses.scheduleWhere(ids),
                           new OrderBy());
  }

  /**
    Get a query that selects a 'schedule' of events for a user
      whose keywords match a particular integer
    @param scheduleType type of the schedule
    @param keywordMatch integer that all keywords should match
    @param user User in question
    @return a query that selects a 'schedule' of events for a user
      whose keywords match a particular integer
   */
  static Query scheduleQuery(char scheduleType, int keywordMatch, User user)
  {
    return new Query(SCHEDULE_TABLES2,
        WhereClauses.scheduleWhere(scheduleType, keywordMatch, user),
        new OrderBy());
  }

  /**
    Get a query that selects a user with a given id
    @param userid the id
    @return a query that selects a user with a given id
   */
  static Query userQuery(int userid)
  {
    return new Query(Tables.USERS, WhereClauses.userWhere(userid));
  }

  /**
    Get a query that selects a user with a given name
    @param username the name
    @return a query that selects a user with a given name
   */
  static Query userQuery(String username)
  {
    return new Query(Tables.USERS, WhereClauses.userWhere(username));
  }

  /** Tables to appear in the select clause of a user's subscriptions
      query
   */
  private static final SimpleTable[] SUBSCRIPTIONS_TABLES = {
    Tables.CALENDARS, Tables.SUBSCRIPTIONS, Tables.USERS
  };

  /**
    Get a query that selects a user's subscriptions to public calendars
    @param user The user
    @return a query that selects a user's subscriptions to public calendars
   */
  static Query subscriptionsQuery(User user)
  {
    return new JoinedQuery(SUBSCRIPTIONS_TABLES,
                           WhereClauses.subscriptionsWhere(user),
                           new OrderBy());
  }

  /**
    Generate and print all of the queries
    @param args Ignored
  /**
    Get a query that selects all the public calendars
    @return a query that selects all the public calendars
   */
  static Query calendarsQuery()
  {
    return new Query(Tables.CALENDARS, new WhereClause(),
                     OrderBys.CALENDAR_ORDERBY);
  }

  /**
    Generate and print all of the queries
    @param args Ignored
   */
  public static void main(String args[])
  {
    System.out.println(locationsQuery(new User("gsbarnes")).toString());
    System.out.println(sponsorsQuery(new User()).toString());
    System.out.println(keywordsQuery(new User()).toString());

    String startdate = "20001225";
    String enddate = "20010101";
    System.out.println(
      recurInstances(new User("gsbarnes"), startdate, enddate).
      toString());
    System.out.println(
      recurInstances(new User(), startdate, enddate).
      toString());
    System.out.println(
      userPubRecurInstancesQuery(new User("gsbarnes"), startdate, enddate).
      toString());
    System.out.println(
      userPubeventsQuery(new User("gsbarnes"), startdate, enddate).toString());
    System.out.println(
      eventsQuery(new User("gsbarnes"), startdate, enddate, true).toString());
    System.out.println(
      eventsQuery(new User(), startdate, enddate, true).toString());
    System.out.println(
      eventsQuery(new User("gsbarnes"), startdate, enddate, false).toString());
    System.out.println(
      eventsQuery(new User(), startdate, enddate, false).toString());
    System.out.println(
      keywordsQuery(new User("gsbarnes"), startdate, enddate).toString());
    System.out.println(
      keywordsQuery(new User(), startdate, enddate).toString());
    System.out.println(lastEventId(new User("gsbarnes")));
    System.out.println(lastLocationId(new User("gsbarnes")));
    System.out.println(lastSponsorId(new User("gsbarnes")));
    System.out.println(lastKeywordId(new User("gsbarnes")));
    System.out.println(simpleEventsQuery(new User("gsbarnes")));
    System.out.println(singleEventQuery(500));
    System.out.println(singleLocationQuery(500));
    System.out.println(singleSponsorQuery(500));
    System.out.println(singleKeywordQuery(500));
    System.out.println(recurInstances(50));
    System.out.println(recurInstance(-50));
    System.out.println(keywordsQuery(500));
    Vector v = new Vector();
    v.add("dummy1");
    v.add("dummy2");
    System.out.println(scheduleQuery(v.iterator()));
    System.out.println(scheduleQuery('S', 20031, new User("gsbarnes")));
    System.out.println(userQuery("gsbarnes"));
    System.out.println(userQuery(7));
    System.out.println(subscriptionsQuery(new User("gsbarnes")));
  }
}
