// java

package edu.washington.cac.calendar.db;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.User;

/** Creates SQL statements for UWCal pubevents
    @author Greg Barnes, Leman Chung
    @version 1.3
  */
public class PubEventsQueries {
  private OrderBy ORDER_BY =
      new OrderBy().addField(Fields.ESTARTDATE).addField(Fields.ESHORTDESC);

  /**
    Get the basic where clauses for most queries
    @return the basic where clauses for most queries
   */
  public WhereClause baseStandardWhere()
  {
    return new WhereClause().
           addClause(WhereClauses.IS_PUBLIC_EVENT).
           addClause("(" + WhereClauses.creatorClause(Tables.KEYWORDS_TABLE,
               DBResources.officialKeywordCreator()) + " or " +
               WhereClauses.creatorClause(Tables.KEYWORDS_TABLE,
               DBResources.unofficialKeywordCreator()) + ")");
  }

  /**
    Get the where clauses that appear in almost all public events queries,
      including joins
    @return Where clauses that appear in almost all public events queries,
      including joins
   */
  private WhereClause standardWhereClause()
  {
    return baseStandardWhere().
        addClauses(WhereClauses.KEYWORD_JOIN_CLAUSES);
  }

  private Query standardQuery(WhereClause whereClause) {
      return new JoinedQuery(Queries.KEYWORD_TABLES,
                             Queries.USER_KEYWORD_FROM_TABLES,
                             whereClause, ORDER_BY);
  }

  public Query adminListUpdatesQuery(User creator)
  {
      return standardQuery(standardWhereClause().
        addClause(WhereClauses.eventsCreatorClause(creator)));
  }

  /**
    Get a query for admin updates by date
    @param startCal Starting date of the query
    @param endCal Ending date of the query
    @return The query
   */
  public Query adminUpViaDatesQuery(MyCalendar startCal,
                                    MyCalendar endCal)
  {
    return standardQuery(standardWhereClause()
      .addClause(
        WhereClauses.dateClause(Fields.ESTARTDATE, ">=",
          startCal.getDateDigits()))
      .addClause(
        WhereClauses.dateClause(Fields.ESTARTDATE, "<=",
          endCal.getDateDigits())));
  }

  public Query listUpdatesQuery(User user)
  {
    return adminListUpdatesQuery(user);
  }

  /**
    Get a query to select the fields from the events table for a single event
    @param id Id of the event
    @return a query to select a single event
   */
  public Query selectAnEvent(int id)
  {
    return new Query(Tables.EVENTS, WhereClauses.eventId(id));
  }

  /**
    Print out the queries we define
    @param args <code>uid, startdate, enddate</code>, where dates are 8-digit
       dates (YYYYMMDD)
    @exception edu.washington.cac.calendar.data.CaldataException If the
        dates are in a bad format
   */
  public static void main(String[] args)
    throws edu.washington.cac.calendar.data.CaldataException
  {
    if (args.length < 3) {
      System.err.println(
        "Usage:  PubEventsQueries <uid> <startdate> <enddate>");
      System.exit(1);
    }

    PubEventsQueries peq = new PubEventsQueries();

    System.out.println(peq.adminListUpdatesQuery(new User(args[0])).toString());
    System.out.println(
        peq.adminUpViaDatesQuery(new MyCalendar(args[1]),
                                 new MyCalendar(args[2])).toString());
    System.out.println(peq.listUpdatesQuery(new User(args[0])).toString());
    System.out.println(peq.selectAnEvent(42).toString());
  }
}
