//java

package edu.washington.cac.calendar.db;

/** An object representing a query on multiple joined
    tables in the database
    @author Greg Barnes
    @version 1.2
  */
public class JoinedQuery extends Query 
{
  /**
    Create a new query
    @param selectTables Tables whose fields should appear in the select clause
    @param fromTables Tables whose fields should only appear in the from
      clause
    @param where WhereClause, which should include joins on all the tables
      in the first two parameters
    @param orderBy how to order the results
   */
  protected JoinedQuery(SimpleTable[] selectTables, 
                        SimpleTable[] fromTables, 
                        WhereClause where,
                        OrderBy orderBy) 
  {
    super(newTable(selectTables, fromTables), where, orderBy);
  }

  /**
    Create a new query
    @param selectTables Tables whose fields should appear in the select clause
    @param where WhereClause, which should include joins on all the tables
    @param orderBy how to order the results
   */
  protected JoinedQuery(SimpleTable[] selectTables, 
                        WhereClause where,
                        OrderBy orderBy) 
  {
    this(selectTables, new SimpleTable[0], where, orderBy);
  }

  /**
    @return a new Table that contains all the tables in the query
    @param selectTables Tables whose fields should appear in the select clause
    @param fromTables Tables whose fields should only appear in the from
      clause
   */
  private static SimpleTable newTable(SimpleTable[] selectTables,
                                      SimpleTable[] fromTables) 
  {
    SimpleTable s = new SimpleTable(mergedName(selectTables, fromTables), "");
    s.setFields(mergedFieldLists(selectTables));
    return s;
  }

  private static int totalFields(SimpleTable[] tables) {
    int i;
    int length = 0;

    for (i=0; i < tables.length; i++) {
      length += tables[i].getFields().length;
    }

    return length;
  }

  private static Field[] mergedFieldLists(SimpleTable[] tables) {
    Field[] newFields = new Field[totalFields(tables)];

    int i, j, offset = 0;
    
    for (i = 0; i < tables.length; i++) {
      for (j = 0; j < tables[i].getFields().length; j++) {
        newFields[offset++] = tables[i].getFields()[j];
      }
    }

    return newFields;
  }

  /**
    @return a new 'table name' consisting of a list of all the tables 
      in the argument
    @param tables A set of tables
   */
  static String mergedName(SimpleTable[] tables)
  {
    int i;
    StringBuffer sb = new StringBuffer();

    for (i = 0; i < tables.length; i++) {
      if (i > 0) {
        sb.append(", ");
      }

      sb.append(tables[i].fromName());
    }

    return sb.toString();
  }

  /**
    @return a new 'table name' consisting of a list of all the tables 
      in all the arguments
    @param tables1 A set of tables
    @param tables2 A set of tables
   */
  private static String mergedName(SimpleTable[] tables1,
                                   SimpleTable[] tables2)
  {
    if (tables2.length == 0) {
      return mergedName(tables1);
    } else if (tables1.length == 0) {
      return mergedName(tables2);
    } else {
      return mergedName(tables1) + ", " + mergedName(tables2);
    }
  }
}
