//java

package edu.washington.cac.calendar.db;

import java.sql.ResultSet;
import java.sql.SQLException;

/** Common fields in the database
   @author Greg Barnes
   @version 2.2
 */
public class Fields
{
  // field names that appear in multiple tables

  /** field name for an event id */
  static final String EVENT_ID = "eventid";

  /** field name for a location id */
  static final String LOCATION_ID = "locationid";

  /** field name for a keyword  id */
  static final String KEYWORD_ID = "keywordid";

  /** field name for a sponsor id */
  static final String SPONSOR_ID = "sponsorid";

  /** field name for a user id */
  static final String USER_ID = "userid";

  /** field name for a calendar id */
  static final String CALENDAR_ID = "calendarid";

  /** field name for an entity's creator */
  static final String CREATOR = "creator";

  /** field name for an email address */
  static final String EMAIL = "email";

  /** field name for a url */
  static final String LINK = "link";

  /** field name for a long description of an entity */
  static final String LONGDESC = "longdesc";

  /** field name for the continuation of a long description */
  static final String LONGDESC1 = "longdesc1";        // continuation

  /** field name for the name of something */
  static final String NAME = "name";

  /** field name for a phone number */
  static final String PHONE = "phone";

  /** field name for a flag indicating whether an entity is public */
  static final String PUBLIC = "publicf";

  /** field name for a start date */
  static final String STARTDATE = "startdate";

  /** field name for a start time */
  static final String STARTTIME = "starttime";

  /** field name for an end date */
  static final String ENDDATE = "enddate";

  /** field name for an end time */
  static final String ENDTIME = "endtime";

  /** field name for a creation timestamp */
  static final String CREATED = "created";

  //%%% the following 3 should be in a Properties file, not here

  /** The number of characters in a short db VARCHAR field. */
  static final int SHORT_VARCHAR = 20;

  /** The number of characters in a long db VARCHAR field. */
  static final int LONG_VARCHAR = 200;

  /** The maximum number of characters in a db VARCHAR field. */
  static final int MAX_VARCHAR = 255;

    // fields in the events table

  /** event id field in the events table */
  public static final SimpleField EEVENT_ID =
    new SimpleField(EVENT_ID, Tables.EVENTS);

  /** column name for last modified time in the events table */
  public static final Field ELASTMOD = new Field("lastmod");

  /** column name for sequence number in the events table */
  public static final Field ESEQ = new Field("seq");

  /** column name for the public field in the events table */
  public static final SimpleField EPUBLIC =
    new SimpleField(PUBLIC, Tables.EVENTS);

  /** column name for creation time in the events table */
  // This name may also be used by users.'create', so make it a SimpleField
  public static final SimpleField ECREATED =
    new SimpleField(CREATED, Tables.EVENTS);

  /** column name for start date in the events table */
  public static final SimpleField ESTARTDATE =
      new SimpleField(STARTDATE, Tables.EVENTS);

  /** column name for start time in the events table */
  public static final SimpleField ESTARTTIME =
      new SimpleField(STARTTIME, Tables.EVENTS);

  /** column name for end date in the events table */
  public static final SimpleField EENDDATE =
      new SimpleField(ENDDATE, Tables.EVENTS);

  /** column name for end time in the events table */
  public static final SimpleField EENDTIME =
      new SimpleField(ENDTIME, Tables.EVENTS);

  /** column name for short description in the events table */
  public static final VarcharField ESHORTDESC =
    new VarcharField("shortdesc", Tables.EVENTS, LONG_VARCHAR);

  /** column name for a longdesc in the events table */
  public static final VarcharField ELONGDESC =
    new VarcharField(LONGDESC, Tables.EVENTS, MAX_VARCHAR);

  /** column name for a longdesc continuation in the events table */
  public static final VarcharField ELONGDESC1 =
    new VarcharField(LONGDESC1, Tables.EVENTS, MAX_VARCHAR);

  /** column name for longdesc continuation II in the events table */
  public static final Field ELONGDESC2 = new Field("longdesc2");

  /** column name for a link in the events table */
  public static final VarcharField ELINK =
    new VarcharField(LINK, Tables.EVENTS, MAX_VARCHAR);

  /** column name for status in the events table */
  public static final Field ESTATUS = new Field("status");

  /** column name for the location id in the events table */
  public static final SimpleField ELOCATION_ID =
    new SimpleField(LOCATION_ID, Tables.EVENTS);

  /** column name for the sponsor id in the events table */
  public static final SimpleField ESPONSOR_ID =
    new SimpleField(SPONSOR_ID, Tables.EVENTS);

  /** column name for a creator in the events table */
  public static final VarcharField ECREATOR =
    new VarcharField(CREATOR, Tables.EVENTS, SHORT_VARCHAR);

  /** column name for cost in the events table */
  public static final VarcharField ECOST =
    new VarcharField("cost", Tables.EVENTS, MAX_VARCHAR);

  /** column name for recurring_flag in the events table */
  public static final Field ERECURRING_STATUS = new Field("recurring_status");

  static final Field[] EVENT_FIELDS = {
    EEVENT_ID, ELASTMOD, ESEQ, EPUBLIC,
    ECREATED,
    ESTARTDATE, ESTARTTIME, EENDDATE, EENDTIME,
    ESHORTDESC, ELONGDESC, ELONGDESC1, ELONGDESC2,
    ELINK, ELOCATION_ID, ESPONSOR_ID,
    ECREATOR, ECOST, ERECURRING_STATUS
  };

  /**
    Extract the longdesc field for an event from the current
    row in a resultset

    @param rs The ResultSet with the event
    @return The longdesc
    @exception SQLException If there's a db problem
   */
  public static String getEventLongdesc(ResultSet rs) throws SQLException
  {
    if (rs.getString(ELONGDESC1.getName()) == null) {
      return rs.getString(ELONGDESC.getName());
    } else {
      return rs.getString(ELONGDESC.getName()) +
             rs.getString(ELONGDESC1.getName());
    }
  }

  /**
      Extract the longdesc field for the database from an entry's
      longdesc field.  The database may have a limit on string lengths,
      so an entry's longdesc can be split into two parts,
      <code>longdesc</code>, and <code>longdesc1</code>

      @param longdesc The entries longdesc
      @return The first part of the longdesc, suitable for the db's
         <code>longdesc</code> field
    */
  public static String getDBLongdesc(String longdesc)
  {
    return ELONGDESC.sizeToFit(longdesc);
  }

  /**
      Extract the longdesc1 field for the database from an entry's
      longdesc field.  The database has a low limit on string lengths,
      so an entry's longdesc can be split into two parts,
      <code>longdesc</code>, and <code>longdesc1</code>

      @param longdesc The entries longdesc
      @return The second part of the longdesc, suitable for the db's
        <code>longdesc1</code> field.  Null if the whole longdesc fits
        in the field
    */
  public static String getDBLongdesc1(String longdesc)
  {
    if (longdesc != null
        && ELONGDESC.indexOfLastDBChar(longdesc) != longdesc.length() - 1) {
      return ELONGDESC1.sizeToFit(
        longdesc.substring(ELONGDESC.indexOfLastDBChar(longdesc) + 1));
    } else {
      return null;
    }
  }

    // fields in the event_keywords table

  /** column name for an event id in the event_keywords table */
  public static final SimpleField EKEVENT_ID =
    new SimpleField(EVENT_ID, Tables.EVENTKEYWORDS);

  /** column name for the keyword id in the event_keywords table */
  public static final SimpleField EKKEYWORD_ID =
    new SimpleField(KEYWORD_ID, Tables.EVENTKEYWORDS);

  static final Field[] EVENTKEYWORD_FIELDS = {
    EKEVENT_ID, EKKEYWORD_ID
  };

    // fields in the keywords table

  /** column name for the keyword id in the keywords table */
  public static final SimpleField KKEYWORD_ID =
    new SimpleField(KEYWORD_ID, Tables.KEYWORDS);

  /** column name for main keyword description */
  public static final VarcharField KWORD =
    new VarcharField("word", Tables.KEYWORDS, LONG_VARCHAR);

  /** column name for a longdesc in the keywords table */
  public static final VarcharField KLONGDESC =
    new VarcharField(LONGDESC, Tables.KEYWORDS, MAX_VARCHAR);

  /** column name for longdesc continuation in the keywords table */
  public static final VarcharField KLONGDESC1 =
    new VarcharField(LONGDESC1, Tables.KEYWORDS, MAX_VARCHAR);

  /** column name for a creator in the keywords table */
  public static final VarcharField KCREATOR =
    new VarcharField(CREATOR, Tables.KEYWORDS, SHORT_VARCHAR);

  /** column name for the public field in the keywords table */
  public static final SimpleField KPUBLIC =
    new SimpleField(PUBLIC, Tables.KEYWORDS);

  static final Field[] KEYWORD_FIELDS = {
    KKEYWORD_ID, KWORD, KLONGDESC, KLONGDESC1,
    KCREATOR, KPUBLIC
  };

    // fields in the sponsors table

  /** column name for the sponsor id in the sponsors table */
  public static final SimpleField SSPONSOR_ID =
    new SimpleField(SPONSOR_ID, Tables.SPONSORS);

  /** column name for sponsor name */
  public static final VarcharField SNAME =
    new VarcharField(NAME, Tables.SPONSORS, LONG_VARCHAR);

  /** column name for phone in the sponsors table */
  public static final VarcharField SPHONE =
    new VarcharField(PHONE, Tables.SPONSORS, MAX_VARCHAR);

  /** column name for email in the sponsors table */
  public static final VarcharField SEMAIL =
    new VarcharField(EMAIL, Tables.SPONSORS, MAX_VARCHAR);

  /** column name for a link in the sponsors table */
  public static final VarcharField SLINK =
    new VarcharField(LINK, Tables.SPONSORS, MAX_VARCHAR);

  /** column name for a creator in the sponsors table */
  public static final VarcharField SCREATOR =
    new VarcharField(CREATOR, Tables.SPONSORS, SHORT_VARCHAR);

  /** column name for the public field in the sponsors table */
  public static final SimpleField SPUBLIC =
    new SimpleField(PUBLIC, Tables.SPONSORS);

  static final Field[] SPONSOR_FIELDS = {
    SSPONSOR_ID, SNAME, SPHONE, SEMAIL,
    SLINK, SCREATOR, SPUBLIC
  };

    // fields in the locations table

  /** column name for the location id in the locations table */
  public static final SimpleField LLOCATION_ID =
    new SimpleField(LOCATION_ID, Tables.LOCATIONS);

  /** column name for an address in the locations table */
  public static final VarcharField LADDRESS =
    new VarcharField("address", Tables.LOCATIONS, LONG_VARCHAR);

  /** column name for a subaddress in the locations table */
  public static final VarcharField LSUBADDRESS =
    new VarcharField("subaddress", Tables.LOCATIONS, MAX_VARCHAR);

  /** column name for a link in the locations table */
  public static final VarcharField LLINK =
    new VarcharField(LINK, Tables.LOCATIONS, MAX_VARCHAR);

  /** column name for a creator in the locations table */
  public static final VarcharField LCREATOR =
    new VarcharField(CREATOR, Tables.LOCATIONS, SHORT_VARCHAR);

  /** column name for the public field in the locations table */
  public static final SimpleField LPUBLIC =
    new SimpleField(PUBLIC, Tables.LOCATIONS);

  static final Field[] LOCATION_FIELDS = {
    LLOCATION_ID, LADDRESS, LSUBADDRESS, LLINK,
    LCREATOR, LPUBLIC
  };

    // fields in the users table

  /** column name for the username field in the users table */
  public static final SimpleField UUSERNAME =
    new SimpleField("username", Tables.USERS);

  /** column name for the userid field in the users table */
  public static final SimpleField UUSER_ID =
    new SimpleField(USER_ID, Tables.USERS);

  /** column name for the create field in the users table */
  public static final SimpleField UCREATED =
    new SimpleField(CREATED, Tables.USERS);

  /** column name for the last login field in the users table */
  public static final Field ULAST_LOGON = new Field("last_logon");

  /** column name for the last access field in the users table */
  public static final Field ULAST_ACCESS = new Field("last_access");

  /** column name for the last modify field in the users table */
  public static final Field ULAST_MODIFY = new Field("last_modify");

  /** All fields in the users table */
  static final Field[] USER_FIELDS = {
    UUSERNAME, UUSER_ID, UCREATED, ULAST_LOGON,
    ULAST_ACCESS, ULAST_MODIFY
  };

    // fields in the eventrefs table

  /** column name for the eventid field in the eventrefs table */
  public static final SimpleField EREVENT_ID =
    new SimpleField(EVENT_ID, Tables.EVENTREFS);

  /** column name for the userid field in the eventrefs table */
  public static final SimpleField ERUSER_ID =
    new SimpleField(USER_ID, Tables.EVENTREFS);

  /** column name for the purpose field in the eventrefs table */
  public static final Field ERPURPOSE = new Field("purpose");

  /** All fields in the eventrefs table */
  static final Field[] EVENTREF_FIELDS = {
    EREVENT_ID, ERUSER_ID, ERPURPOSE
  };

    // fields in the master/detail table

  /** column name for the detail eventid field in the master/detail table */
  public static final Field MDDETAIL_ID = new Field("detailid");

  /** column name for the master eventid field in the master/detail table */
  public static final Field MDMASTER_EVENT_ID = new Field("master_eventid");

  /** column name for start date in the master/detail table */
  public static final SimpleField MDSTARTDATE =
      new SimpleField(STARTDATE, Tables.MASTERDETAILS);

  /** column name for start time in the master/detail table */
  public static final SimpleField MDSTARTTIME =
      new SimpleField(STARTTIME, Tables.MASTERDETAILS);

  /** column name for end date in the master/detail table */
  public static final SimpleField MDENDDATE =
      new SimpleField(ENDDATE, Tables.MASTERDETAILS);

  /** column name for end time in the master/detail table */
  public static final SimpleField MDENDTIME =
      new SimpleField(ENDTIME, Tables.MASTERDETAILS);

  /** All fields in the master/details table */
  static final Field[] MASTERDETAILS_FIELDS = {
    MDDETAIL_ID, MDMASTER_EVENT_ID, MDSTARTDATE, MDSTARTTIME, MDENDDATE,
    MDENDTIME
  };

  /**
    Get the eventid for an instance of a recurring event.  Note that instance
    id's might clash with standard event ids in the database, as they are
    two different fields.  The values returned by this method will not,
    however, clash with eventids in the database, so they can be used as
    ids in the Events cache.

    @param rs ResultSet from a query that returned recurring event instances
    @return the eventid for an instance of a recurring event
    @exception SQLException If there's a db problem
   */
  /* to avoid clashes between detail ids and eventids, we negate
     detailids in the events cache
   */
  public static int getInstanceId(ResultSet rs) throws SQLException
  {
    return -rs.getInt(MDDETAIL_ID.getName());
  }

  /**
    Get the instanceid for an instance of a recurring event in the
    database, given the
    eventid assigned in the Events cache.
    @param instanceid ID used in the events cache
    @return the instanceid used in the database
    @see #getInstanceId(ResultSet)
   */
  public static int getDBInstanceId(int instanceid)
  {
    return -instanceid;
  }

  // fields in the authorization table

  /** column name for the userid in the authorization table */
  public static final SimpleField AUSERID =
    new SimpleField(USER_ID, Tables.AUTH);

  /** column name for the user type in the authorization table */
  public static final Field AUSERTYPE = new Field("usertype");

  /** column name for the last name in the authorization table */
  public static final VarcharField ALASTNAME =
    new VarcharField("lastname", Tables.AUTH, MAX_VARCHAR);

  /** column name for the first name in the authorization table */
  public static final VarcharField AFIRSTNAME =
    new VarcharField("firstname", Tables.AUTH, MAX_VARCHAR);

  /** column name for the phone in the authorization table */
  public static final VarcharField APHONE =
    new VarcharField(PHONE, Tables.AUTH, MAX_VARCHAR);

  /** column name for the email in the authorization table */
  public static final VarcharField AEMAIL =
    new VarcharField(EMAIL, Tables.AUTH, MAX_VARCHAR);

  /** column name for the department in the authorization table */
  public static final VarcharField ADEPARTMENT =
    new VarcharField("department", Tables.AUTH, MAX_VARCHAR);

  /** All the fields in the authorization table */
  static final Field[] AUTH_FIELDS = {
    AUSERID, AUSERTYPE, ALASTNAME, AFIRSTNAME,
    APHONE, AEMAIL, ADEPARTMENT
  };

  // fields in the calendars table

  /** column name for the calendarid in the calendars table */
  public static final Field CCALENDAR_ID =
    new SimpleField(CALENDAR_ID, Tables.CALENDARS);

  /** column name for the calendar name in the calendars table */
  public static final VarcharField CNAME =
    new VarcharField(NAME, Tables.CALENDARS, LONG_VARCHAR);

  /** column name for the calendar type in the calendars table */
  public static final Field CTYPE = new Field("type");

  /** column name for the ref in the calendars table */
  public static final Field CREF_NUM = new Field("ref_num");

  /** column name for the parent calendarid in the calendars table */
  public static final Field CPARENT = new Field("parent");

  /** All the fields in the calendars table */
  static final Field[] CALENDAR_FIELDS = {
    CCALENDAR_ID, CNAME, CTYPE, CREF_NUM, CPARENT
  };

  // fields in the subscriptions table

  /** column name for the calendarid in the subscriptions table */
  public static final Field SUCALENDAR_ID =
    new SimpleField(CALENDAR_ID, Tables.SUBSCRIPTIONS);

  /** column name for the userid field in the subscriptions table */
  public static final SimpleField SUUSER_ID =
    new SimpleField(USER_ID, Tables.SUBSCRIPTIONS);

  /** All the fields in the subscriptions table */
  static final Field[] SUBSCRIPTION_FIELDS = {
    SUCALENDAR_ID, SUUSER_ID
  };

  // fields in the redalerts table

  /** column name for an event id in the redalerts table */
  public static final SimpleField REVENT_ID =
    new SimpleField(EVENT_ID, Tables.REDALERTS);

  /** All the fields in the redalerts table */
  static final Field[] REDALERT_FIELDS = {
    REVENT_ID
  };

  // fields in the markedevents table

  /** column name for an event id in the markedevents table */
  public static final SimpleField MEVENT_ID =
    new SimpleField(EVENT_ID, Tables.MARKEDEVENTS);

  /** All the fields in the markedevents table */
  static final Field[] MARKEDEVENT_FIELDS = {
    MEVENT_ID
  };

  // associate the sets of fields with the tables
  static {
    Tables.EVENTS.setFields(EVENT_FIELDS);
    Tables.EVENTKEYWORDS.setFields(EVENTKEYWORD_FIELDS);
    Tables.KEYWORDS.setFields(KEYWORD_FIELDS);
    Tables.LOCATIONS.setFields(LOCATION_FIELDS);
    Tables.SPONSORS.setFields(SPONSOR_FIELDS);
    Tables.USERS.setFields(USER_FIELDS);
    Tables.EVENTREFS.setFields(EVENTREF_FIELDS);
    Tables.MASTERDETAILS.setFields(MASTERDETAILS_FIELDS);
    Tables.AUTH.setFields(AUTH_FIELDS);
    Tables.CALENDARS.setFields(CALENDAR_FIELDS);
    Tables.SUBSCRIPTIONS.setFields(SUBSCRIPTION_FIELDS);
    Tables.REDALERTS.setFields(REDALERT_FIELDS);
    Tables.MARKEDEVENTS.setFields(MARKEDEVENT_FIELDS);
  }

  /**
    Test the getDBLongdesc routines
    @param args First argument should be a <em>longdesc</em> to be split
       up by the methods
   */
  public static void main(String[] args)
  {
    System.err.println("XX" + getDBLongdesc(args[0]) + "XX");
    System.err.println("XX" + getDBLongdesc1(args[0]) + "XX");
  }
}
