// java

package edu.washington.cac.calendar.db;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.data.CaldataException;
import edu.washington.cac.calendar.data.EntityRef;
import edu.washington.cac.calendar.data.Event;
import edu.washington.cac.calendar.data.EventRef;
import edu.washington.cac.calendar.data.Events;
import edu.washington.cac.calendar.data.ItemAccessException;
import edu.washington.cac.calendar.data.ItemException;
import edu.washington.cac.calendar.data.Keyword;
import edu.washington.cac.calendar.data.Keywords;
import edu.washington.cac.calendar.data.Location;
import edu.washington.cac.calendar.data.Locations;
import edu.washington.cac.calendar.data.NoSuchItemException;
import edu.washington.cac.calendar.data.PersonalEvents;
import edu.washington.cac.calendar.data.PublicEvents;
import edu.washington.cac.calendar.data.Recurrence;
import edu.washington.cac.calendar.data.Sponsor;
import edu.washington.cac.calendar.data.Sponsors;
import edu.washington.cac.calendar.data.User;
import edu.washington.cac.calendar.filter.Calendar;
import edu.washington.cac.calendar.filter.CalendarRef;
import edu.washington.cac.calendar.filter.PublicCalendars;

//%%% code that generates statements badly needs to be refactored
//%%% this code is not threadsafe

/** Accesses a calendar database for UWCal
  @author Greg Barnes
  @version 2.7
 */
public class Caldata
{
  /** Connection to the database */
  private CalConnection cconn;

  /** Get a Caldata object
   */
  public Caldata()
  {
    this.cconn = new CalConnection();
  }

  /** load the data corresponding to a user's events for a given date

    @param user the user
    @param date Date in question, in the form <code>YYYYMMDD</code>
    @param es holds the Events
    @param ls holds associated Locations
    @param ss holds associated Sponsors
    @param ks holds the Keywords

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadDaysEvents(User user,
                             String date,
                             Events es,
                             Locations ls,
                             Sponsors ss,
                             Keywords ks)
    throws SQLException, ItemException
  {
    loadEvents(user, date, date, es, ls, ss, ks, true);
  }

  /** load the data corresponding to a user's events for a given date

    @param user the user
    @param date Date in question, in the form <code>YYYYMMDD</code>
    @param es holds the Events
    @param ls holds associated Locations
    @param ss holds associated Sponsors
    @param ks holds the Keywords
    @param loadAllInfo if all information about events should be loaded
           If false, only information suitable for displaying a short
           version of events is loaded

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadDaysEvents(User user,
                             String date,
                             Events es,
                             Locations ls,
                             Sponsors ss,
                             Keywords ks,
                             boolean loadAllInfo)
    throws SQLException, ItemException
  {
    loadEvents(user, date, date, es, ls, ss, ks, loadAllInfo);
  }

  /** load the data corresponding to a user's events for a given
    series of dates

    @param user the user
    @param startDate first date in the series, in the form
      <code>YYYYMMDD</code>.  If null, load all events
    @param endDate last date in the series, in the form
      <code>YYYYMMDD</code>
    @param es holds the Events
    @param ls holds associated Locations
    @param ss holds associated Sponsors
    @param ks holds the Keywords

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadEvents(User user,
                         String startDate,
                         String endDate,
                         Events es,
                         Locations ls,
                         Sponsors ss,
                         Keywords ks)
    throws SQLException, ItemException
  {
    loadEvents(user, startDate, endDate, es, ls, ss, ks, true);
  }

  /**
    Should the events query load location and sponsor data as well?
    @param ls Locations cache
    @param ss Sponsors cache
    @return Should the events query load location and sponsor data as well?
   */
  private boolean shouldLoadAuxData(Locations ls, Sponsors ss)
  {
    return !ls.isLoaded() || !ss.isLoaded();
  }

  /** load the data corresponding to a user's events for a given
    series of dates

    @param user the user
    @param startDate first date in the series, in the form
      <code>YYYYMMDD</code>.  If null, load all events
    @param endDate last date in the series, in the form
      <code>YYYYMMDD</code>
    @param es holds the Events
    @param ls holds associated Locations
    @param ss holds associated Sponsors
    @param ks holds the Keywords
    @param loadAllInfo if all information about events should be loaded
           If false, only information suitable for displaying a short
           version of events is loaded

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadEvents(User user,
                         String startDate,
                         String endDate,
                         Events es,
                         Locations ls,
                         Sponsors ss,
                         Keywords ks,
                         boolean loadAllInfo)
    throws SQLException, ItemException
  {
    loadEvents(getPersonalEvents(user, startDate, endDate,
                                 shouldLoadAuxData(ls, ss)), es, ls, ss);

    loadRecurInstances(getRecurInstances(user, startDate, endDate), es);

//    if (!user.isGuest()) {
    if (es instanceof PersonalEvents) {
      loadEventRefs(getPublicEvents(user, startDate, endDate),
                    (PersonalEvents) es, false);
      loadEventRefs(getPublicRecurInstances(user, startDate, endDate),
                    (PersonalEvents) es, true);
    }

    if (loadAllInfo) {
      loadKeywords(getAllKeywords(user, startDate, endDate), es, ks);
    }
  }

  /**
    Does an eventid correspond to an instance of a recurring event?
    @param eventid id in question
    @return Does an eventid correspond to an instance of a recurring event?
   */
  public static boolean isInstanceId(int eventid)
  {
    return eventid < 0;
  }

  /** load the data corresponding to a single event

    @param eventid id of the event
    @param es holds the Events

    @exception SQLException if it has trouble getting the data
    @exception CaldataException if the data is bad
    @exception NoSuchItemException if the event does not exist
    @exception ItemAccessException if the item can't be added to a cache
  */
  public void loadOneEvent(int eventid, Events es)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    if (isInstanceId(eventid)) {
      loadRecurInstances(getRecurInstance(eventid), es);
      /* No need to load keywords, as loadRecurInstances will
         load the master if needed, which will load the keywords */
    } else {
      loadEvents(getOneEvent(eventid), es, es.getLocations(),
                 es.getSponsors());

      if (es.getEvent(eventid).getRecurrence().isMaster()) {
        loadRecurInstances(getRecurInstances(eventid), es);
      }

      loadKeywords(getAllKeywords(eventid), es, es.getKeywords());
    }
  }

  /** load the data corresponding to a single location

    @param locationid id of the location
    @param ls holds the Locations

    @exception SQLException if it has trouble getting the data
    @exception CaldataException if the data is bad
    @exception ItemAccessException if the location can't be added to the cache
  */
  public void loadOneLocation(int locationid, Locations ls)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    loadLocations(getOneLocation(locationid), ls);
  }

  /** load the data corresponding to a single sponsor

    @param sponsorid id of the sponsor
    @param ss holds the Sponsors

    @exception SQLException if it has trouble getting the data
    @exception CaldataException if the data is bad
    @exception ItemAccessException if the sponsor can't be added to the cache
  */
  public void loadOneSponsor(int sponsorid, Sponsors ss)
      throws SQLException, CaldataException, ItemAccessException
  {
    loadSponsors(getOneSponsor(sponsorid), ss);
  }

  /** load the data corresponding to a single keywords

    @param keywordid id of the keywords
    @param ks holds the Keywords

    @exception SQLException if it has trouble getting the data
    @exception CaldataException if the data is bad
    @exception ItemAccessException if the keyword can't be added to the cache
  */
  public void loadOneKeyword(int keywordid, Keywords ks)
      throws SQLException, CaldataException, ItemAccessException
  {
    loadKeywords(getOneKeyword(keywordid), ks);
  }

  /**
    Does the user have any events at all?
    @param user The user
    @return if the user has events in the database
    @exception SQLException if there's a problem with the database
   */
  public boolean userHasEvents(User user)
    throws SQLException
  {
    ResultSet rs = getAll(Queries.simpleEventsQuery(user));

    try {
      return rs.next();
    } finally {
      this.cconn.close();
    }
  }

  /**
    load all interesting locations for a user (not just those associated
    with events on a particular date)

    @param user the user
    @param ls holds associated Locations

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadLocations(User user, Locations ls)
      throws SQLException, ItemException
  {
    loadLocations(getAllLocations(user), ls);
  }

  /**
    load all interesting sponsors for a user (not just those associated
    with events on a particular date)

    @param user the user
    @param ss holds associated Sponsors

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadSponsors(User user, Sponsors ss)
      throws SQLException, ItemException
  {
    loadSponsors(getAllSponsors(user), ss);
  }

  /**
    load all interesting keywords for a user (not just those associated
    with events on a particular date)

    @param user the user
    @param ks holds associated Keywords
    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadKeywords(User user, Keywords ks)
      throws SQLException, ItemException
  {
    if (!ks.isLoaded()) {
      loadKeywords(getAllKeywords(user), ks);
    }
  }

  /**
    load a user's subscriptions

    @param pe The user's personal events
    @param user The user
    @exception SQLException if it has trouble getting the data
    @exception CaldataException if the data is bad
  */
  public void loadSubscriptions(PersonalEvents pe, User user)
      throws SQLException, CaldataException
  {
    loadSubscriptions(getSubscriptions(user), pe);
  }

  /**
    load all public calendars

    @param pc holds public calendars

    @exception SQLException if it has trouble getting the data
    @exception ItemException if the data is bad
  */
  public void loadCalendars(PublicCalendars pc)
      throws SQLException, ItemException
  {
    loadCalendars(getAllCalendars(), pc);
  }

  private static final int INITIAL_SEQ = 1;

  /**
    Get the current date/time, as a String
    @return the current date/time, as a String
   */
  public static String now()
  {
    return WhereClauses.quote(new Timestamp(new Date().getTime()).toString());
  }

  /**
    Get the current date/time, as a Timestamp
    @return the current date/time, as a Timestamp
   */
  private static Timestamp nowTs()
  {
    return new Timestamp(new Date().getTime());
  }

  /** SQL to add an event */
  private static final String ADD_STMT =
      "insert into " + Tables.EVENTS + " (" +
      Fields.ESEQ.getInsertName() + ", " +
      Fields.ELASTMOD.getInsertName() + ", " +
      Fields.EPUBLIC.getInsertName() + ", " +
      Fields.ECREATED.getInsertName() + ", " +
      Fields.ESTARTDATE.getInsertName() + ", " +
      Fields.ESTARTTIME.getInsertName() + ", " +
      Fields.EENDDATE.getInsertName() + ", " +
      Fields.EENDTIME.getInsertName() + ", " +
      Fields.ESHORTDESC.getInsertName() + ", " +
      Fields.ELONGDESC.getInsertName() + ", " +
      Fields.ELONGDESC1.getInsertName() + ", " +
      Fields.ELINK.getInsertName() + ", " +
      Fields.ESTATUS.getInsertName() + ", " +
      Fields.ELOCATION_ID.getInsertName() + ", " +
      Fields.ESPONSOR_ID.getInsertName() + ", " +
      Fields.ECREATOR.getInsertName() + ", " +
      Fields.ERECURRING_STATUS.getInsertName() + ", " +
      Fields.ECOST.getInsertName() + ") VALUES ("
      + INITIAL_SEQ + ", ?, ?, ?, " +
      "?, ?, ?, ?, ?, ?, ?, ?, \'F\', ?, ?, ?, ?, ?)";

  /** Add an event to the database for a user

      @param e The event to be added
      @param user The user

      @return The database's unique ID of the new event
      @exception SQLException if there's a problem
    */
  public int add(Event e, User user) throws SQLException
  {
    PreparedStatement s = null;
//new CalLogFactory().create().println(ADD_STMT);

    try {
      s = this.cconn.getPreparedStatement(ADD_STMT);
      s.setTimestamp(1, nowTs());
      s.setString(2, e.isPublic() ? TRUE : "F");
      s.setTimestamp(3, nowTs());
      s.setDate(4, e.getStartdate());
      s.setTime(5, e.getStarttime());
      s.setDate(6, e.getEnddate());
      s.setTime(7, e.getEndtime());
      s.setString(8, Fields.ESHORTDESC.sizeToFit(e.getShortdesc()));
      s.setString(9, Fields.ELONGDESC.sizeToFit(Fields.getDBLongdesc(e.getLongdesc())));
      s.setString(10, Fields.ELONGDESC1.sizeToFit(Fields.getDBLongdesc1(e.getLongdesc())));
      s.setString(11, Fields.ELINK.sizeToFit(e.getLink()));
      s.setInt(12, e.getLocationid());
      s.setInt(13, e.getSponsorid());
      s.setString(14, Fields.ECREATOR.sizeToFit(user.getNameDB()));
      s.setString(15, Recurrence.NO_RECUR_VALUE);
      s.setString(16, Fields.ECOST.sizeToFit(e.getCost()));
      s.executeUpdate();
      return getLastEventId(user);
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to delete an event */
  private static final String DELETE_EVENT_STMT =
      "delete from " + Tables.EVENTS + " where " +
      Fields.ECREATOR.getInsertName() + "=? and " +
      Fields.EEVENT_ID.getInsertName() + "=?";

  /** Delete an event from the database for a user

    @param eventid The id of the event to be deleted
    @param user The user

    @exception SQLException if there's a problem
   */
  public void delete(int eventid, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(DELETE_EVENT_STMT);
      s.setString(1, user.getNameDB());
      s.setInt(2, eventid);
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to change an event */
  private static final String REPLACE_STMT =
      "update " + Tables.EVENTS + " set " +
      Fields.ESEQ.getInsertName() + " = " +
      Fields.ESEQ.getInsertName() + " + 1, " +
      Fields.ELASTMOD.getInsertName() + "=?, " +
      Fields.ESTARTDATE.getInsertName() + "=?, " +
      Fields.ESTARTTIME.getInsertName() + "=?, " +
      Fields.EENDDATE.getInsertName() + "=?, " +
      Fields.EENDTIME.getInsertName() + "=?, " +
      Fields.ESHORTDESC.getInsertName() + "=?, " +
      Fields.ELONGDESC.getInsertName() + "=?, " +
      Fields.ELONGDESC1.getInsertName() + "=?, " +
      Fields.ELINK.getInsertName() + "=?, " +
      Fields.ELOCATION_ID.getInsertName() + "=?, " +
      Fields.ESPONSOR_ID.getInsertName() + "=?, " +
      Fields.ECOST.getInsertName() + "=? where " +
      Fields.ECREATOR.getInsertName() + "=? and " +
      Fields.EEVENT_ID.getInsertName() + "=?";

  /** Replace an event in the database for a user

    @param e The event to be replaced
    @param user the user

    @exception SQLException if there's a problem
   */
  public void replace(Event e, User user) throws SQLException
  {
      PreparedStatement s = null;

      try {
          s = this.cconn.getPreparedStatement(REPLACE_STMT);
          s.setTimestamp(1, nowTs());
          s.setDate(2, e.getStartdate());
          s.setTime(3, e.getStarttime());
          s.setDate(4, e.getEnddate());
          s.setTime(5, e.getEndtime());
          s.setString(6, Fields.ESHORTDESC.sizeToFit(e.getShortdesc()));
          s.setString(7, Fields.ELONGDESC.sizeToFit(Fields.getDBLongdesc(e.getLongdesc())));
          s.setString(8, Fields.ELONGDESC1.sizeToFit(Fields.getDBLongdesc1(e.getLongdesc())));
          s.setString(9, Fields.ELINK.sizeToFit(e.getLink()));
          s.setInt(10, e.getLocationid());
          s.setInt(11, e.getSponsorid());
          s.setString(12, Fields.ECOST.sizeToFit(e.getCost()));
          s.setString(13, user.getNameDB());
          s.setInt(14, e.getId());
          s.executeUpdate();
      } finally {
          this.cconn.close();
      }
  }

  /** SQL to delete a location */
  private static final String DELETE_LOCATION_STMT =
      "delete from " + Tables.LOCATIONS + " where " +
      Fields.LCREATOR.getInsertName() + "=? and " +
      Fields.LLOCATION_ID.getInsertName() + "=?";

  /** SQL to change all events with a particular location to the
      deleted location */
  private static final String SET_TO_DELETED_LOCATION =
      "update " + Tables.EVENTS + " set " +
      Fields.ELOCATION_ID.getInsertName() + "=" +
      Location.DELETED_LOCATION_ID + " where " +
      Fields.ELOCATION_ID.getInsertName() + "=?";

  /** Delete a location in the database for a user

    @param locationid The id of the location to be deleted
    @param user the user

    @exception SQLException if there's a problem
   */
  public void deleteLocation(int locationid, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(DELETE_LOCATION_STMT);
      s.setString(1, user.getNameDB());
      s.setInt(2, locationid);
      s.executeUpdate();

      s = this.cconn.getPreparedStatement(SET_TO_DELETED_LOCATION);
      s.setInt(1, locationid);
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  private static final String TRUE = WhereClauses.TRUE;

  /** SQL to change a location */
  private static final String REPLACE_LOCATION_STMT =
      "update " + Tables.LOCATIONS + " set " +
      Fields.LADDRESS.getInsertName() + "=?, " +
      Fields.LSUBADDRESS.getInsertName() + "=?, " +
      Fields.LLINK.getInsertName() + "=?, " +
      Fields.LPUBLIC.getInsertName() + "=? where " +
      Fields.LCREATOR.getInsertName() + "=? and " +
      Fields.LLOCATION_ID.getInsertName() + "=?";

  /** Replace a location in the database for a user

    @param l The location to be replaced
    @param user the user

    @exception SQLException if there's a problem
   */
  public void replaceLocation(Location l, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(REPLACE_LOCATION_STMT);
      s.setString(1, Fields.LADDRESS.sizeToFit(l.getAddress()));
      s.setString(2, Fields.LSUBADDRESS.sizeToFit(l.getSubaddress()));
      s.setString(3, Fields.LLINK.sizeToFit(l.getLink()));
      s.setString(4, l.isPublic() ? TRUE : "F");
      s.setString(5, user.getNameDB());
      s.setInt(6, l.getId());
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }


  /** SQL to add a location */
  private static final String ADD_LOCATION_STMT =
      "insert into " + Tables.LOCATIONS + " (" +
      Fields.LADDRESS.getInsertName() + ", " +
      Fields.LSUBADDRESS.getInsertName() + ", " +
      Fields.LLINK.getInsertName() + ", " +
      Fields.LCREATOR.getInsertName() + ", " +
      Fields.LPUBLIC.getInsertName() + ") VALUES (?, ?, ?, ?, ?)";

    /** Add a location to the database for a user

        @param l The location to be added
        @param user the user

        @return The database's unique ID of the new location
        @exception SQLException if there's a problem
      */
  public int addLocation(Location l, User user) throws SQLException
  {
      PreparedStatement s = null;

      try {
          s = this.cconn.getPreparedStatement(ADD_LOCATION_STMT);
          s.setString(1, Fields.LADDRESS.sizeToFit(l.getAddress()));
          s.setString(2, Fields.LSUBADDRESS.sizeToFit(l.getSubaddress()));
          s.setString(3, Fields.LLINK.sizeToFit(l.getLink()));
          s.setString(4, Fields.LCREATOR.sizeToFit(user.getNameDB()));
          s.setString(5, l.isPublic() ? TRUE : "F");
          s.executeUpdate();
          return getLastLocationId(user);
      } finally {
          this.cconn.close();
      }
  }

  /** SQL to delete a sponsor */
  private static final String DELETE_SPONSOR_STMT =
      "delete from " + Tables.SPONSORS + " where " +
      Fields.SCREATOR.getInsertName() + "=? and " +
      Fields.SSPONSOR_ID.getInsertName() + "=?";

  /** SQL to update all events with a newly deleted sponsor */
  private static final String CHANGE_TO_DELETED_SPONSOR =
      "update " + Tables.EVENTS + " set " +
      Fields.ESPONSOR_ID.getInsertName() + "=" +
      Sponsor.DELETED_SPONSOR_ID + " where " +
      Fields.ESPONSOR_ID.getInsertName() + "=?";

  /** Delete a sponsor in the database for a user

    @param sponsorid The id of the sponsor to be delete
    @param user the user

    @exception SQLException if there's a problem
   */
  public void deleteSponsor(int sponsorid, User user)
    throws SQLException
  {
      PreparedStatement s = null;

      try {
          s = this.cconn.getPreparedStatement(DELETE_SPONSOR_STMT);
          s.setString(1, user.getNameDB());
          s.setInt(2, sponsorid);
          s.executeUpdate();

          s = this.cconn.getPreparedStatement(CHANGE_TO_DELETED_SPONSOR);
          s.setInt(1, sponsorid);
          s.executeUpdate();
      } finally {
          this.cconn.close();
      }
  }

  /** SQL to change a sponsor */
  private static final String REPLACE_SPONSOR_STMT =
      "update " + Tables.SPONSORS + " set " +
      Fields.SNAME.getInsertName() + "=?, " +
      Fields.SPHONE.getInsertName() + "=?, " +
      Fields.SEMAIL.getInsertName() + "=?, " +
      Fields.SLINK.getInsertName() + "=?, " +
      Fields.SPUBLIC.getInsertName() + "=? where " +
      Fields.SCREATOR.getInsertName() + "=? and " +
      Fields.SSPONSOR_ID.getInsertName() + "=?";

  /** Replace a sponsor in the database for a user

    @param sp The sponsor to be replaced
    @param user the user

    @exception SQLException if there's a problem
   */
  public void replaceSponsor(Sponsor sp, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(REPLACE_SPONSOR_STMT);
      s.setString(1, Fields.SNAME.sizeToFit(sp.getName()));
      s.setString(2, Fields.SPHONE.sizeToFit(sp.getPhone()));
      s.setString(3, Fields.SEMAIL.sizeToFit(sp.getEmail()));
      s.setString(4, Fields.SLINK.sizeToFit(sp.getLink()));
      s.setString(5, sp.isPublic() ? TRUE : "F");
      s.setString(6, user.getNameDB());
      s.setInt(7, sp.getId());
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to add a sponsor */
  private static final String ADD_SPONSOR_STMT =
      "insert into " + Tables.SPONSORS + " (" +
      Fields.SNAME.getInsertName() + ", " +
      Fields.SPHONE.getInsertName() + ", " +
      Fields.SEMAIL.getInsertName() + ", " +
      Fields.SLINK.getInsertName() + ", " +
      Fields.SCREATOR.getInsertName() + ", " +
      Fields.SPUBLIC.getInsertName() + ") VALUES (?, ?, ?, ?, ?, ?)";

    /** Add a sponsor to the database for a user

        @param sp The sponsor to be added
        @param user the user

        @return The database's unique ID of the new sponsor
        @exception SQLException if there's a problem
      */
  public int addSponsor(Sponsor sp, User user) throws SQLException
  {
      PreparedStatement s = null;

      try {
          s = this.cconn.getPreparedStatement(ADD_SPONSOR_STMT);
          s.setString(1, Fields.SNAME.sizeToFit(sp.getName()));
          s.setString(2, Fields.SPHONE.sizeToFit(sp.getPhone()));
          s.setString(3, Fields.SEMAIL.sizeToFit(sp.getEmail()));
          s.setString(4, Fields.SLINK.sizeToFit(sp.getLink()));
          s.setString(5, Fields.SCREATOR.sizeToFit(user.getNameDB()));
          s.setString(6, sp.isPublic() ? TRUE : "F");
          s.executeUpdate();
          return getLastSponsorId(user);
      } finally {
          this.cconn.close();
      }
  }

  /** SQL to delete a keyword */
  private static final String DELETE_KEYWORD_STMT =
      "delete from " + Tables.KEYWORDS + " where " +
      Fields.KCREATOR.getInsertName() + "=? and " +
      Fields.KKEYWORD_ID.getInsertName() + "=?";

  /** SQL to change all instances of a deleted keyword to the 'deleted'
      keyword */
  private static final String CHANGE_TO_DELETED_KEYWORD =
      "update " + Tables.EVENTKEYWORDS + " set " +
      Fields.EKKEYWORD_ID.getInsertName() + "=0 where " +
      Fields.EKKEYWORD_ID.getInsertName() + "=?";

  /** Delete a keyword in the database for a user

    @param keywordid The id of the keyword to be delete
    @param user the user

    @exception SQLException if there's a problem
   */
  public void deleteKeyword(int keywordid, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(DELETE_KEYWORD_STMT);
      s.setString(1, user.getNameDB());
      s.setInt(2, keywordid);
      s.executeUpdate();

      s = this.cconn.getPreparedStatement(CHANGE_TO_DELETED_KEYWORD);
      s.setInt(1, keywordid);
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to change a keyword */
  private static final String REPLACE_KEYWORD_STMT =
      "update " + Tables.KEYWORDS + " set " +
      Fields.KWORD.getInsertName() + "=?, " +
      Fields.KLONGDESC.getInsertName() + "=?, " +
      Fields.KPUBLIC.getInsertName() + "=? where " +
      Fields.KCREATOR.getInsertName() + "=? and " +
      Fields.KKEYWORD_ID.getInsertName() + "=?";

  /** Replace a keyword in the database for a user

    @param k The keyword to be replaced
    @param user the user

    @exception SQLException if there's a problem
   */
  public void replaceKeyword(Keyword k, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(REPLACE_KEYWORD_STMT);
      s.setString(1, Fields.KWORD.sizeToFit(k.getWord()));
      s.setString(2, Fields.KLONGDESC.sizeToFit(k.getLongdesc()));
      s.setString(3, k.isPublic() ? TRUE : "F");
      s.setString(4, user.getNameDB());
      s.setInt(5, k.getId());
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to add a keyword */
  private static final String ADD_KEYWORD_STMT =
      "insert into " + Tables.KEYWORDS + " (" +
      Fields.KWORD.getInsertName() + ", " +
      Fields.KLONGDESC.getInsertName() + ", " +
      Fields.KCREATOR.getInsertName() + ", " +
      Fields.KPUBLIC.getInsertName() + ") values (?, ?, ?, ?)";

  /** Add a keyword to the database for a user

    @param k The keyword to be added
    @param user the user

    @return The database's unique ID of the new keyword
    @exception SQLException if there's a problem
   */
  public int addKeyword(Keyword k, User user) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(ADD_KEYWORD_STMT);
      s.setString(1, Fields.KWORD.sizeToFit(k.getWord()));
      s.setString(2, Fields.KLONGDESC.sizeToFit(k.getLongdesc()));
      s.setString(3, Fields.KCREATOR.sizeToFit(user.getNameDB()));
      s.setString(4, k.isPublic() ? TRUE : "F");
      s.executeUpdate();
      return getLastKeywordId(user);
    } finally {
      this.cconn.close();
    }
  }

  /** get SQL to delete a reference to an entity
    @param table Name of the table containing the references
    @param idField Name of the id field for the entity
    @param userField Name of the user field
    @return SQL to delete a reference to an entity
   */
  private static String deleteRefStmt(Table table,
                                      String idField,
                                      String userField)
  {
    return "delete from " + table + " where " + idField + "=? and " +
           userField + "=?";
  }

  /** Delete an reference to an entity in the database for a user

    @param sql SQL deleteRefStmt
    @param er entity to delete
    @exception SQLException if there's a problem
   */
  public void deleteRef(String sql, EntityRef er) throws SQLException
  {
    PreparedStatement s = null;

    try {
      s = this.cconn.getPreparedStatement(sql);
      s.setInt(1, er.getId());
      s.setInt(2, er.getUser().getId());
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to delete an eventref */
  private static final String DELETE_EVENTREF_STMT =
      deleteRefStmt(Tables.EVENTREFS, Fields.EREVENT_ID.getInsertName(),
                    Fields.ERUSER_ID.getInsertName());

  /** Delete an eventref in the database for a user

    @param er The eventref to delete
    @exception SQLException if there's a problem
   */
  public void deleteEventRef(EventRef er) throws SQLException
  {
    deleteRef(DELETE_EVENTREF_STMT, er);
  }

  /** SQL to add an eventref */

  private static final String ADD_EVENTREF_STMT =
      "insert into " + Tables.EVENTREFS + " (" +
      Fields.EREVENT_ID.getInsertName() + ", " +
      Fields.ERUSER_ID.getInsertName() + ", " +
      Fields.ERPURPOSE.getInsertName() + ") values " +
      "(?, ?, ?)";


  /** Add an eventref to the database
      @param er The EventRef to be added
      @return The number of rows changed in the database as a result
        (should be 1)
      @exception SQLException if there's a problem
    */
  public int addEventRef(EventRef er) throws SQLException
  {
    try {
      PreparedStatement s = this.cconn.getPreparedStatement(ADD_EVENTREF_STMT);

      s.setInt(1, er.getId());
      s.setInt(2, er.getUser().getId());
      s.setString(3, er.getPurpose() + "");

      return s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /** SQL to delete a subscription */
  private static final String DELETE_SUBSCRIPTION_STMT =
      deleteRefStmt(Tables.SUBSCRIPTIONS, Fields.SUCALENDAR_ID.getInsertName(),
                    Fields.SUUSER_ID.getInsertName());

  /** Delete a subscription in the database for a user

    @param cr The calendar to delete
    @exception SQLException if there's a problem
   */
  public void deleteSubscription(CalendarRef cr) throws SQLException
  {
    deleteRef(DELETE_SUBSCRIPTION_STMT, cr);
  }

  /** SQL to add a subscription */
  private static final String ADD_SUBSCRIPTION_STMT =
      "insert into " + Tables.SUBSCRIPTIONS + " (" +
      Fields.SUCALENDAR_ID.getInsertName() + ", " +
      Fields.SUUSER_ID.getInsertName() + ") values " +
      "(?, ?)";

  /** Add a subscription to the database
      @param cr The Calendar Ref to be added
      @return The number of rows changed in the database as a result
        (should be 1)
      @exception SQLException if there's a problem
    */
  public int addSubscription(CalendarRef cr) throws SQLException
  {
    try {
      PreparedStatement s =
          this.cconn.getPreparedStatement(ADD_SUBSCRIPTION_STMT);

      s.setInt(1, cr.getId());
      s.setInt(2, cr.getUser().getId());

      return s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /**
    @return The result of executing the query, or null if the query is null.
    @param query The query to execute
    @exception SQLException If there's a problem with the database
   */
  private ResultSet getAll(Query query)
      throws SQLException
  {
    return query == null ? null : getAll(query.toString());
  }

  /**
    @return The result of executing the query, or null if the query is null.
    @param query The query to execute
    @exception SQLException If there's a problem with the database
   */
  private ResultSet getAll(String query)
      throws SQLException
  {
    return query == null ? null : this.cconn.executeQuery(query);
  }

    /* return the results of a query that returns one row with one integer
       result */
  private int getSingleInt(Query query)
    throws SQLException
  {
      ResultSet rs;

      rs = getAll(query);
      rs.next();
      return rs.getInt(1);
  }

  private int getLastEventId(User user) throws SQLException
  {
      return getSingleInt(Queries.lastEventId(user));
  }

  private int getLastLocationId(User user)
    throws SQLException
  {
      return getSingleInt(Queries.lastLocationId(user));
  }

  private int getLastSponsorId(User user)
    throws SQLException
  {
      return getSingleInt(Queries.lastSponsorId(user));
  }

  private int getLastKeywordId(User user)
    throws SQLException
  {
      return getSingleInt(Queries.lastKeywordId(user));
  }

  /** get a user's events for a given series of dates

      @param user the user
      @param startDate first date in the series, in the form
        <code>YYYYMMDD</code>. If null, get all personal events
      @param endDate last date in the series, in the form
        <code>YYYYMMDD</code>
      @param loadAuxData get location and sponsor data as well?

      @exception SQLException if it has trouble getting the data
    */
  private ResultSet getPersonalEvents(User user,
                                     String startDate,
                                     String endDate,
                                     boolean loadAuxData)
      throws SQLException
  {
    return getAll(Queries.eventsQuery(user, startDate, endDate, loadAuxData));
  }

  /** get a user's public events for a given series of dates

      @param user the user
      @param startDate first date in the series, in the form
        <code>YYYYMMDD</code>.  If null, get all public events
      @param endDate last date in the series, in the form
        <code>YYYYMMDD</code>

      @exception SQLException if it has trouble getting the data
      @exception CaldataException if the db contains invalid data
    */
  public ResultSet getPublicEvents(User user,
                                   String startDate,
                                   String endDate)
      throws SQLException, CaldataException
  {
    if (user.isGuest()) {
      throw new IllegalArgumentException("Guest user does not have event refs");
    }

    return getAll(Queries.userPubeventsQuery(user, startDate, endDate));
  }

  /** get a user's recurring public events for a given series of dates

      @param user the user
      @param startDate first date in the series, in the form
        <code>YYYYMMDD</code>.  If null, get all recurring public events
      @param endDate last date in the series, in the form
        <code>YYYYMMDD</code>

      @exception SQLException if it has trouble getting the data
      @exception CaldataException if the db contains invalid data
    */
  public ResultSet getPublicRecurInstances(User user,
                                           String startDate,
                                           String endDate)
      throws SQLException, CaldataException
  {
    if (user.isGuest()) {
      throw new IllegalArgumentException("Guest user does not have event refs");
    }

    return getAll(Queries.userPubRecurInstancesQuery(user, startDate,
                                                        endDate));
  }

  /** get information about recurring events that occur during a
      given series of dates

      @param user the user
      @param startDate first date in the series, in the form
        <code>YYYYMMDD</code>.  If null, get all recurrence info
      @param endDate last date in the series, in the form
        <code>YYYYMMDD</code>

      @exception SQLException if it has trouble getting the data
      @exception CaldataException if the db contains invalid data
    */
  public ResultSet getRecurInstances(User user,
                                     String startDate,
                                     String endDate)
      throws SQLException, CaldataException
  {
    return getAll(Queries.recurInstances(user, startDate, endDate));
  }

  /** get information about instances of a recurring event

      @param eventid id of the master event for the recurrence
      @exception SQLException if it has trouble getting the data
      @exception CaldataException if the db contains invalid data
    */
  public ResultSet getRecurInstances(int eventid)
      throws SQLException, CaldataException
  {
    return getAll(Queries.recurInstances(eventid));
  }

  /** get information about a single instance of a recurring event

      @param eventid id of the instance
      @exception SQLException if it has trouble getting the data
      @exception CaldataException if the db contains invalid data
    */
  public ResultSet getRecurInstance(int eventid)
      throws SQLException, CaldataException
  {
    return getAll(Queries.recurInstance(eventid));
  }

  /**
    Get a single event from the db
    @param eventid ID of the event
    @return Data corresponding to one event
    @exception SQLException If there's a database problem
   */
  private ResultSet getOneEvent(int eventid) throws SQLException
  {
    return getAll(Queries.singleEventQuery(eventid));
  }

  /**
    Get a single location from the db
    @param locationid ID of the location
    @return Data corresponding to one location
    @exception SQLException If there's a database problem
   */
  private ResultSet getOneLocation(int locationid) throws SQLException
  {
    return getAll(Queries.singleLocationQuery(locationid));
  }

  /**
    Get a single sponsor from the db
    @param sponsorid ID of the sponsor
    @return Data corresponding to one sponsor
    @exception SQLException If there's a database problem
   */
  private ResultSet getOneSponsor(int sponsorid) throws SQLException
  {
    return getAll(Queries.singleSponsorQuery(sponsorid));
  }

  /**
    Get a single keyword from the db
    @param keywordid ID of the keyword
    @return Data corresponding to one keyword
    @exception SQLException If there's a database problem
   */
  private ResultSet getOneKeyword(int keywordid) throws SQLException
  {
    return getAll(Queries.singleKeywordQuery(keywordid));
  }

  /** @return The keywords for one event
    @param eventid ID of the event
    @exception SQLException If there's a database problem
   */
  private ResultSet getAllKeywords(int eventid) throws SQLException
  {
    return getAll(Queries.keywordsQuery(eventid));
  }

  /**
    @return The keywords related to the user's events for a series of days
    @param user the user
    @param startDate first date in the series, in the form
      <code>YYYYMMDD</code>.  If null, get keywords for all events
    @param endDate last date in the series, in the form
        <code>YYYYMMDD</code>

    @exception SQLException if it has trouble getting the data
   */
  private ResultSet getAllKeywords(User user,
                                   String startDate,
                                   String endDate)
      throws SQLException
  {
    return getAll(Queries.keywordsQuery(user, startDate, endDate));
  }

  /**
    Get all the keywords associated with all of a user's events
    @param user The user
    @return all the keywords associated with all of a user's events
    @exception SQLException If there's a db problem
   */
  private ResultSet getAllKeywords(User user)
      throws SQLException
  {
    return getAll(Queries.keywordsQuery(user));
  }

  /**
    Get all the sponsors associated with all of a user's events
    @param user The user
    @return all the sponsors associated with all of a user's events
    @exception SQLException If there's a db problem
   */
  private ResultSet getAllSponsors(User user)
      throws SQLException
  {
    return getAll(Queries.sponsorsQuery(user));
  }

  /**
    Get all the locations associated with all of a user's events
    @param user The user
    @return all the locations associated with all of a user's events
    @exception SQLException If there's a db problem
   */
  private ResultSet getAllLocations(User user)
      throws SQLException
  {
    return getAll(Queries.locationsQuery(user));
  }

  /**
    Get a user's subscriptions
    @param user The User
    @return the user's subscriptions
    @exception SQLException If there's a db problem
   */
  private ResultSet getSubscriptions(User user) throws SQLException
  {
    return getAll(Queries.subscriptionsQuery(user));
  }

  /**
    Get all the public calendars
    @return all the public calendars
    @exception SQLException If there's a db problem
   */
  private ResultSet getAllCalendars() throws SQLException
  {
    return getAll(Queries.calendarsQuery());
  }

  /**
    load a single keyword from the next record in a ResultSet
    @param rs ResultSet
    @param ks Cache of keywords to load keyword in
    @return The keyword
    @exception SQLException if there's a problem with the ResultSet
    @exception CaldataException if the results contain invalid keyword data
    @exception ItemAccessException if the keyword can't be added to the cache
   */
  private Keyword loadOneKeyword(ResultSet rs, Keywords ks)
      throws SQLException, CaldataException, ItemAccessException
  {
    Keyword k = new Keyword(rs);
    ks.add(k, false);
    return k;
  }

  /**
    load a single sponsor from the next record in a ResultSet
    @param rs ResultSet
    @param ss Cache of sponsors to load sponsor in
    @return The sponsor
    @exception SQLException if there's a problem with the ResultSet
    @exception CaldataException if the results contain invalid data
    @exception ItemAccessException if the sponsor can't be added to the cache
   */
  private Sponsor loadOneSponsor(ResultSet rs, Sponsors ss)
      throws SQLException, CaldataException, ItemAccessException
  {
    Sponsor s = new Sponsor(rs);
    ss.add(s, false);
    return s;
  }

  /**
    load a single location from the next record in a ResultSet
    @param rs ResultSet
    @param ls Cache of locations to load location in
    @return The location
    @exception SQLException if there's a problem with the ResultSet
    @exception CaldataException if the results contain invalid data
    @exception ItemAccessException if the location can't be added to the cache
   */
  private Location loadOneLocation(ResultSet rs, Locations ls)
      throws SQLException, CaldataException, ItemAccessException
  {
    Location l = new Location(rs);
    ls.add(l, false);
    return l;
  }

  /**
    load keywords from a result set into a cache of keywords and events
    @param rs The ResultSet containing the keywords
    @param es The cache of Events
    @param ks The cache of Keywords
    @exception SQLException if there's a problem with the resultset
    @exception CaldataException if the results contain invalid keyword data
    @exception ItemAccessException if an item can't be added to a cache
   */
  private void loadKeywords(ResultSet rs,
                            Events es,
                            Keywords ks)
    throws SQLException, CaldataException, ItemAccessException
  {
    try {
      if (rs == null) {
        return;
      }

      while (rs.next()) {
        Keyword k = loadOneKeyword(rs, ks);

        // add keyword to list for corresponding event
        try {
          es.getEvent(rs.getInt(Fields.EEVENT_ID.getName())).
              addKeyword(k.getId());
        } catch (NoSuchItemException e) {
          throw new RuntimeException(
              "no event (" + rs.getInt(Fields.EEVENT_ID.getName()) +
              "), but we just loaded it (" + k.getId() + ")");
        }
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of events into our caches

    @param rs The <code>ResultSet</code> to load
    @param es holds the Events
    @param ls holds associated Locations
    @param ss holds associated Sponsors
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid event,
      location, or sponsor data
    @exception ItemAccessException if an item can't be added to a cache
   */
  public void loadEvents(ResultSet rs,
                         Events es,
                         Locations ls,
                         Sponsors ss)
      throws SQLException, CaldataException, ItemAccessException
  {
    try {
      if (rs == null) {
        return;
      }

      while (rs.next()) {
        if (shouldLoadAuxData(ls, ss)) {
          Location l = new Location(rs);
          ls.add(l, false);
          Sponsor s = new Sponsor(rs);
          ss.add(s, false);
        }

        es.add(new Event(rs), false);
      }
    } finally {
      this.cconn.close();
    }
  }

  /**
    load the public events referenced by a <code>ResultSet</code> of
      <code>EventRefs</code>
    If the eventref refers to an unloaded event, the event will be loaded

    @param rs The <code>ResultSet</code> to load
    @param pe holds the Events
    @param recurInstances The events references in the result set are
       recurrence instances

    @exception SQLException if there's a problem with the resultset
    @exception CaldataException if there's a problem with the data when
      loading an unloaded event
    @exception NoSuchItemException if the resultset referenced an
      invalid event, location, sponsor or keyword
    @exception ItemAccessException If an item can't be added to the cache
   */
  public void loadEventRefs(ResultSet rs,
                            PersonalEvents pe,
                            boolean recurInstances)
      throws SQLException, NoSuchItemException, CaldataException,
             ItemAccessException
  {
    try {
      if (rs == null) {
        return;
      }

      while (rs.next()) {
        pe.addPublicEvent(recurInstances ?
                          Fields.getInstanceId(rs) :
                          rs.getInt(Fields.EEVENT_ID.getName()));
      }
    } finally {
      this.cconn.close();
    }
  }

  /**
    load recurring events from a <code>ResultSet</code>
    If the info references an unloaded 'master' event, the event will be
      loaded

    @param rs The <code>ResultSet</code> to load
    @param es holds the Events

    @exception SQLException if there's a problem with the resultset,
      or loading an unloaded event
    @exception CaldataException if there's a problem with the data when
      loading an unloaded event
    @exception NoSuchItemException if the ResultSet references a nonexistant
      event
    @exception ItemAccessException if the ResultSet references an event
      that can't be loaded into <code>Events</code>
   */
  public void loadRecurInstances(ResultSet rs, Events es)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    try {
      if (rs == null) {
        return;
      }

      while (rs.next()) {
        es.addRecurInstance(rs.getInt(Fields.MDMASTER_EVENT_ID.getName()),
                            Fields.getInstanceId(rs), rs);
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of Locations

    @param rs The <code>ResultSet</code> to load
    @param ls holds the Locations
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid location data
    @exception ItemAccessException if the location can't be added to the cache
   */
  public void loadLocations(ResultSet rs,
                            Locations ls)
      throws SQLException, CaldataException, ItemAccessException
  {
    try {
      while (rs.next()) {
        loadOneLocation(rs, ls);
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of Public Calendars

    @param rs The <code>ResultSet</code> to load
    @param pc holds the Public Calendars
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid data
    @exception NoSuchItemException if the resultset contains a reference
       to a non-existent Calendar, Location, Keyword, Sponsor, or User
    @exception ItemAccessException if a calendar can't be added to the
       cache of public calendars
   */
  public void loadCalendars(ResultSet rs, PublicCalendars pc)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    try {
      while (rs.next()) {
        try {
          pc.getCalendar(rs.getInt(Fields.CCALENDAR_ID.getName()));
        } catch (NoSuchItemException e) {
          Calendar c = new Calendar(rs);
          pc.add(c, false);
        }
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of a user's subscriptions

    @param rs The <code>ResultSet</code> to load
    @param pe The user's PersonalEvents object
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid data
   */
  public void loadSubscriptions(ResultSet rs, PersonalEvents pe)
      throws SQLException, CaldataException
  {
    try {
      while (rs.next()) {
        try {
          pe.addSubscription(rs.getInt(Fields.CCALENDAR_ID.getName()));
        } catch (NoSuchItemException e) {
          Logger.getLogger(this.getClass()).error(
                 "missing public calendar", e);
          throw new IllegalStateException("missing public calendar");
        }
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of Sponsors

    @param rs The <code>ResultSet</code> to load
    @param ss holds the Sponsors
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid sponsor data
    @exception ItemAccessException if a sponsor can't be added to the cache
   */
  public void loadSponsors(ResultSet rs,
                           Sponsors ss)
      throws SQLException, CaldataException, ItemAccessException
  {
    try {
      while (rs.next()) {
        loadOneSponsor(rs, ss);
      }
    } finally {
      this.cconn.close();
    }
  }

  /** load a <code>ResultSet</code> of Keywords

    @param rs The <code>ResultSet</code> to load
    @param ks holds the Keywords
    @exception SQLException if there's a problem with the resultset
    @exception CalDataException if the resultset contains invalid location data
    @exception ItemAccessException if a keyword can't be added to the cache
   */
  public void loadKeywords(ResultSet rs,
                           Keywords ks)
      throws SQLException, CaldataException, ItemAccessException
  {
    try {
      while (rs.next()) {
        loadOneKeyword(rs, ks);
      }
    } finally {
      this.cconn.close();
    }
  }

  /** DB statement to add a user */
  private static final String ADD_USER_STMT =
      "insert into " + Tables.USERS + " (" +
      Fields.UUSERNAME.getInsertName() + ", " +
      Fields.UCREATED.getInsertName() + ", " +
      Fields.ULAST_LOGON.getInsertName() + ") VALUES (?, ?, ?)";

  /**
    Add a user to the database
    @param user User to add
    @param loggingIn True if the creation is triggered by the user logging in
    @return The id for the new user
    @exception SQLException if there's a database problem
   */
  public int add(User user, boolean loggingIn) throws SQLException
  {
    try {
      PreparedStatement s = this.cconn.getPreparedStatement(ADD_USER_STMT);
      s.setString(1, user.getNameDB());
      s.setTimestamp(2, nowTs());
      s.setTimestamp(3, loggingIn ? new Timestamp(new Date().getTime()) : null);
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }

    return getUserid(user);
  }

  /** SQL to update a user's last login time */
  private static final String UPDATE_LOGON_STMT =
      "update " + Tables.USERS + " set " +
      Fields.ULAST_LOGON.getInsertName() + "=? where " +
      Fields.UUSER_ID.getStmtName() + "=?";

  /**
    Update the last login time for an existing user
    @param user User to add
    @exception SQLException if there's a database problem
   */
  public void updateLastLogon(User user) throws SQLException
  {
    try {
      PreparedStatement s = this.cconn.getPreparedStatement(UPDATE_LOGON_STMT);
      s.setTimestamp(1, new Timestamp(new Date().getTime()));
      s.setInt(2, user.getId());
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /**
    Add a schedule of public events, indicated by a set of keyword ids,
    to a user's calendar
    @param ids Set of keyword ids
    @param scheduleType Type of the schedule
    @param user User in question
    @param pe user's set of events
    @exception SQLException if there is trouble getting the data
    @exception CaldataException if the data is bad
    @exception NoSuchItemException if the events added reference a non-existant
       keyword, sponsor, or location
    @exception ItemAccessException if an item cannot be added to the cache
   */
  public void addSchedule(Iterator ids,
                          char scheduleType,
                          User user,
                          PersonalEvents pe)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    ResultSet rs = getAll(Queries.scheduleQuery(ids));

    try {
      while (rs.next()) {
        pe.addRef(new EventRef(rs.getInt(Fields.EEVENT_ID.getName()),
                                    user, scheduleType),
                  new MyCalendar(rs.getDate(Fields.ESTARTDATE.getName())));
      }
    } finally {
      this.cconn.close();
    }
  }

  /**
    Remove a schedule of public events, indicated by an integer in all
       of the schedule's keywords, and the schedule's type
       from a user's calendar
    @param scheduleType Type of the schedule
    @param keywordMatch integer to match in the keywords
    @param user User in question
    @param pe user's set of events
    @exception SQLException if there is trouble deleting the schedule
    @exception CaldataException if the data is bad
   */
  public void removeSchedule(char scheduleType,
                             int keywordMatch,
                             User user,
                             PersonalEvents pe)
      throws SQLException, CaldataException
  {
    ResultSet rs = getAll(Queries.scheduleQuery(scheduleType, keywordMatch,
                                                user));

    try {
      while (rs.next()) {
        try {
          // if in cache already, loads it and deletes it from the database
          pe.deleteRef(new EventRef(rs.getInt(Fields.EEVENT_ID.getName()),
                                    user));
        } catch (NoSuchItemException e) {
          // otherwise, just delete it from the database
          new EventRef(rs.getInt(Fields.EEVENT_ID.getName()), user).delete();
        } catch (ItemException e) {
          Logger.getLogger(this.getClass()).error(
                "ItemException", e);
          throw new IllegalStateException(
              "only ItemExceptions here should be NoSuchItemExceptions");
        }
      }
    } finally {
      this.cconn.close();
    }
  }

  /** DB statement to add a calendar */
  private static final String ADD_CALENDAR_STMT =
      "insert into " + Tables.CALENDARS + " (" +
      Fields.CCALENDAR_ID.getInsertName() + ", " +
      Fields.CNAME.getInsertName() + ", " +
      Fields.CTYPE.getInsertName() + ", " +
      Fields.CREF_NUM.getInsertName() + ", " +
      Fields.CPARENT.getInsertName() + ") VALUES ";

  /**
    Add a calendar to the database
    @param values String containing values for the calendar
    @exception SQLException if there's a database problem
   */
  public void addCalendar(String values) throws SQLException
  {
    try {
      PreparedStatement s =
          this.cconn.getPreparedStatement(ADD_CALENDAR_STMT + values);
      s.executeUpdate();
    } finally {
      this.cconn.close();
    }
  }

  /**
    Get the id of a user with a given name
    @param user the user
    @return db id of the user if user exists, User.INVALID_ID if it doesn't.
    @exception SQLException if there's a problem
   */
  public int getUserid(User user) throws SQLException
  {
    try {
      ResultSet rs = getAll(Queries.userQuery(user.getNameDB()));
      return rs.next() ? rs.getInt(Fields.UUSER_ID.getName()) :
                         User.INVALID_ID;
    } finally {
      this.cconn.close();
    }
  }

  /**
    Get the name of the user with a given id #
    @param id Id # of the user
    @return the name of the user with the given id #
    @exception SQLException if there's a problem with the database
    @exception NoSuchItemException if there's no such user
   */
  public String getUsername(int id)
      throws SQLException, NoSuchItemException
  {
    try {
      ResultSet rs = getAll(Queries.userQuery(id));

      if (!rs.next()) {
        throw new NoSuchItemException("No user with id " + id);
      } else {
//System.err.println("here " + id + " " + rs.getString(Fields.UUSERNAME.getName()));
        return rs.getString(Fields.UUSERNAME.getName());
      }
    } finally {
      this.cconn.close();
    }
  }

  /**
    Print the SQL commands used above to the standard output
    @param args ignored
   */
  public static void main(String[] args)
  {
    System.out.println(ADD_STMT);
    System.out.println(DELETE_EVENT_STMT);
    System.out.println(REPLACE_STMT);
    System.out.println(DELETE_LOCATION_STMT);
    System.out.println(SET_TO_DELETED_LOCATION);
    System.out.println(REPLACE_LOCATION_STMT);
    System.out.println(ADD_LOCATION_STMT);
    System.out.println(DELETE_SPONSOR_STMT);
    System.out.println(CHANGE_TO_DELETED_SPONSOR);
    System.out.println(REPLACE_SPONSOR_STMT);
    System.out.println(ADD_SPONSOR_STMT);
    System.out.println(DELETE_KEYWORD_STMT);
    System.out.println(CHANGE_TO_DELETED_KEYWORD);
    System.out.println(REPLACE_KEYWORD_STMT);
    System.out.println(ADD_KEYWORD_STMT);
    System.out.println(DELETE_EVENTREF_STMT);
    System.out.println(ADD_EVENTREF_STMT);
    System.out.println(DELETE_SUBSCRIPTION_STMT);
    System.out.println(ADD_SUBSCRIPTION_STMT);
    System.out.println(ADD_USER_STMT);
    System.out.println(UPDATE_LOGON_STMT);
  }
}
