// java

package edu.washington.cac.calendar.db;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import edu.washington.cac.calenv.CalEnv;

/** A database connection, statements and result sets

    This object is not thread safe; different threads should use different
    copies of this object.

    The close method should not be called on the objects returned by this
    object.  Instead, this object's close method should be called when
    a statement has been completely processed.

    @author Greg Barnes
    @version 2.0
  */
public class CalConnection implements Serializable {
  /** The connection we are currently using */
  private transient Connection conn = null;

  /** The statement we are currently using */
  private transient Statement stmt = null;

  /** The results of the latest query */
  private transient ResultSet rs = null;

  /** A file to write message to */

  /** Create a new connection
   */
  public CalConnection() {
  }

  /** Ensure the connection is open
    @exception SQLException If we have to open a connection, and there's
      a problem
   */
  private void setConnection() throws SQLException {
    if (conn == null) {
      try {
        conn = CalEnv.getEnv().getConnection();
      } catch (Throwable t) {
        t.printStackTrace();
      }
    }
  }

  /** Get a new <code>PreparedStatement</code>
    @param sql SQL for the prepared statemnt
    @exception SQLException if there's a problem
   */
  public PreparedStatement getPreparedStatement(String sql) throws SQLException
  {
    closeStatement();
    setConnection();
    stmt = conn.prepareStatement(sql);
    return (PreparedStatement) stmt;
  }

  /** Create a new statement
    @exception SQLException if there's a problem
   */
  private void setStatement() throws SQLException
  {
    closeStatement();
    setConnection();
    stmt = conn.createStatement();
  }

  /** Execute a query
    @param sql Query to execute
    @return The result set of the query
    @exception SQLException if there's a problem
   */
  public ResultSet executeQuery(String sql) throws SQLException
  {
    closeResultSet();
    setStatement();
//Timer timer = new Timer();
//timer.start();
    rs = stmt.executeQuery(sql);
//timer.printElapsedTime(log, "executeQuery " + sql + ":");

    return rs;
  }

  /** Close the current statement */
  private void closeStatement()
  {
    try {
      if (stmt != null) {
        stmt.close();
      }
    } catch (SQLException e) {	    // don't care
    } finally {
      stmt = null;
    }
  }

  /** Close the current connection */
  private void closeConnection()
  {
    try {
      if (conn != null) {
        conn.close();
      }
    } catch (SQLException e) {	    // don't care
    } finally {
      conn = null;
    }
  }

  /** Close the current result set */
  private void closeResultSet()
  {
    try {
      if (rs != null) {
        rs.close();
      }
    } catch (SQLException e) {	    // don't care
    } finally {
      rs = null;
    }
  }

  /** Close all db-related things that may be open (connections, statements,
     and result sets
   */
  public void close() {
    closeResultSet();
    closeStatement();
    closeConnection();
  }
}
