package  edu.washington.cac.calendar.data;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Timestamp;
import java.sql.SQLException;
import java.util.Date;

import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.db.Fields;

/**
   Represents a user

   @author slh, Greg Barnes
   @version 0.05
 */
public class User extends DBObject
{
  public static final String GUESTID = "xyztseug";

  /** Name as user is known to the database. */
  private String name = null;

  /** Name as user is known to the application. */
  private String nameApp = null;

  /** id # to use for invalid entries */
  public static final int INVALID_ID = -1;

  private int id = INVALID_ID;

  /** unniquely identifies the user session.
      used in URL and Forms. */
  private int sessionIdentifier ;

  /** Value from edu.washington.cac.calendar.data.PublicAccessRights
      Only affects PublicEvents administration
   */
  private int rights;

  /** True if this user is trying to acces public events.
   */
  private boolean publicEvents;

  /**
   * Create a User for the guest/public account/events.
   */
  public User()
  {
//    this(UWSession.GUESTID, UWSession.GUESTID);
    this(GUESTID, GUESTID);
  }

  /**
   * Create a User using supplied name.
   *
   * @param name database and application user name
   */
  public User(String name)
  {
    this(name, name);
  }


  /**
   * Create a User for the guest/public account/events.
   *
   * The isGuest parameter is to distinguish this form from
   * the User(String) form and allow creation with the special guest db name
   * (i.e. there is never a reason to call with isGuest == false;
   *  though it would do the right thing).
   *
   * @param nameApp application user name
   * @param isGuest create as a guest user?
   */
  public User(String nameApp, boolean isGuest)
  {
//    this(isGuest ? UWSession.GUESTID : nameApp, nameApp);
    this(isGuest ? GUESTID : nameApp, nameApp);
  }


  /*Note: this is the base constructor*/
  /**
   * Create a User using supplied names.
   *
   * @param name         database user name
   * @param nameApp      application user name
   */
  public User(String name, String nameApp)
  {
    super();

    if (name == null) {
      throw new NullPointerException(this.getClass().getName() +
                ": name is null");
    }

    if (nameApp == null) {
      throw new NullPointerException(this.getClass().getName() +
                ": nameApp is null");
    }

    this.name = name;
    this.nameApp = nameApp;
  }

  /**
   * @return Is this user the guest user?
   */
  /*
   * A guest is a user with the special internally recognized db user name
   * (e.g. as created with User() or User(String, true)).
   */
  public boolean isGuest()
  {
//    return this.name.equals(UWSession.GUESTID);
    return this.name.equals(GUESTID);
  }

  /** Set (db internal) user id.
   * Valid values are != INVALID_ID.
   * id is valid only after a succesful logon().
   *
   * <p>Note this is only here so that we can copy back values from
   * value objects. The id must have been originally obtained from a
   * succesful logon of this class.
   */
  public void setId(int val) {
    id = val;
  }

  /**
   * Get (db internal) user id.
   *
   * Valid values are != INVALID_ID.
   *
   * id is valid only after a succesful logon().
   */
  public int getId()
  {
    return this.id;
  }


  /**
   * Get (external) database user name.
   */
  public String getNameDB()
  {
    return this.name;
  }

  /**
    @return A string representation of this user
   */
  public String toString()
  {
    return getNameDB();
  }

  /**
    @return Is this the same User?
    @param o Object to compare.  If the <code>o</code> is not a User, return
      false
   */
  public boolean equals(Object o)
  {
    if (o instanceof User) {
      return ((User) o).getName().equals(getName());
    }

    return false;
  }

  /**
   * Get application user name.
   */
  public String getName()
  {
    return this.nameApp;
  }

  /**
   * Set the sessionIdentifier
   */
  public void setIdentifier (int value)
  {
    this.sessionIdentifier = value;
  }

  /**
   * Get the sessionIdentifier
   */
  public int getIdentifier ()
  {
    return this.sessionIdentifier;
  }

  /** Set the rights this user has
   *
   * @param val    int value as defined by
   *               edu.washington.cac.calfacade.shared.UserAuth
   */
  public void setRights(int val) {
    rights = val;
  }

  /** Get the rights this user has
   *
   * @return int   value as defined by
   *               edu.washington.cac.calfacade.shared.UserAuth
   */
  public int getRights() {
    return rights;
  }

  /** Show if this user accessing public events
   */
  public void setPublicEvents(boolean val) {
    publicEvents = val;
  }

  /** Is this user accessing public events
   */
  public boolean getPublicEvents() {
    return publicEvents;
  }

  /**
   * Get application user name.
   */
  public String getNameApp()
  {
    return this.nameApp;
  }

  /** @return Is a user id valid?
    @param id ID to check
   */
  private boolean isValid(int id)
  {
    return this.id > INVALID_ID;
  }

  /**
    Logon user, possibly creating a new record.
    @exception SQLException if there's a database problem
   */
  public void logon() throws SQLException
  {
    this.id = getCaldata().getUserid(this);

    if (!isValid(this.id)) {
      this.id = getCaldata().add(this, true);
    } else {
      getCaldata().updateLastLogon(this);
    }
  }
}
