// java

package edu.washington.cac.calendar.data;

import java.sql.ResultSet;
import java.sql.SQLException;

import edu.washington.cac.calendar.db.DBResources;
import edu.washington.cac.calendar.db.Fields;
import edu.washington.cac.calendar.db.WhereClauses;

/**
  The sponsor of an event
  @author Greg Barnes
  @version 2.2
 */
public class Sponsor extends CalendarObject implements LinkEntity
{
  private int sponsorid;
  private String name, phone, email, link;

  public static final int DELETED_SPONSOR_ID = 3;
  public static final int NO_SPONSOR_ID = 1;
  public static final int UNKNOWN_SPONSOR_ID = 2;
  public static final int DEFAULT_SPONSOR_ID = NO_SPONSOR_ID;

  // /** A dummy sponsor */
  // public static Sponsor DUMMY_SPONSOR;        // final, really

  /** A deleted sponsor */
  public static Sponsor DELETED_SPONSOR;        // final, really

  /** Setup special static sponsor variables */
  static {
    /* try {
      DUMMY_SPONSOR = new Sponsor(DEFAULT_SPONSOR_ID, "no sponsor",
                                  null, null, null,
                                  DBResources.specialCreator(), true);
    } catch (CaldataException e) {  // shouldn't happen
      DUMMY_SPONSOR = null;
    } */

    try {
      DELETED_SPONSOR = new Sponsor(DELETED_SPONSOR_ID, "deleted",
                                    null, null, null,
                                    DBResources.specialCreator(), true);
    } catch (CaldataException e) {  // shouldn't happen
      DELETED_SPONSOR = null;
    }
  }

  /**
    Create a sponsor specifying all fields

    @param sponsorid Unique ID for the sponsor
    @param name Name of the sponsor
    @param phone Phone number
    @param email E-mail
    @param link URL with more info
    @param creator The person who created the event
    @param isPublic Is the event viewable by everyone?
    @exception CaldataException if the fields are not valid in some way
   */
  public Sponsor(int sponsorid,
                 String name,
                 String phone,
                 String email,
                 String link,
                 User creator,
                 boolean isPublic)
      throws CaldataException
  {
    super(creator, isPublic);
    DataChecker.checkShortdesc(name);

    this.sponsorid = sponsorid;
    this.name = name;
    this.phone = phone;
    this.email = email;
    this.link = link;
  }

  /**
    Create a sponsor from a JDBC result set
      @param rs The result set with the sponsor's fields
      @exception SQLException if there's a problem getting the data
      @exception CaldataException if the fields are not valid in some way
    */
  public Sponsor(ResultSet rs)
      throws SQLException, CaldataException
  {
    this(rs.getInt(Fields.SSPONSOR_ID.getName()),
         rs.getString(Fields.SNAME.getName()),
         rs.getString(Fields.SPHONE.getName()),
         rs.getString(Fields.SEMAIL.getName()),
         rs.getString(Fields.SLINK.getName()),
         new User(rs.getString(Fields.SCREATOR.getName())),
         rs.getString(Fields.SPUBLIC.getName()).equals(WhereClauses.TRUE));
  }

  public void setId(int sponsorid)
  {
    this.sponsorid = sponsorid;
  }

  public int getId()
  {
    return sponsorid;
  }

  public String getName()
  {
    return name;
  }

  public String getPhone()
  {
    return phone;
  }

  public String getEmail()
  {
    return email;
  }

  public String getLink()
  {
    return link;
  }

  /**
    Set the sponsor's URL
    @param link The new URL
   */
  public void setLink(String link)
  {
    this.link = link;
  }

  public void delete(User user) throws SQLException
  {
    getCaldata().deleteSponsor(getId(), user);
  }

  /**
    Replace the old version of this sponsor in the db with this version
    @param user User requesting the change
    @exception SQLException if there's a db error
   */
  public void replace(User user) throws SQLException
  {
    getCaldata().replaceSponsor(this, user);
  }

  /**
    Add this sponsor to the db
    @param user User requesting the add
    @exception SQLException if there's a db error
   */
  public void add(User user) throws SQLException
  {
    sponsorid = getCaldata().addSponsor(this, user);
  }

  /**
    @return whether this sponsor is 'less than' another
    @param c Sponsor to compare to.  If <code>c</code> is not a sponsor,
      return false
   */
  public boolean lessThan(CalendarObject c)
  {
    if (!(c instanceof Sponsor)) {
      return false;
    }

    Sponsor s = (Sponsor) c;

    if (getName().compareTo(s.getName()) != 0) {
      return (getName().compareTo(s.getName()) < 0);
    } else {
      return getCreator().getNameDB().compareTo(s.getCreator().getNameDB()) < 0;
    }
  }
}
