// java

package edu.washington.cac.calendar.data;

/**
  Recurrence information for an event.

  Currently, all recurrences are expanded in the Events object.
  If an event is recurring, it is either the 'master' instance of the 
  recurring event, or not.

  The id of the master instance is used when adding public eventrefs 
  to a user's personal calendar, and when removing a recurring event.

  @author Greg Barnes
  @version 1.0
 */
public abstract class Recurrence 
{
  /** masterId when the master id is not know or doesn't exist */
  public static final int NO_MASTER = -1; 

  /** id of the master event of the recurrence */
  private int masterId;

  /**
    Create a new instance
    @param masterId Id of the master event of the recurrence
   */
  public Recurrence(int masterId)
  {
    this.masterId = masterId;
  }

  /** The value of the recurring flag field in the database 
     for a non-recurring event */
  public static final String NO_RECUR_VALUE = "N";

  /** The value of the recurring flag field in the database 
     for a master event */
  public static final String MASTER_VALUE = "M";

  /** Factory method to get a recurrence corresponding to the RECURRING_FLAG
    in the database
    @param flag Value of the RECURRING_FLAG field in the database.  
    @param id Id of the event whose recurrence this is
   */
  public static Recurrence create(String flag, int id)
  {
    if (flag.equals(NO_RECUR_VALUE)) {
      return NoRecurrence.NO_RECUR;
    } else if (flag.equals(MASTER_VALUE)) {
      return new MasterRecurrence(id);
    } else {
      return new InstanceRecurrence(InstanceRecurrence.NO_MASTER);
    }
  }

  /** 
    Is this a master recurrence?
    @return Is this a master recurrence?
   */
  public abstract boolean isMaster();

  /** 
    Is this a recurring recurrence?
    @return Is this a recurring recurrence?
   */
  public abstract boolean recurs();


  /**
    Set the id of the master event
    @param masterId the id of the master event
   */
  public void setMasterId(int masterId)
  {
    this.masterId = masterId;
  }

  /** 
    Get the id of the master id of this recurrence (assuming it exists)
    @return the id of the master id of this recurrence (assuming it exists)
   */
  public int getMasterId() 
  {
    return this.masterId;
  }
}
