// java

package edu.washington.cac.calendar.data;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.db.CalConnection;  // lastmod
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.filter.PublicCalendars;
import edu.washington.cac.calenv.CalEnv;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Iterator;
import java.util.Vector;

import org.apache.log4j.Logger;

/**
  A global singleton set of all public events.  This is a thin wrapper
  around an Events object with the appropriate methods synchronized,
  and other methods disallowed.

  @author Greg Barnes
  @version 1.0
 */
public class PublicEvents extends Events
{
  /** The set of public calendars */
  private static PublicCalendars pc = null;

  /** The single instance of this class */
  private static PublicEvents pe = null;

  /* ------------------- lastmod stuff ----------------------------- */
  /** Last time PublicEvents changed */
  private static Timestamp lastmod;

  /** Last time we checked for PublicEvents changing */
  private static long lastUpdateCheckTime;

  /** How often we check for PublicEvents changing. Less than 0 means never
   */
  private static long updateCheckInterval;

  private static boolean updateCheckInitialised;

  /** This all ought to be elsewhere so it can be customized.
      I want to change as few source files as possible at the moment.

      Mike Douglass.
   */
  private final static String lastmodQuery =
      "select lastmod from lastmods where name='pubevents'";

  private static CalConnection calConn =
       new CalConnection();

  private static boolean loadingCalendars = false;

  /** Initialize the singleton
  static {
    initializePublicEvents();
  } */

  /** Lastmod related routine - flushes the data and gets a new set if the
   *  lastmod has changed.
   *
   * @param user    User object - presumably for admin
   */
  private static synchronized PublicEvents getPe(User usr) {
    try {
      if (!updateCheckInitialised) {
        updateCheckInterval = CalEnv.getEnv().getUpdateCheckInterval();
        updateCheckInitialised = true;
      }

      if (usr == null) {
        usr = new User();
      }

      boolean check = false;
      long now = System.currentTimeMillis();

      if (updateCheckInterval >= 0) {
        check = (now - lastUpdateCheckTime) > updateCheckInterval;
      }

      if (check) {
        lastUpdateCheckTime = now;

        /* Get the timestamp. */
        Timestamp dbLastmod = null;

        ResultSet rs = calConn.executeQuery(lastmodQuery);

        if (rs == null) {
          /** Assume no change */
        } else if (rs.next()) {
          dbLastmod = rs.getTimestamp("lastmod");
        }

        if (pe == null) {
          if (dbLastmod != null) {
            lastmod = dbLastmod;
          }
        } else if (!dbLastmod.equals(lastmod)) {
          pe = null;
          lastmod = dbLastmod;
        }
      }

      initializePublicEvents(usr);
    } catch (Throwable t) {
      Logger.getLogger("edu.washington.cac.calendar.data.PublicEvents").error(
            "Could not get public events lastmod", t);
      pe = null;
    } finally {
      calConn.close();
    }

    return pe;
  }

  /** Force a flush of the data.
   */
  public static synchronized void forceFlush() {
    pe = null;
  }

  /**
   * Initialize the singleton set of public events
   *
   * @param user    User object - presumably for admin
   */
  private static synchronized void initializePublicEvents(User usr)
  {
    if (pe == null || pc == null) {
      try {
        pe = new PublicEvents(usr);
        pc = new PublicCalendars();
        /* Note: liable to get infinite loop if you load calendars in
           public calendar constructor */
        loadingCalendars = true;
        try {
          pc.loadAll();
        } catch (Exception e) {
          e.printStackTrace();
        }
        loadingCalendars = false;
      } catch (Exception e) {
      Logger.getLogger("edu.washington.cac.calendar.data.PublicEvents").error(
              "Could not create public events", e);
        pe = null;
        pc = null;
      }
    }
  }

  /**
   * Get the set of public events
   *
   * @return the set of public events
    */
  public static PublicEvents getPublicEvents() {
    return getPublicEvents(null);
  }

  /**
   * Get the set of public events
   *
   * @param usr             User object - presumably for admin
   * @return PublicEvents   the set of public events
    */
  public static PublicEvents getPublicEvents(User usr) {
    /** See if we need a flush - will init if required */
    getPe(usr);

    if (pe == null) {
      throw new RuntimeException("Could not get public events");
    }

    return pe;
  }

  /** Get the public calendars
    @return The public calendars
   */
  public static PublicCalendars getPublicCalendars() {
    return getPublicCalendars(null);
  }

  /** Get the public calendars
   *
   * @param usr               User object - presumably for admin
   * @return PublicCalendars  public calendars object
   */
  public static PublicCalendars getPublicCalendars(User usr) {
    /** See if we need a flush - will init if required */
    if (loadingCalendars && pc != null) {
      return pc;
    }

    getPe(usr);

    if (pc == null) {
      throw new RuntimeException("Could not get public calendars");
    }

    return pc;
  }

  /** Get the public keywords
    @return The public keywords
   */
  public static Keywords getPublicKeywords()
  {
    return getPe(null).getKeywords();
  }

  /** Get the public keywords
   *
   * @param usr            User object - presumably for admin
   * @return Keywords      public keywords
   */
  public static Keywords getPublicKeywords(User usr)
  {
    return getPe(usr).getKeywords();
  }

  /** Get the public sponsors
    @return The public sponsors
   */
  public static Sponsors getPublicSponsors()
  {
    return getPe(null).getSponsors();
  }

  /** Get the public sponsors
   *
   * @param usr            User object - presumably for admin
   * @return Sponsors      public sponsors
   */
  public static Sponsors getPublicSponsors(User usr)
  {
    return getPe(usr).getSponsors();
  }

  /** Get the public locations
    @return The public locations
   */
  public static Locations getPublicLocations()
  {
    return getPe(null).getLocations();
  }

  /** Get the public locations
   *
   * @param usr            User object - presumably for admin
   * @return Locations     public locations
   */
  public static Locations getPublicLocations(User usr)
  {
    return getPe(usr).getLocations();
  }

  public long getLastmod() {
    getPe(null);

    return lastmod.getTime();
  }

  /** Initialize the set, including loading locations, sponsors and keywords
    @exception SQLException if the auxiliary items cannot be loaded
    @exception ItemException if the auxiliary items contain bad data
   */
  private PublicEvents() throws SQLException, ItemException
  {
    // loads locations, sponsors, keywords
    super(new User(), true);
    this.loadAll();
  }

  /** Initialize the set, including loading locations, sponsors and keywords
   *
   * @param usr    User object - presumably for admin
   * @exception SQLException if the auxiliary items cannot be loaded
   * @exception ItemException if the auxiliary items contain bad data
   */
  private PublicEvents(User usr) throws SQLException, ItemException {
    // loads locations, sponsors, keywords

    super(usr, true);
    this.loadAll();
  }

  /** not allowed
    @exception IllegalStateException Always thrown
   */
  public void addEventRef(EventRef er, MyCalendar mc)
     throws IllegalStateException
  {
    throw new IllegalStateException("cannot add event refs");
  }

  /* All other public methods are synchronized to avoid inconsistent views
     of the data (note:  public methods that are simply a call to
     another public method with different parameters are left unsynchronized
   */
  /**
    @return The number of items in the Cache
   */
  public synchronized int size()
  {
    return super.size();
  }

  /**
    @return All the items in the cache
   */
  public synchronized Iterator elements()
  {
    return super.elements();
  }

  public synchronized void add(CalendarObject c, boolean saveToDB)
      throws SQLException, ItemAccessException
  {
    super.add(c, saveToDB);
  }

  public synchronized void delete(CalendarObject c)
      throws SQLException, ItemException {
    super.delete(c);
  }

  public synchronized void replace(CalendarObject c)
      throws SQLException, ItemException {
    super.replace(c);
  }

  /**
    Load a single item into the cache
    @param id Id of the item to load
    @exception SQLException If there's a problem accessing the database
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If the item does not exist
    @exception ItemAccessException If the item can't be added to the cache
   */
  protected synchronized void loadOne(int id)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    super.loadOne(id);
  }

  /**
    get an item from the Cache

    @param id ID of the item
    @return The CalendarObject with the same id
    @exception NoSuchItemException if no item exists with that id
   */
  public synchronized CalendarObject get(int id) throws NoSuchItemException
  {
    return super.get(id);
  }

  /**
    @return If the key refers to an object in the cache
    @param o key to check
   */
  public synchronized boolean containsKey(Object o)
  {
    return super.containsKey(o);
  }

  /**
    @return all the elements in the cache, in sorted order
   */
  public synchronized Iterator sortedElements()
  {
    return super.sortedElements();
  }

  /** Set the keywords cache
    @param ks new Keywords cache
   */
  public synchronized void setKeywords(Keywords ks)
  {
    super.setKeywords(ks);
  }

  /** Set the Locations cache
    @param ls new Locations cache
   */
  public synchronized void setLocations(Locations ls)
  {
    super.setLocations(ls);
  }

  /** Set the sponsors cache
    @param ss new Sponsors cache
   */
  public synchronized void setSponsors(Sponsors ss)
  {
    super.setSponsors(ss);
  }

  /**
      Get the set of associated locations for a user
      @return the set of associated locations for a user
    */
  public synchronized Locations getLocations()
  {
    return super.getLocations();
  }

  /**
    Get the set of associated sponsors for a user
    @return the set of associated sponsors for a user
    */
  public synchronized Sponsors getSponsors()
  {
    return super.getSponsors();
  }

  /**
    Get the set of associated keywords for a user
    @return the set of associated keywords for a user
    */
  public synchronized Keywords getKeywords()
  {
    return super.getKeywords();
  }

  /**
    Load all public events
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
    */
  public synchronized void doLoadAll() throws SQLException, ItemException
  {
    super.doLoadAll();
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadKeywords should the keywords be loaded (if the events are?)

    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public synchronized Event[] oneDaysEvents(MyCalendar mc, boolean loadKeywords)
      throws SQLException, ItemException
  {
    return super.oneDaysEvents(mc, loadKeywords);
  }

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  public synchronized void prepareManyDays(Calendar start,
                                           Calendar end)
      throws SQLException, ItemException
  {
    super.prepareManyDays(start, end);
//new edu.washington.cac.myuw.servlet.Memory().printStatus(new CalLogFactory().create());
//new CalLogFactory().create().println(size());
  }

  /**
    Was this event loaded by this cache?
    @param e Event
    @return Was this event loaded by this cache?
   */
  protected boolean loadedByThisCache(Event e)
  {
    return e.isPublic();
  }

  /**
    Get all events loaded by this cache
    @return all events loaded by this cache
   */
  protected Vector eventsLoadedByThisCache()
  {
    Vector v = new Vector();
    Iterator i = elements();

    while (i.hasNext()) {
      Event e = (Event) i.next();

      if (e.wasLoadedByDefault()) {
        v.addElement(e);
      }
    }

    return v;
  }

  /**
    Invalidate the cache.  Events not loaded by default are not invalidated.
    All other events are deleted and all day's caches are invalidated.
   */
  synchronized void invalidate()
  {
    /* Note that we cannot iterate through this.elements() and delete things
       in one loop, as this causes a ConcurrentModificationException.  So
       first we get the set of events to delete, then we delete the
       members of that set
     */
    Iterator i = eventsLoadedByThisCache().iterator();

    while (i.hasNext()) {
      try {
        localDelete((Event) i.next());
      } catch (ItemException ex) {// shouldn't happen
        Logger.getLogger(this.getClass()).error(
              "error invalidating public events", ex);
        throw new RuntimeException("error invalidating public events");
      }
    }

    invalidateAllDays();
  }

  /**
    Do an initial public events load
    @param args Ignored
   */
  public static void main(String[] args)
  {
    getPublicEvents();
  }
}
