// java

package edu.washington.cac.calendar.data;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.log4j.Logger;

import edu.washington.cac.calendar.MyCalendar;
import edu.washington.cac.calendar.db.Caldata;
import edu.washington.cac.calendar.filter.CalendarRef;
import edu.washington.cac.calendar.filter.Filter;
import edu.washington.cac.calendar.filter.FilteredEvents;
import edu.washington.cac.calendar.filter.NoFilter;
import edu.washington.cac.calendar.filter.OrFilter;
import edu.washington.cac.calendar.filter.PublicCalendars;

  // note potential name class with java.util.Calendar
import edu.washington.cac.calendar.filter.Calendar;

/**
  A set of Events for a single, non-guest user.

  @author Greg Barnes
  @version 1.0
 */
public class PersonalEvents extends Events
{
  /** The subscriptions for a user, if any */
  private Set subscriptions = new TreeSet();

  /** The events corresponding to those subscriptions */
  private FilteredEvents subscribedEvents = new FilteredEvents(new NoFilter());

  /** Initialize the set
    @param user the user
    @param load Whether to load auxiliary items on creation (locations,
       sponsors, and keywords)
    @exception SQLException if the auxiliary items can't be loaded
    @exception ItemException if the auxiliary item data is bad
    */
  public PersonalEvents(User user, boolean load)
      throws SQLException, ItemException
  {
    super(user, load);
    initializeSubscriptions();
  }

  /** Initialize the set
    @param user the user
    @exception SQLException if the auxiliary items can't be loaded
    @exception ItemException if auxiliary item data is bad
    */
  public PersonalEvents(User user)
      throws SQLException, ItemException
  {
    this(user, true);  //loads Locations, sponsors, keywords
  }

  /**
    Add information about a public event to this cache.  If the public
    event is not initially in the public events cache, it will be loaded
    from the database

    @param eventid Id of the public event to load
    @exception SQLException If there's a problem loading the event
    @exception CaldataException If there's a problem with the data in the db
    @exception NoSuchItemException If an event with the referenced id does not
       exist
    @exception ItemAccessException If an item can't be added to the cache
   */
  public void addPublicEvent(int eventid)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
//new CalLogFactory().create().println("addPublicEvent: " + eventid);
    ensureLoaded(eventid, true);

    Event e = PublicEvents.getPublicEvents().getEvent(eventid);
    getLocations().localAdd(PublicEvents.getPublicLocations().
        getLocation(e.getLocationid()));
    getSponsors().localAdd(PublicEvents.getPublicSponsors().
        getSponsor(e.getSponsorid()));
    localAdd(e);

    Iterator keywords = e.getKeywords();

    while (keywords.hasNext()) {
      getKeywords().localAdd(PublicEvents.getPublicKeywords().
          getKeyword(((Integer) keywords.next()).intValue()));
    }

    if (e.getRecurrence().isMaster()) {
      Iterator instances =
          ((MasterRecurrence) e.getRecurrence()).getInstances();

      while (instances.hasNext()) {
        localAdd((Event) instances.next());
        // location, sponsor, keywords already added as part of master
      }
    } else if (e.getRecurrence().recurs()) {
      // make sure the master is added, too
      addPublicEvent(e.getRecurrence().getMasterId());
    }
  }

  /**
    Generate the subscribedEvents object from the set of subscriptions
   */
  private void generateSubscribedEvents()
  {
    if (subscriptions.size() == 0) {
      this.subscribedEvents = new FilteredEvents(new NoFilter());
    } else {
      Iterator i = this.subscriptions.iterator();
      OrFilter allSubscriptions = new OrFilter(null);
      Calendar c;

      for (; i.hasNext(); ) {
        allSubscriptions.addChild((Calendar) i.next());
      }

      this.subscribedEvents = new FilteredEvents(allSubscriptions);
    }
  }

  /**
    Add a calendar subscription
    @param calendarId Id of the calendar
    @exception NoSuchItemException If the calendar does not exist
   */
  public void addSubscription(int calendarId) throws NoSuchItemException
  {
    this.subscriptions.add(
        PublicCalendars.getPublicCalendars().getCalendar(calendarId));
    generateSubscribedEvents();
  }

  /**
    Delete a calendar subscription
    @param calendarId Id of the calendar
    @exception NoSuchItemException If the calendar is not currently subscribed
      to
   */
  public void deleteSubscription(int calendarId) throws NoSuchItemException
  {
    this.subscriptions.remove(
        PublicCalendars.getPublicCalendars().getCalendar(calendarId));
    generateSubscribedEvents();
  }

  /**
    initialize the user's set of subscriptions
    @exception SQLException if subscriptions can't be loaded
    @exception CaldataException if subscription data is bad
   */
  /* This could call addSubscription many times, which leads to
     generateSubscribedEvents being called needlessly, but I'm guessing
     it won't matter
   */
  private void initializeSubscriptions()
      throws SQLException, CaldataException
  {
    getCaldata().loadSubscriptions(this, this.user);
  }

  /**
    Add a reference to some other entity to the cache

    @param er The reference.  Currently, either to a single event or
              multiple events (i.e., a calendar)
    @param mc The start day of the entity
    @exception SQLException if there is database trouble
    @exception CaldataException if the data we get back is bad
    @exception NoSuchItemException if the entity doesn't exist, or
      references a non-existent location, sponsor or keyword
    @exception ItemAccessException If an item can't be added to the cache
   */
  public void addRef(EntityRef er, MyCalendar mc)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    // Before we add a ref to an event, make sure the event exists
    if (er.isSingleEntity()) {
      PublicEvents.getPublicEvents().ensureLoaded(er.getId());
    }

    if (!er.add()) {
      throw new SQLException("addRef failed for entity: "
                             + er.getId() + " " + er);
    }

    if (er.isSingleEntity()) {
      addPublicEvent(er.getId());
    } else {  // multiple entities
      addSubscription(er.getId());
    }
  }

  /**
    Ensure that the users subscriptions match a set of calendars
    @param crs set of calendars
    @return The number of calendars subscribed to, and the number of
      calendars unsubscribed to
    @exception SQLException if there is database trouble
    @exception ItemAccessException If an item can't be added to the cache
   */
  public int[] fixSubscriptions(TreeSet crs)
      throws SQLException, ItemAccessException
  {
    int subs = 0;
    int unsubs = 0;
    int unknown = 0;

    /*
      First, delete all subscriptions that aren't in crs.
      Straightforward idea is to iterator over subscriptions, but
      iterators don't like it when you delete something that's
      in the underlying set.  So clone the set first.
     */
    Iterator i = new TreeSet(subscriptions).iterator();

    while (i.hasNext()) {
      Calendar c = (Calendar) i.next();

      if (!crs.contains(new Integer(c.getId()))) {
        try {
          deleteRef(new CalendarRef(c.getId(), this.user));
          unsubs++;
        } catch (ItemException e) { // shouldn't happen
          throwLog(e, new RuntimeException(
                      "Got ItemException while unsubscribing"));
        }
      }
    }

    i = crs.iterator();

    while (i.hasNext()) {
      int id = ((Integer) i.next()).intValue();

      try {
        if (!containsSubscription(id)) {
          addRef(new CalendarRef(id, this.user), null);
          subs++;
        }
      } catch (NoSuchItemException e) {
        unknown++;
      } catch (CaldataException e) { // shouldn't happen
        throwLog(e, new RuntimeException(
                      "Got CaldataException while subscribing"));
      }
    }

    int[] returnValue = {subs, unsubs, unknown};
    return returnValue;
  }

  /**
    Delete a reference to an entity from the cache
    @param er The reference.  Currently either a single event or multiple
              events (i.e., a calendar)
    @exception SQLException if there is database trouble
    @exception ItemException if there's a problem with the entity
   */
  public void deleteRef(EntityRef er) throws SQLException, ItemException
  {
    if (er.isSingleEntity()) {
      CalendarObject c = get(er.getId());

      if (c.canBeDeletedBy(er.getUser())) {
        throw new ItemAccessException("Attempt to delete object " + c + " as if it were an eventref!");
      }

      localDelete(c);  // deletes from this cache
    } else {
      deleteSubscription(er.getId());   // delete from this cache
    }

    er.delete();        // deletes from db
  }

  /**
    Add a schedule to a user's calendar
    @param ids Collection of keyword ids for the schedule's events
    @param scheduleType type of the schedule
    @exception SQLException if there is database trouble
    @exception CaldataException if the data we get back is bad
    @exception NoSuchItemException if the event references a non-existent
      location, sponsor or keyword
    @exception ItemAccessException If an item can't be added to the cache
   */
  public void addSchedule(Iterator ids, char scheduleType)
      throws SQLException, CaldataException, NoSuchItemException,
             ItemAccessException
  {
    getCaldata().addSchedule(ids, scheduleType, this.user, this);
  }

  /**
    Remove a schedule from a user's calendar
    @param scheduleType type of the schedule
    @param keywordMatch int to match on keywords for events to delete
    @exception SQLException if there is database trouble
    @exception CaldataException if the data we get back is bad
   */
  public void removeSchedule(char scheduleType, int keywordMatch)
      throws SQLException, CaldataException
  {
    getCaldata().removeSchedule(scheduleType, keywordMatch, this.user, this);
  }

  /**
    delete an event from the Cache

    @param c the item to delete
    @exception ItemException if there's a problem with the item
    @exception SQLException if there is a problem with the database
   */
  public void delete(CalendarObject c) throws SQLException, ItemException
  {
    try {
      super.delete(c);
    } catch (NoSuchItemException e) {
      if (subscribedEvents.containsKey(key(c))) {
        throw new SubscriptionItemException(
            "Cannot delete a single subscribed event");
      } else {
        throw e;
      }
    }
  }

  /**
    replace an item in the Cache

    @param c the item to replace (the current item with the same id
      as c will be replaced with c)
    @exception SQLException if there is a problem with the database
    @exception ItemException if there's a problem with the item specified
   */
  public void replace(CalendarObject c)
      throws SQLException, ItemException
  {
    try {
      super.replace(c);
    } catch (NoSuchItemException e) {
      /* Given the canBeEditedBy test in super.replace(), this code may
         never be executed */
      if (subscribedEvents.containsKey(key(c))) {
        throw new SubscriptionItemException(
            "Cannot replace a subscribed event");
      } else {
        throw e;
      }
    }
  }

  /**
    get an item from the Cache

    @param id ID of the item
    @return The CalendarObject with the same id
    @exception NoSuchItemException if no item exists with that id
   */
  public CalendarObject get(int id) throws NoSuchItemException
  {
    try {
      return super.get(id);
    } catch (NoSuchItemException e) {
      return subscribedEvents.get(id);
    }
  }

  /**
    Remove duplicate items from a Vector whose items are sorted
    @param v The vector
   */
  private static void removeDuplicates(Vector v)
  {
    if (v.size() < 2) {
      return;
    }

    for (int i = 1; i < v.size(); i++) {
      while (i < v.size() && v.elementAt(i).equals(v.elementAt(i-1))) {
        v.remove(i);
      }
    }
  }

  /**
    Get the events for a user on a given day, sorted in an appropriate
    way, as an array

    @param mc Date, represented as a <code>MyCalendar</code>
    @param loadAllInfo should all associated info be loaded (if the events are?)

    @return the events for a user on a given day, sorted
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public Event[] oneDaysEvents(MyCalendar mc, boolean loadAllInfo)
      throws SQLException, ItemException
  {
    /* We must copy the vector returned by super.oneDaysVEvents, as it
       is also stored in the Events object, and we may modify it below */
    Vector v = new Vector();
    v.addAll(super.oneDaysVEvents(mc, loadAllInfo));

    /* FilteredEvents can only give events as array, so convert to a list
       to add them to the Vector */
    v.addAll(Arrays.asList(subscribedEvents.oneDaysEvents(mc, loadAllInfo)));

    Collections.sort(v);
    removeDuplicates(v);
    return eventArray(v);
  }

  /**
    Get the events for a user over a series of days

    @param startDate first day in the series, represented as a
        <code>MyCalendar</code>
    @param endDate last day in the series, represented as a
        <code>MyCalendar</code>

    @return the events for the user over the series of days
    @exception SQLException if there is a problem loading from the database
    @exception ItemException if the database contains bad data
  */
  public List manyDaysEvents(MyCalendar startDate, MyCalendar endDate)
      throws SQLException, ItemException
  {
    Vector v = new Vector();
    v.addAll(super.manyDaysEvents(startDate, endDate));
    v.addAll(subscribedEvents.manyDaysEvents(startDate, endDate));
    Collections.sort(v);
    removeDuplicates(v);
    return v;
  }

  /**
    Alert the object that events from many days are about to be
    requested.  This need not be called, but should speed up a
    series of calls to <code>oneDaysEvents()</code> if it is

    @param start The first date in a series of dates
    @param end The last date in a series of dates
    @exception SQLException if we have trouble loading data
    @exception ItemException if the data we try to load is bad
  */
  /* Note nameclash between java.util.calendar and
     edu.washington.cac.calendar.filter.Calendar
   */
  public void prepareManyDays(java.util.Calendar start, java.util.Calendar end)
      throws SQLException, ItemException
  {
    super.prepareManyDays(start, end);
    subscribedEvents.prepareManyDays(start, end);
  }

  /**
    Get the number of items in the Cache
    @return The number of items in the Cache
   */
  public int size()
  {
    return values().size();
  }

  // Note that below, we are always creating a sorted set.
  // sortedElements depends on this

  /**
    Get all the items in the cache, as an iterator
    @return All the items in the cache
   */
  public Iterator elements()
  {
    return values().iterator();
  }

  /**
    Get all the items in the cache, as a Collection
    @return All the items in the cache
   */
  public Collection values()
  {
    TreeSet s = new TreeSet(super.values());
    s.addAll(subscribedEvents.values());
    return s;
  }

  /**
    Get all the elements in the cache, in sorted order
    @return all the elements in the cache, in sorted order
   */
  public Iterator sortedElements()
  {
    return elements();
  }

  /**
    Does the user subscribe to a particular calendar?
    @param calendar The calendar
    @return Does the user subscribe to the calendar?
   */
  public boolean containsSubscription(Calendar calendar)
  {
    return subscriptions.contains(calendar);
  }

  /**
    Does the user subscribe to a particular calendar?
    @param id Calendar id
    @return Does the user subscribe to the calendar?
    @exception NoSuchItemException if there is no such public calendar
   */
  public boolean containsSubscription(int id) throws NoSuchItemException
  {
    return containsSubscription(PublicCalendars.getPublicCalendars().
                                                getCalendar(id));
  }

  /**
    Is an event explicitly in the cache (i.e., not just part of a
        subscription?)
    @param e The event
    @return Is the event explicitly in the cache
   */
  public boolean explicitlyContains(Event e)
  {
    return containsKey(key(e));
  }

  /**
    Get the set of associated locations for a user
    @return the set of associated locations for a user
    */
  public Locations getLocations() {
    Locations l = super.getLocations();

    try {
      l.addAll(subscribedEvents.getLocations().values(), false);
    } catch (ItemAccessException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
                "Shouldn't throw ItemAccessException, but it did"));
    } catch (SQLException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
                "Shouldn't throw SQLException, but it did"));
    }
    return l;
  }

  /**
    Get the set of associated sponsors for a user
    @return the set of associated sponsors for a user
   */
  public Sponsors getSponsors() {
    Sponsors s = super.getSponsors();

    try {
      s.addAll(subscribedEvents.getSponsors().values(), false);
    } catch (ItemAccessException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
          "Shouldn't throw ItemAccessException, but it did"));
    } catch (SQLException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
                "Shouldn't throw SQLException, but it did"));
    }

    return s;
  }

  /**
     Get the set of associated keywords for a user
     @return the set of associated keywords for a user
   */
  public Keywords getKeywords() {
    Keywords k = super.getKeywords();

    try {
      k.addAll(subscribedEvents.getKeywords().values(), false);
    } catch (ItemAccessException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
          "Shouldn't throw ItemAccessException, but it did"));
    } catch (SQLException e) {  // shouldn't happen
      throwLog(e, new RuntimeException(
                "Shouldn't throw SQLException, but it did"));
    }

    return k;
  }

  private void throwLog(Throwable t, RuntimeException newT)
          throws RuntimeException {
    Logger.getLogger(this.getClass()).error(newT.getMessage(), t);
    throw newT;
  }

  /**
    Print events for the next week for a user
    @param args First argument is the name of the user
    @exception Exception if there's a problem
   */
  public static void main(String args[]) throws Exception
  {
    if (args.length != 1) {
      System.out.println("Usage: PersonalEvents <user>");
      return;
    }

    PersonalEvents pe = new PersonalEvents(new User(args[0]));
    Iterator i =
        pe.manyDaysEvents(new MyCalendar(),
            new MyCalendar().addTime(java.util.Calendar.WEEK_OF_YEAR, 1)).
        iterator();

    for (; i.hasNext(); ) {
      Event e = (Event) i.next();
      System.out.println(e.getStartdate() + " -- " + e.getEnddate() + ": " +
                         e.getShortdesc());
    }
  }
}
